package com.ibm.ulc.application;

import com.ibm.ulc.util.UlcHashtable;
import java.util.Hashtable;
/**
 * ULC table models internally use oid to identify rows independently from the row's 
 * index in the model's rows.
 * Subclasses implement different mapping schemes to solve specific needs. 
 * Indexed item lists will use the ULCArrayedIndexToOidMap by default.
 */
abstract public class ULCIndexToOidMapAbstract  {
	/**
	 * The data structure holding the mapping from oid to index
	 */
	protected Hashtable fOidToIndexMap = new UlcHashtable();

	/**
	 * The itemList for which the receiver performs the mappings
	 */
	protected ULCIndexedItemList fItemList;

	/**
	 * Internal identifier for invalid oids
	 */
	protected int fInvalidOid = -1;
/**
 * Reset the receiver's oid to index map
 */
protected void clear() {
	fOidToIndexMap.clear();
	rebuildIndexToOidMap();
}
/**
 * Empty the receiver's oid to index map
 */
protected void clearOidToIndexMap(int newSize) {
	int initSize = Math.min((int) Math.round(newSize * 1.25), 5000);
	if (initSize == 0)
		fOidToIndexMap = new UlcHashtable();
	else
		fOidToIndexMap = new UlcHashtable(initSize);
}
/**
 * Return the index for the rowId specified. Answer -1 if the oid was not found.
 *
 * @param oid	int	The rowId whose index is required.
 */
public int getIndexForOid(int oid) {
	Integer rowIndex = (Integer) fOidToIndexMap.get(new Integer(oid));
	if (rowIndex != null)
		return rowIndex.intValue();
	else
		return ULCIndexedItemList.INVALID_OID;
}
/**
 * Answer the itemList of the receiver
 */
protected ULCIndexedItemList getItemList() {
	return fItemList;
}
/**
 * Answer the receiver's oids in an int array. 
 * The index is implicit in the position of each oid.
 */
abstract protected int[] getOidArray(int invalidOid);
/**
 * Return the object Id for the row Index or @defaultOid if not found
 *
 * @param 	index		int		the row index whose oid is required
 * @param	defaultOid	int		the default oid to be answered
 */
abstract public int getOidForIndex(int index, int defaultOid);
/**
 * Initialize another oid map with the receiver's current mappings.
 *
 * @param map 	ULCIndexToOidMapAbstract	The map to be initialized
 */
protected void initialize(ULCIndexToOidMapAbstract map) {
	if (fOidToIndexMap.size() > 0)
		map.setRows(getOidArray(map.fInvalidOid), false);
}
/**
 * Insert a range of new rows into the receiver.
 *
 * @param itemList	The itemList of the receiver, who wil be notified of the new mappings	
 * @param start 	The index of the first row to be inserted
 * @param end 		The index of the last row to be inserted
 */
abstract protected void insertRows(ULCDefaultItemList itemList, int start, int end);
/**
 * Insert the given oids to the receiver. Start this insertion at the specified index.
 *
 * @param insertionIndex 	the index at which the oids are to be inserted
 * @param oids				the array of oids to be inserted
 */
abstract public void insertRowsAt(int insertionIndex, int[] oids);
/**
 * Invert the the receiver's rows.
 */
abstract protected void invertRows();
/**
 * Rebuild the row index map based on values from the Oid map.
 */
abstract protected void rebuildIndexToOidMap();
/**
 * Rebuild the row index map based on values from the Oid map.
 */
abstract protected void rebuildOidToIndexMap();
/**
 * Update the receiver's maps by removing all mappings for @rowIds
 *
 * @param rowIds the oids of the rows to remove
 */
abstract protected void removeRows(int[] rowIds);
/**
 * Set the mapping for @index to @oid
 */
abstract protected void setIndexToOidMapping(int index, int oid);
/**
 * Set the receiver's itemList to the one provided
 *
 * @param itemList 	ULCIndexedItemList	The new itemList for the receiver
 */
protected void setItemList(ULCIndexedItemList itemList) {
	fItemList = itemList;
	fInvalidOid = itemList.INVALID_OID;
}
/**
 * Set the receiver's rows at the given rowIds. The index is implicit in the position of each oid.
 * Invert the order of the elements if so indicated by the inverted flag.
 *
 * @param rowIds	int[]	The rowIds of the rows to be set
 * @param inverted	boolean	Whether the order should be reversed
 */	
abstract protected void setRows(int[] rowIds, boolean inverted);
/**
 * Answer true if the receiver's current state indicates that the receiver 
 * may no longer be in synch with its model.
 *
 * @param currentRowCount	int		The current number of rows in the receiver
 *
 */
abstract protected boolean shouldNotifyContentsChanged(int currentRowCount);
}
