package com.ibm.ulc.application;

import com.ibm.ulc.util.UlcHashtable;
import com.ibm.ulc.util.UlcRange;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
/**
 * ULC table models internally use oid to identify rows independently from the row's index in the model's rows.
 * This class manages the index-to-oid mappings in a Hashtable. This solution emphasizes (initial) memory requirement over speed
 * since the Hashtable is populated only as the requests come from the instance's itemList. 
 * In most cases, this class would NOT be the one to use. Only in cases of enormous data on the table model, which the application
 * developer is sure will never be viewed by the user (but then, why even show so many rows?) would this class be preferable.
 */
public class ULCLookupIndexToOidMap extends ULCIndexToOidMapAbstract {
	protected Hashtable fIndexToOidMap = new UlcHashtable();
/**
 * Create and answer a vector of index ranges from the array of @rowIds.
 *
 * @param rowIds the oids to convert in index ranges
 */
protected Vector createIndexRangesFromOids(int[] rowIds) {
	Integer rowid;
	int[] indices = new int[rowIds.length];
	for (int i = 0; i < rowIds.length; i++) {
		rowid = new Integer(rowIds[i]);
		indices[i] = getIndexForOid(rowIds[i]);
	}
	return UlcRange.createFromIntArray(indices);
}
/**
 * Answer the value of the largets index in the receiver
 */
protected int getMaximumKnownIndex() {
	return getItemList().getRowCount();
}
/**
 * Answer the receiver's oids in an int array. 
 * The index is implicit in the position of each oid.
 */
protected int[] getOidArray(int invalidOid) {
	int rowCount = getItemList().getRowCount();
	int[] answer = new int[rowCount];
	int oid;
	for (int i = 0; i < rowCount; i++) {
		answer[i] = getOidForIndex(i, invalidOid);
	}
	return answer;
}
/**
 * Return the object Id for the row Index or @defaultOid if not found
 *
 * @param 	index		int		the row index whose oid is required
 * @param	defaultOid	int		the default oid to be answered
 */
public int getOidForIndex(int index, int defaultOid) {
	int oid = defaultOid;
	Integer key = new Integer(index);
	if (fIndexToOidMap.containsKey(key))
		oid = ((Integer) fIndexToOidMap.get(key)).intValue();
	if (oid == defaultOid)
		oid = getItemList().handleMissingOid(index);
	return oid;
}
/**
 * Insert a range of new rows into the receiver.
 * Assign oids to the items inserted and inform the itemList's model about the new mapping
 *
 * @param itemList	The itemList of the receiver	
 * @param start 	The index of the first row to be inserted
 * @param end 		The index of the last row to be inserted
 */
protected void insertRows(ULCDefaultItemList itemList, int start, int end) {
	if (start > getMaximumKnownIndex())
		return;
	int[] oids = new int[end - start + 1];
	for (int i = 0; i < oids.length; i++) {
		oids[i] = itemList.getNextOid();
		itemList.getModel().indexWasMapped(i + start, oids[i]);
	}
	insertRowsAt(start, oids);
}
/**
 * Insert the given oids to the receiver. Start this insertion at the specified index.
 *
 * @param insertionIndex 	the index at which the oids are to be inserted
 * @param oids				the array of oids to be inserted
 */
public void insertRowsAt(int insertionIndex, int[] oids) {
	prepareForIndexInsertion(insertionIndex, insertionIndex + oids.length - 1);
	for (int i = 0; i < oids.length; i++) {
		setIndexToOidMapping(insertionIndex + i, oids[i]);
	}
}
/**
 * Invert the the receiver's rows.
 */
protected void invertRows() {
	int rowCount = getItemList().getRowCount() - 1;
	fIndexToOidMap = newIndexToOidMap();
	Enumeration keys = fOidToIndexMap.keys();
	Integer oid;
	Integer index;
	while (keys.hasMoreElements()) {
		oid = (Integer) keys.nextElement();
		index = (Integer) fOidToIndexMap.get(oid);
		fIndexToOidMap.put(new Integer(rowCount - index.intValue()), oid);
	}
	rebuildOidToIndexMap();
}
/**
 * Create and answer a new index-oid map for the receiver
 */
protected Hashtable newIndexToOidMap() {
	return new UlcHashtable();
}
/**
 * Increase the indices of all rows known at or after index 
 * @start by the size of the inserted range.
 *
 * @param start			the insertion point for the new rows
 * @param end			the last index to insert
 */
protected void prepareForIndexInsertion(int start, int end) {
	int maxIndex = getMaximumKnownIndex();
	int delta = end - start + 1;
	Integer key;
	Integer value;
	for (int i = start; i < maxIndex; i++) {
		key = new Integer(i);
		value = (Integer) fIndexToOidMap.get(key);
		if (value != null)
			fOidToIndexMap.put(value, new Integer(key.intValue() + delta));
	}
	rebuildIndexToOidMap();
}
/**
 * Rebuild the row index map based on values from the Oid map.
 */
protected void rebuildIndexToOidMap() {
	fIndexToOidMap = newIndexToOidMap();
	Enumeration keys = fOidToIndexMap.keys();
	while (keys.hasMoreElements()) {
		Object key = keys.nextElement();
		fIndexToOidMap.put(fOidToIndexMap.get(key), key);
	}
}
/**
 * Rebuild the row index map based on values from the Oid map.
 */
protected void rebuildOidToIndexMap() {
	clearOidToIndexMap(fIndexToOidMap.size());
	Enumeration e = fIndexToOidMap.keys();
	while (e.hasMoreElements()) {
		Integer key = (Integer) e.nextElement();
		fOidToIndexMap.put(((Integer) fIndexToOidMap.get(key)), key);
	}
}
/**
 * Update the receiver's maps by removing all mappings for 
 * the given indices.
 *
 * @param start			the first index to remove
 * @param end			the last index to remove
 */
protected void removeIndicesFromTo(int start, int end) {
	int maxIndex = getMaximumKnownIndex();
	int delta = end - start + 1;
	Integer key;
	Integer value;
	for (int i = start; i < maxIndex; i++) {
		key = new Integer(i);
		value = (Integer) fIndexToOidMap.get(key);
		if (value != null)
			fOidToIndexMap.put(value, new Integer(key.intValue() - delta));
	}
}
/**
 * Update the receiver's maps by removing all mappings for @rowIds
 *
 * @param rowIds the oids of the rows to remove
 */
protected void removeRows(int[] rowIds) {
	Vector ranges = createIndexRangesFromOids(rowIds);
	for (int i = ranges.size() - 1; i >= 0; i--) {
		UlcRange range = (UlcRange) ranges.elementAt(i);
		removeIndicesFromTo(range.fStartIndex, range.fEndIndex);
		rebuildIndexToOidMap();
	}
}
/**
 * Set the mapping for @index to @oid
 */
protected void setIndexToOidMapping(int index, int oid) {
	fIndexToOidMap.put(new Integer(index), new Integer(oid));
	fOidToIndexMap.put(new Integer(oid), new Integer(index));
}
/**
 * Set the receiver's rows at the given rowIds. The index is implicit in the position of each oid.
 * Invert the order of the elements if so indicated by the inverted flag.
 *
 * @param rowIds	int[]	The rowIds of the rows to be set
 * @param inverted	boolean	Whether the order should be reversed
 */	
protected void setRows(int[] rowIds, boolean inverted) {
	if (inverted) {
		int length = rowIds.length;
		for (int i = 0; i >= length; i++) {
			setIndexToOidMapping(i, rowIds[length - i]);
		}
	}
	else {
		for (int i = 0; i < rowIds.length; i++) {
			setIndexToOidMapping(i, rowIds[i]);
		}
	}
	rebuildOidToIndexMap();
}
/**
 * Answer true if the receiver's current state indicates that the receiver 
 * may no longer be in synch with its model. This is assumed if the size 
 * of the receiver's mappings is equal to the @currentRowCount of the model.
 *
 * @param currentRowCount	int		The current number of rows in the receiver
 *
 */
protected boolean shouldNotifyContentsChanged(int currentRowCount) {
	return fIndexToOidMap.size() != currentRowCount;
}
}
