package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.Assert;
import com.ibm.ulc.comm.*;
import java.awt.event.InputEvent;

/**
 * Represents an ULCMenuItem. 
 * Each MenuItem can be associated with an action to be performed
 * when that item is selected. Short-cuts(accelerators) and mnemonics
 * are also supported, as is the enabling/disabling of menu-items using
 * the enabler interface.
 */
public class ULCMenuItem extends ULCComponent {
	/**
	 * The <code>int</code> value of the character to be used as a mnemonic.
	 * @serial	 	 	 
	 */
	protected int fMnemonic = -1;
	/**
	 * The ULCIcon if specified to be displayed.
	 * @serial	 	 	 
	 */
	protected ULCIcon fIcon = null;
	/**
	 * The ULCAccelerator for this item.
	 * @serial	 	 	 
	 */
	protected ULCAccelerator fAccelerator = null;
	/**
	 * Any valid ULCTrigger object which will be triggered in the UI when 
	 * this widgets actionPerformed or itemStateChanged (ToggleMenuItems) method is called
	 * @serial	 
	 */
	protected ITrigger fTrigger = null;
public ULCMenuItem() {
}
/**
 * Creates a menuitem with text.
 *
 * @param text The <code>String</code> value to be displayed.
 */
public ULCMenuItem(String text) {
	super(text);
}
/**
 * Creates a menuitem with text and registers the given listener for it.
 *
 * @param text 	the text string to be displayed.
 * @param listener	the widget that will listen for my action events.
 */
public ULCMenuItem(String text, IActionListener l) {
	super(text);
	addActionListener(l);
}
/**
 * Creates a menuitem with text and an icon.
 *
 * @param text 	the text string to be displayed.
 * @param icon		the <code>ULCicon</code> that should be displayed along with the label.
 */
public ULCMenuItem(String text, ULCIcon icon) {
	super(text);
	fIcon = icon;
}
/**
 * Creates a menuitem with text and an icon registers the given listener for it.
 *
 * @param text		The text string to be displayed.
 * @param icon 		The <code>ULCIcon</code> to be displayed alongside the label.
 * @param listener	The widget interested in my actionEvents.
 */
public ULCMenuItem(String text, ULCIcon icon, IActionListener l) {
	this(text, l);
	fIcon = icon;
}
/**
 * Registers the given listener to begin receiving notifications
 * when the menu item was fired.
 *
 * @param listener	The object interested in my actionEvents.
 */
public void addActionListener(IActionListener listener) {
	internalAddListener("action", listener);
}
/**
 * Gets the given keyboard shortcut for this menuitem.
 *
 * @return accelerator 	The <code>ULCAccelerator</code> to be used as the accelerator.
 */
public ULCAccelerator getAccelerator() {
	return fAccelerator;
}
/**
 * Return the current icon.
 */
public ULCIcon getIcon() {
	return fIcon;
}
/**
 * Gets the MenuItem's label.
 *
 * @return The current label string 
 */
public String getLabel() {
	return fLabel;
}
/**
 * Gets this widgets mnemonic code.
 *
 * @return keyCode The keyCode that represents the character to be used as a mnemonic
 */
public int getMnemonic() {
	return fMnemonic;
}
/**
 * Returns the configured ULCTrigger object which will be triggered in the UI when 
 * this widget's actionPerformed method is called.
 *
 * @return ITrigger	 
 */
public ITrigger getTrigger() {
	return fTrigger;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("action")) {
			distributeToListeners("action", new ULCActionEvent(this, args.get("cmd", "???")));
		}
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive action events. 
 *
 * @param listener	The widget that was registered to receive my actionEvents.
 */
public void removeActionListener(IActionListener listener) {
	internalRemoveListener("action", listener);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fIcon != null)
		a.put("icon", fIcon.getRef(fContext));
	if (fAccelerator != null)
		a.put("accelerator", fAccelerator.asStcAnything());
	if (fMnemonic != -1)
		a.put("mnemonic", fMnemonic);
	if (fTrigger != null)
		a.put("trigger", ((ULCProxy) fTrigger).getRef(fContext));
}
/**
 * Sets the given keyboard shortcut for this menuitem.
 * <pre>
 * valid modifiers are:
 * InputEvent.SHIFT_MASK
 * InputEvent.CTRL_MASK
 * InputEvent.META_MASK
 * InputEvent.ALT_MASK
 * </pre>
 *
 * @param keyCode 		The <code>int</code> character value of the key to be used as the accelerator.
 * @param modifier		The keyboard mask to be tested for along with the keyCode.
 * @param onKeyRelease 	If true, the accelerator is triggered on key release.
 */
public void setAccelerator(int keyCode, int modifier, boolean onKeyRelease) {
	setAccelerator(ULCAccelerator.getAccelerator(keyCode, modifier, onKeyRelease));
}
/**
 * Sets the given keyboard shortcut for this menuitem.
 *
 * @param accelerator 	The <code>ULCAccelerator</code> to be used as the accelerator.
 */
public void setAccelerator(ULCAccelerator accelerator) {
	if ((fAccelerator == null) || (!accelerator.equals(fAccelerator))) {
		fAccelerator = accelerator;
		sendUI("setAccelerator", fAccelerator.asStcAnything());
	}
}
/**
 * Set the icon of the button.
 *
 * @param icon The ULCIcon to be displayed.
 */
public void setIcon(ULCIcon icon) {
	if (icon != fIcon) {
		fIcon = icon;
		sendUI("setIcon", fIcon);
	}
}
/**
 * Sets the widgets's <code>String</code> label.
 * Note: If the label contains the <code>IDefaults.MNEMONIC_INDICATOR</code> then
 * the character after this indicator is treated as the mnemonic character for this widget.
 *
 * @param label the label <code>String</code> of the widget.
 */
public void setLabel(String label) {
	internalSetLabel(label);
}
/**
 * Sets this widgets mnemonic code.
 *
 * @param keyCode The keyCode that represents the character to be used as a mnemonic
 */
public void setMnemonic(int keyCode) {
	if (keyCode != fMnemonic) {
		fMnemonic = keyCode;
		sendUI("setMnemonic", new Anything(fMnemonic));
	}
}
/**
 * Do nothing since MenuItem does not support a popupMenu
 *
 * @param 	popupMenu 
 *			The <code>ULCMenu</code> to be displayed when a popupRequest is received
 */
public void setPopupMenu(ULCMenu popupMenu) {
}
/**
 * Sets the configured ULCTrigger which will be triggered in the UI when 
 * this widgets actionPerformed method is called
 *
 * @param ULCTrigger	 
 */
public void setTrigger(ITrigger trigger) {
	if (trigger != fTrigger) {
		fTrigger = trigger;
		sendUI("setTrigger", (ULCTrigger)fTrigger);
	}
}
}
