package com.ibm.ulc.application;

import com.ibm.ulc.util.UlcHashtable;
import com.ibm.ulc.util.ITableRow;
import java.lang.reflect.Method;
/**
 * The ULCRowAdapter is used to get and set data from the domain models.
 * Essentially it implements a getValue / setValue api to do this.
 *
 * @see IRowAdapter
 *
 * In order that the receiver adapts an object, we assume that the (domain) object 
 * adheres to the JavaBeans convention, so that the get/set methods for a given
 * attribute can be computed based on this convention. And java reflection can then
 * be used to obtain the corresponding value.
 *
 * @see #getAccessMethod(Object, String)
 *
 */
public class ULCRowAdapter extends ULCAdapter implements IRowAdapter {
/**
 * Construct a new ULCRowAdapter
 */
public ULCRowAdapter() {
	super();
}
/**
 * Return the getter method that matches the attributeName.
 */
protected Method getAccessMethod(Object object, String attributeName) {
	String selector = "get" + (attributeName.substring(0, 1)).toUpperCase() + attributeName.substring(1);
	UlcHashtable methodCache = getMethodCache(object);
	Object o = methodCache.get(selector);
	if (o != null)
		return (java.lang.reflect.Method) o;
	Method method = null;
	try {
		method = object.getClass().getMethod(selector, new Class[0]);
	} catch (Throwable t) {
		return null;
	}
	methodCache.put(selector, method);
	return method;
}
/**
 * Return the receiver's methodCache for the specified class name
 */
private UlcHashtable getMethodCache(Object model) {
	return super.getMethodCache(model.getClass().getName());
}
/**
 * Return the setter method that matches the attributeName
 */
protected Method getSetterMethod(Object object, String attributeName, Object value) {
	String selector = "set" + (attributeName.substring(0, 1)).toUpperCase() + attributeName.substring(1);
	UlcHashtable methodCache = getMethodCache(object);
	Object o = methodCache.get(selector);
	if (o != null)
		return (Method) o;
	Class valueClass = value.getClass();
	try {
		Method[] methods = object.getClass().getMethods();
		Method objectMethod = null;
		for (int i = 0; i < methods.length; i++) {
			Method m = methods[i];
			if (m.getName().equals(selector)) {
				Class parms[] = m.getParameterTypes();
				if ((parms.length == 1) && (parms[0] == valueClass)) {
					methodCache.put(selector, m);
					return m;
				}
				if ((parms.length == 1) && (parms[0] == Object.class)) {
					objectMethod = m;
				}
			}
		}
		if (objectMethod != null) {
			methodCache.put(selector, objectMethod);
			return objectMethod;
		}
	} catch (Throwable t) {
		return null;
	}
	return null;
}
/**
 * Answer the value for the given ITableRow object at the given attribute.
 * Since ITableRow implementors can answer this information directly, this
 * call is dispatched to the given ITableRow object.
 *
 * @param attributeName 	String		The attribute at which the value is requested
 * @param object			ITableRow	The object whose value at the above attribute
 *										is asked for.
 */
public Object getValue(String attributeName, com.ibm.ulc.util.ITableRow object) {
	return object.getValueAt(attributeName);
}
/**
 * Answer the value for the given object at the given attribute.
 * We assume that the (domain) object adheres to the JavaBeans convention. Hence
 * the value is obtained using java reflection.
 *
 * @param attributeName 	String		The attribute at which the value is requested
 * @param object			Object	The object whose value at the above attribute
 *										is asked for.
 */
public Object getValue(String attributeName, Object object) {
	Object answer = null;
	try {
		Method method = getAccessMethod(object, attributeName);
		answer = method.invoke(object, new Object[0]);
	}
	catch (Throwable t) {
		//trouble("getValue()", "Exception getting value at attributeName: " + attributeName);
		answer = object.toString() + "->" + attributeName;
	}
	return answer;
}
/**
 * Internal method to set the attributeName value of the object to the specified value.
 *
 * @param attributeName 	String	The attribute
 * @param value				Object	The value to which the above attribute
 *									is being set.
 * @param object			Object	The object whose value at the above attribute
 *									is being set.
 */
protected void internalSetValue(String attributeName, Object value, Object object) {
	try {
		setValue(attributeName, value, (ITableRow) object);
	}
	catch (ClassCastException ce) {
		setValue(attributeName, value, object);
	}
}
/**
 * Set the attributeName value of the object to the specified <code>byte</code> value.
 *
 * @param attributeName 	String	The attribute
 * @param value				byte	The value to which the above attribute
 *									is being set.
 * @param object			Object	The object whose value at the above attribute
 *									is being set.
 */
public void setValue(String attributeName, byte value, Object object) {
	internalSetValue(attributeName, new Byte(value), object);
}
/**
 * Set the attributeName value of the object to the specified <code>char</code> value.
 *
 * @param attributeName 	String	The attribute
 * @param value				char	The value to which the above attribute
 *									is being set.
 * @param object			Object	The object whose value at the above attribute
 *									is being set.
 */
public void setValue(String attributeName, char value, Object object) {
	internalSetValue(attributeName, new Character(value), object);
}
/**
 * Set the attributeName value of the object to the specified <code>double</code>value.
 *
 * @param attributeName 	String	The attribute
 * @param value				double	The value to which the above attribute
 *									is being set.
 * @param object			Object	The object whose value at the above attribute
 *									is being set.
 */
public void setValue(String attributeName, double value, Object object) {
	internalSetValue(attributeName, new Double(value), object);
}
/**
 * Set the attributeName value of the object to the specified <code>int</code>value.
 *
 * @param attributeName 	String	The attribute
 * @param value				int		The value to which the above attribute
 *									is being set.
 * @param object			Object	The object whose value at the above attribute
 *									is being set.
 */
public void setValue(String attributeName, int value, Object object) {
	internalSetValue(attributeName, new Integer(value), object);
}
/**
 * Set the attributeName value of the object to the specified <code>long</code>value.
 *
 * @param attributeName 	String	The attribute
 * @param value				long	The value to which the above attribute
 *									is being set.
 * @param object			Object	The object whose value at the above attribute
 *									is being set.
 */
public void setValue(String attributeName, long value, Object object) {
	internalSetValue(attributeName, new Long(value), object);
}
/**
 * Set the attributeName value of the object to the specified value.
 * Since this is an ITableRow object, it's value can be set without needing
 * reflection.
 *
 * @param attributeName 	String	The attribute
 * @param value				Object	The value to which the above attribute
 *									is being set.
 * @param object			ITableRow	The object whose value at the above attribute
 *									is being set.
 */
public void setValue(String attributeName, Object value, ITableRow object) {
	object.setValueAt(value, attributeName);
}
/**
 * Set the attributeName value of the object to the specified value.
 * Since this is an object of no specified type, we use java reflection
 * to set this value.
 *
 * @param attributeName 	String	The attribute
 * @param value				Object	The value to which the above attribute
 *									is being set.
 * @param object			ITableRow	The object whose value at the above attribute
 *									is being set.
 */
public void setValue(String attributeName, Object value, Object object) {
	try {
		Method method = getSetterMethod(object, attributeName, value);
		if (method != null) {
			method.invoke(object, new Object[] {value});
		} else
			System.out.println("ULCRowAdapter.setValue: No method found for object[" + object + "] attributeNamed[" + attributeName + "] value[" + value + "] value class[" + value.getClass() + "]");
	} catch (Throwable t) {
		System.out.println("ULCRowAdapter : Exception setting value at attributeName: " + attributeName);
	}
}
/**
 * Set the attributeName value of the object to the specified <code>short</code> value.
 *
 * @param attributeName 	String	The attribute
 * @param value				short	The value to which the above attribute
 *									is being set.
 * @param object			Object	The object whose value at the above attribute
 *									is being set.
 */
public void setValue(String attributeName, short value, Object object) {
	internalSetValue(attributeName, new Short(value), object);
}
/**
 * Set the attributeName value of the object to the specified <code>boolean</code> value.
 *
 * @param attributeName 	String	The attribute
 * @param value				boolean	The value to which the above attribute
 *									is being set.
 * @param object			Object	The object whose value at the above attribute
 *									is being set.
 */
public void setValue(String attributeName, boolean value, Object object) {
	setValue(attributeName, new Boolean(value), object);
}
}
