package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.util.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;

/**
 * A ULCSortingItemList is an ItemList implementation that will sort the rows of a given TableModel
 * in ascending / descending order based on the list of attributes provided. Using the receiver will touch
 * all rows of its tableModel at startup time. If the data of the receiver's tableModel is huge, this class
 * should probably not be used. Instead, a special IndexedItemList should be created that gets its rows
 * from a database view like construct.
 * 
 */
public class ULCSortedItemList extends ULCIndexedItemList {
	/**
	 * Specify whether the receiver sorts items in ascending order or not.
	 */
	protected boolean fAscending = true;

	/**
	 * The collection of attributeNames that this receiver is sorted on.
	 * Note: in the current release only the first attribute supplied is used 
 	 * for sorting
	 */
	protected String[] fAttributeNames = new String[0];

	/**
	 * The object which performs the actual comparision required
	 * for sorting
	 */
	protected IComparator fComparator = null;
/**
 * Construct a new ULCSortedItemList
 */
public ULCSortedItemList() {
	fComparator = new ULCDefaultComparator();
}
/**
 * Construct a new ULCSortedItemList, using the given comparator
 * which will be used for determining the sort order.
 *
 * @param comparator	IComparator		The comparator to be used for comparing objects
 */
public ULCSortedItemList(IComparator comparator) {
	fComparator = comparator;
}
/**
 * Compares the object at leftOid to rightOid answer 0 if they are equal -1 if its less and 1 if its greater.
 *
 * @return  the value <code>0</code> if the attribute at attributeIndex
 *			of the row leftOid is equal to that in rightOid;
 * 			a value less than <code>0</code> if its less; and a
 *          value greater than <code>0</code> if its greater.
 */
protected int compare(int leftOid, int rightOid, int attributeIndex) {
	Object valueLeft = getValueAtRowId(fAttributeNames[attributeIndex], leftOid);
	Object valueRight = getValueAtRowId(fAttributeNames[attributeIndex], rightOid);
	return fComparator.compare(valueLeft, valueRight);
}
/**
 * The contents of the receiver's tableModel has changed. The receiver must reinitialize
 * itself. It is called when the receiver has been uploaded upon changes in the receiver's
 * TableModel, or before the receiver is saving its state.
 *
 * At this point the receiver re-sorts its rows
 *
 * @see #saveState
 */
protected void contentsChanged() {
	sortRows();
}
/**
 * Return the collection of attributeNames that this list is sorted on.
 * Note: in the current release only the first attribute supplied is used 
 * for sorting.
 */
public String[] getAttributeNames() {
	return fAttributeNames;
}
/**
 * Routing method to answer the value for attributeName of the row specified by rowid.
 *
 * @param	attributeName	The key of the attribute to be retrieved.
 * @param	rowId			The oid of the row being accessed.
 * @since	R3.1
 */
protected Object getValueAtRowId(String attributeName, int oid) {
	return fModel.getValueAtRowId(attributeName, oid);;
}
/**
 * An oid was not defined at the given index.
 * Answer the invalid oid.
 *
 * @param index		int		The index at which the oid is missing
 */
protected int handleMissingOid(int index) {
	return INVALID_OID;
}
/**
 * Return true if this sorted list is in ascending order.
 */
public boolean isAscending() {
	return fAscending;
}
/**
 * The rows specified by rowIds have been added to the receiver's model. 
 * Update the receiver and its UI proxy accordingly.
 *
 * @param rowIds	int[]	The set of rowIds that have been added to my model
 */
public void notifyRowsAdded(int[] rowIds) {
	if (isUploaded() && shouldProcessNotification()) {
		sortRows();
	}
	super.notifyRowsAdded(rowIds);
}
/**
 * The rows specified by rowIds have changed in the specified attributes.
 * No attribute specified indicates that the rows have completely changed.
 * Update the receiver and its UI proxy accordingly.
 *
 * @param rowIds 			int[] 		The rowids of the changed nodes.
 * @param attributes 		String[] 	The changed attributes
 */
public void notifyRowsChanged(int[] rowIds, String[] attributes) {
	String[] names = getAttributeNames();
	for (int i = 0; i < names.length; i++) {
		for (int ai = 0; ai < attributes.length; ai++)
			if (names[i].equals(attributes[ai])) {
				if (shouldProcessNotification())
					notifyContentsChanged();
				return;
			}
	}
}
/**
 * Sort the nodes in the given array.
 */
private void quickSortRowIds(int[] list, int left, int right) {
	int original_left = left;
	int original_right = right;
	int mid = list[ (left + right) / 2];
	do {
		while (compare(list[left], mid, 0) < 0) {
			left++;
		}
		while (compare(mid, list[right], 0) < 0) {
			right--;
		}
		if (left <= right) {
			int tmp = list[left];
			list[left] = list[right];
			list[right] = tmp;
			left++;
			right--;
		}
	} while (left <= right);
	if (original_left < right) {
		quickSortRowIds(list, original_left, right);
	}
	if (left < original_right) {
		quickSortRowIds(list, left, original_right);
	}
}
/**
 * Set boolean whether this list should contain a collection of items 
 * in ascending order.
 *
 * @param ascending		boolean 	true indicates that the items should be in
 *									ascending order, false indicates descending.
 */
public void setAscending(boolean ascending) {
	if (fAscending != ascending) {
		fAscending = ascending;
		if (shouldProcessNotification()) {
			getOidMap().invertRows();
			sendContentsChanged();
		}
	}
}
/**
 * Set the collection of attributeNames that this list is sorted on.
 * Note: in the current release only the first attribute supplied is used 
 * for sorting.
 *
 * @param attributeNames	String[]	The attributeNames to sort the receiver on.
 */
public void setAttributeNames(String[] attributeNames) {
	if (attributeNames == null)
		fAttributeNames = new String[0];
	else
		fAttributeNames = attributeNames;
	if (shouldProcessNotification())
		sortRows();
}
/**
 * Force the mapping table to be rebuilt based on the current 
 * sort order and attributesNames chosen.
 */
protected void sortRows() {
	int[] allRowIds = getModel().getAllRowIds();
	if ((allRowIds.length > 0) && (fAttributeNames.length != 0)) {
		quickSortRowIds(allRowIds, 0, allRowIds.length - 1);
	}
	getOidMap().setRows(allRowIds, !fAscending);
}
}
