package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.util.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.*;

/**
 * A ULCTableModel is a default implementation for a 'remote' model, that is a model which is split
 * across the UIEngine and the ULC application side. It implements the protocol defined by the UIEngine
 * for synchronizing two model halfs in terms of a simple getValue/setValue API.
 * Clients have to subclass from ULCTableModel to at least implement the <code>getValueAt</code> and
 * <code>getRowCount</code> methods.
 * Typically subclasses are not storing the data itself but are adapters which redirect the getValueAt and
 * setValueAt calls to some other model or data structure.
 */
abstract public class ULCTableModel extends ULCAbstractTableModel implements IRowAdapter, IIndexedTableModel {
	/**
	 * @serial
	 * Internal use. 
	 */
	private static final boolean DEBUG = false;
	/**
	 * The default itemList maintains the order of rows the way the receiver
	 * did in releases prior to R1.6
	 */
	protected ULCDefaultItemList fDefaultItemList;
/**
 * Constructs a ULCTableModel.
 */
public ULCTableModel() {
	super();
}
/**
 * Constructs a ULCTableModel with the specified number of prefetch rows.
 *
 * @param prefetch 	int	Determines how many rows of data are transmitted in a single request.
 */
public ULCTableModel(int prefetch) {
	super(prefetch);
}
/**
 * Constructs a ULCTableModel with the specified number of prefetch rows and notification policy
 *
 * @param prefetch determines how many rows of data are transmitted in a single request.
 * @param policy	The notification policy
 * <pre>
 * Valid values are :
 *	TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE
 * 	TABLE_EDIT_UPDATE_ON_REQUEST
 *	TABLE_EDIT_UPDATE_ON_ROW_CHANGE
 * </pre>
 */
public ULCTableModel(int prefetch, int policy) {
	this(prefetch);
	fNotificationPolicy = policy;
}
/**
 * Constructs a ULCTableModel with the specified number of prefetch rows and notification policy
 * If veto is true then all updates are sent to the application for confirmation.
 *
 * @param prefetch 	determines how many rows of data are transmitted in a single request.
 * @param veto 		determines whether changes of a cell modify the UI Engine's cache
 *					directly or whether they are reported back to the application via the setValueAt calls.
 * @param policy	The notification policy
 * <pre>
 * Valid values are :
 *	TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE
 * 	TABLE_EDIT_UPDATE_ON_REQUEST
 *	TABLE_EDIT_UPDATE_ON_ROW_CHANGE
 * </pre>
 */
public ULCTableModel(int prefetch, boolean veto, int policy) {
	super(prefetch, veto, policy);
}
/**
 * Constructs a ULCTableModel.
 *
 * @param boolean veto ; Determines whether changes of a cell modify the UI Engine's 
 *							cache directly or whether they are reported back to the 
 *							application via the setValueAt calls.
 */
public ULCTableModel(boolean veto) {
	super(veto);
}
/**
 * Constructs a ULCTableModel with the specified veto and notification policy.
 * If veto is true then all updates are sent to the application for confirmation.
 *
 * @param veto 		determines whether changes of a cell modify the UI Engine's cache
 *					directly or whether they are reported back to the application via the setValueAt calls.
 * @param policy	The notification policy
 * <pre>
 * Valid values are :
 *	TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE
 * 	TABLE_EDIT_UPDATE_ON_REQUEST
 *	TABLE_EDIT_UPDATE_ON_ROW_CHANGE
 * </pre>
 */
public ULCTableModel(boolean veto, int policy) {
	super(veto, policy);
}
/**
 * Answer an array with all the receiver's itemLists, including the default itemList,
 * which is not contained in the <code>fItemLists</code>.
 */
protected void allItemLists(Vector lists) {
	lists.addElement(getIndexedItemList());//make sure the default itemlist is first in the list
	super.allItemLists(lists);
}
/**
 * Answer the receiver's current itemList.
 *
 * The receiver's <code>ULCItemlist</code> is responsible for the order in which
 * the receiver's rows are displayed.
 *
 * @see ULCDefaultItemList
 * @see ULCSortedItemList
 *
 */
protected ULCDefaultItemList getDefaultItemList() {
	return fDefaultItemList;
}
/**
 * Answer the receiver's current itemList.
 *
 * The receiver's <code>ULCItemlist</code> is responsible for the order in which
 * the receiver's rows are displayed.
 *
 * @see ULCDefaultItemList
 * @see ULCSortedItemList
 *
 */
public IIndexedItemList getIndexedItemList() {
	return getDefaultItemList();
}
/*
 * Answer the adapter for the receiver's rows. It is used to access the values of the rows.
 */
protected IRowAdapter getRowAdapter() {
	return this;
}
/**
 * Override this method to return the requested row.
 *
 * @param row	The index of the row being accessed.
 */
abstract public Object getRowAt(int row);
/**
 * This method is defined by the interface IIndexedTableModel. It is used to access rows
 * at the specified index when the mapping from index to oid is done, but the underlying row
 * is not available in the receiver's rowCache. The method is redirected to the getRowAt API.
 *
 * @param index 		The index of the row being accessed.
 */
public Object getRowAtIndex(int index) {
	return getRowAt(index);
}
/**
 * Override this method to return the number of rows of this table.
 */
abstract public int getRowCount();
/**
 * Override this method to return the requested value for the specified cell.
 *
 * @param attributeName	The key of the attribute to be retrieved.
 * @param row			The index of the row being accessed.
 */
public Object getValue(String attributeName, Object object) {
	return getValueAt(attributeName, getIndexForOid(getRowIdFor(object)));
}
/**
 * Override this method to return the requested value for the specified cell.
 *
 * @param attributeName	The key of the attribute to be retrieved.
 * @param row			The index of the row being accessed.
 */
abstract public Object getValueAt(String attributeName, int rowIndex);
/**
 * Override this method to return the requested value for the specified cell.
 *
 * @param attributeName	The key of the attribute to be retrieved.
 * @param rowId			The oid of the row being accessed.
 */
public Object getValueAtRowId(String attributeName, int rowId) {
	int index = getRowIndexForOid(rowId);
	if (index >= 0)
		// this can happen in the case when the rows have just been deleted
		return getValueAt(attributeName, index);
	else
		return null;
}
/*
 * extract and answer all oids defined by the ranges of indices
 *
 * @param ranges A Vector containing <code>UlcRange</code> objects
 */
private int[] indexRangesToOids(Vector ranges) {
	Vector allOids = new Vector();
	for (int i = 0; i < ranges.size(); i++) {
		UlcRange range = (UlcRange) ranges.elementAt(i);
		Anything rowIds = new Anything();
		for (int ri = range.fStartIndex; ri <= range.fEndIndex; i++) {
			int oid = getOidForIndex(ri);
			if (oid > -1)
				allOids.addElement(new Integer(oid));
		}
	}
	int[] answer = new int[allOids.size()];
	for (int i = 0; i < answer.length; i++) {
		answer[i] = ((Integer) allOids.elementAt(i)).intValue();
	}
	return answer;
}
/**
 * Notification that the receiver's DefaultItemList has mapped index to an oid.
 *
 * @param	index	int		The index which was mapped
 * @param	oid		int		The oid to which the index was mapped
 */
protected void indexWasMapped(int index, int oid) {
	getRowCache().setRowIdFor(oid, getRowAt(index));
}
/**
 * Initialize the receiver.
 */
protected void initialize() {
	super.initialize();
	resetDefaultItemList(null);
}
/**
 * Notify UI TableModel that either contents or size of
 * ULC TableModel has changed.
 *
 * @param type 	The kind of change.
 * <pre>
 * type can be one of:
 *	TABLE_MODEL_CONTENTS_CHANGED;
 *	TABLE_MODEL_ROWS_ADDED;
 *	TABLE_MODEL_ROWS_REMOVED;
 *	TABLE_MODEL_ROWS_CHANGED;
 *	TABLE_MODEL_CELL_CHANGED;
 * </pre>
 * @param colId The column's identifier.
 * @param start The first row where the change occurred.
 * @param end 	The last row where the change occurred.
 */
public void notify(int type, String colId, int start, int end) {
	Vector ranges = new Vector();
	ranges.addElement(new UlcRange(start, end));
	notify(type, colId, ranges);
}
/**
 * Notify UI TableModel that either contents or size of
 * ULC TableModel has changed.
 * This message is passed to the receiver's defaultItemList for processing
 *
 * @param type 	The kind of change.
 * <pre>
 * type can be one of:
 *	TABLE_MODEL_CONTENTS_CHANGED;
 *	TABLE_MODEL_ROWS_ADDED;
 *	TABLE_MODEL_ROWS_REMOVED;
 *	TABLE_MODEL_ROWS_CHANGED;
 *	TABLE_MODEL_CELL_CHANGED;
 * </pre>
 * @param colId The column's identifier.
 * @param ranges A Vector containing <code>UlcRange</code> objects
 */
public void notify(int type, String colId, Vector ranges) {
	getDefaultItemList().notify(type, colId, ranges);
}
/**
 * Reset the receiver's default itemList.
 */
protected void resetDefaultItemList(IIndexedItemList itemList) {
	if (itemList == null || itemList == getDefaultItemList()) {
		fDefaultItemList = new ULCDefaultItemList();
		fDefaultItemList.setModel(this);
	}
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveItemLists(Anything a) {
	a.put("iil", getIndexedItemList().getRef(fContext));
}
/**
 * Set the receiver's RowCache to it's default value
 */
protected void setRowCache() {
	setRowCache(new ULCIndexedRowCache(this));
}
/**
 * Override this method to store the given value in the specified cell.
 * The default implementation prints the given argument on the console.
 *
 * @param value 		The new <code>Object</code> value to be set.
 * @param attributeName	The key of the attribute that has changed.
 * @param Object		The <code>Object</code> whose value is to be set at the given attribute.
 * 
 */
public void setValue(String attributeName, Object object, Object value) {
	setValueAt(object, attributeName, getIndexForOid(getRowIdFor(object)));
}
/**
 * Override this method to store the given value in the specified cell.
 * The default implementation prints the given argument on the console.
 *
 * @param value 		The new <code>Object</code> value to be set.
 * @param attributeName	The key of the attribute that has changed.
 * @param rowIndex		The index of the row that has changed.
 * 
 */
public void setValueAt(Object value, String attributeName, int rowIndex) {
	System.out.println("ULCTableModel.setValueAt(" + attributeName + ", " + rowIndex + "): " + value.toString());
}
/**
 * Override this method to store the given value in the specified cell.
 * The default implementation prints the given argument on the console.
 *
 * @param value 		The new <code>Object</code> value to be set.
 * @param attributeName	The key of the attribute that has changed.
 * @param rowIndex		The index of the row that has changed.
 * 
 */
public void setValueAtRowId(Object value, String attributeName, int rowId) {
	setValueAt(value, attributeName, getIndexForOid(rowId));
}
}
