package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;

/**
 * This class serves as model for the ULCTree widget.
 * W.e.f. R3.0, it is recommended that ULCTreeTableModel be used instead,
 * along with ULCTreeTable (instead of ULCTree).
 */
abstract public class ULCTreeModel extends ULCTreeTableModel {
/**
 * Constructs a ULCTreeModel.
 */
public ULCTreeModel() {
	super();
}
/**
 * Constructs a TreeModel with the specifiend number of prefetch rows.
 *
 * @param prefetchCount 	determines the maximum number of children of a node 
 * 							that are transmitted in a single request.
 */
public ULCTreeModel(int prefetchCount) {
	super(prefetchCount);
}
/**
 * Convenience method to add a new child to an existing parent.
 *
 * @param parent		The parent to whom the new child is to be added
 * @param newChild		The child to be added to this parent.
 *						The child is added at the end of the parent's
 *						existing list of children.
 *
 */
public void addChild(Object parent, Object newChild) {
	internalNotifyRowsAdded(new int[] {getRowIdFor(newChild)});
}
/**
 * Convenience method to add a new child to an existing parent, at the given
 * index.
 *
 * @param parent		The parent to whom the new child is to be added
 * @param newChild		The child to be added to this parent.
 * @param index			The index in the parent's child-collection, where
 *						this new child is to be inserted. If this index
 * 						is invaild, the adding happens at the end of the
 *						parent's existing list of children.
 *
 */
public void addChild(Object parent, Object newlyAddedChild, int index) {
	internalNotifyRowsAdded(new int[] {getRowIdFor(newlyAddedChild)});
}
/**
 * Convenience method to add a new child to an existing parent, after a given
 * child.
 *
 * @param parent		The parent to whom the new child is to be added
 * @param newChild		The child to be added to this parent.
 * @param existingChild		The already uploaded child, after which the 
 *						new child is to be inserted. 
 *
 */
public void addChildAfter(Object parent, Object newlyAddedChild, Object existingChild) {
	addChild(parent, newlyAddedChild, getChildIndexFor(existingChild, parent));
}
/**
 * Convenience method to add a new child to an existing parent, before a given
 * child.
 *
 * @param parent		The parent to whom the new child is to be added
 * @param newChild		The child to be added to this parent.
 * @param existingChild		The already uploaded child, before which the 
 *						new child is to be inserted. 
 *
 */
public void addChildBefore(Object parent, Object newlyAddedChild, Object existingChild) {
	addChild(parent, newlyAddedChild, getChildIndexFor(existingChild, parent) - 1);
}
/**
 * Add the given children to an existing parent.
 *
 * @param parent		The parent to whom the new children are to be added
 * @param newChildren	The children to be added to this parent.
 *						The children are added at the end of the parent's
 *						existing list of children.
 *
 */
public void addChildren(Object parent, Vector newChildren) {
	int[] rowIds = new int[newChildren.size()];
	for (int i = 0; i < newChildren.size(); i++) {
		rowIds[i] = getRowIdFor(newChildren.elementAt(i));
	}
	internalNotifyRowsAdded(rowIds);
}
/**
 * Add the given children to an existing parent, at the specified indices.
 *
 * @param parent		The parent to whom the new children are to be added
 * @param newChildren	The children to be added to this parent.
 * @param indices		The respective indices where each child is to be
 *						added. If any of these indices is invalid, the
 *						corresponding child is added at the end of the parent's
 *						existing list of children.
 *
 */
public void addChildren(Object parent, Vector newlyAddedChildren, Vector indices) {
	com.ibm.ulc.util.Assert.isTrue(newlyAddedChildren.size() == indices.size());
	int[] rowIds = new int[newlyAddedChildren.size()];
	for (int i = 0; i < newlyAddedChildren.size(); i++) {
		rowIds[i] = getRowIdFor(newlyAddedChildren.elementAt(i));
	}
	internalNotifyRowsAdded(rowIds);
}
/**
 * Notify UI TreeModel that "everything" should be marked as invalid.
 * This will cause all nodes, including the root node,
 * to be re-fetched from the ULC to the UI.
 * This would be done in order to change the root, say.
 * 
 */
public void changed() {
	reset();
}
/**
 * Convenience method to delete an existing child.
 *
 * @param childToBeDeleted	The child to be deleted from it's parent.
 *
 */
public void deleteChild(Object childToBeDeleted) {
	notifyDelete(childToBeDeleted);
}
/**
 * Convenience method to delete a collection of existing children.
 *
 * @param childrenToBeDeleted	Vector 	The children to be deleted from 
 *											their respective parents.
 *
 */
public void deleteChildren(Vector childrenToBeDeleted) {
	notifyDelete(childrenToBeDeleted);
}
/**
 * Override this method to return the icon to be displayed on the UI,
 * for the given userObject (parent).
 *
 * @param userObject	The domain object whose icon is required.
 */
abstract public ULCIcon getIcon(Object userObject);
/**
 * Internal method to locate the key corresponding to the 
 * given object in the hashTable.
 *
 */
protected int getIdFor(Object userObject) {
	return getRowIdFor(userObject);
}
/**
 * Override this method to return the label to be displayed on the UI,
 * for the given userObject (parent).
 *
 * @param userObject	The domain object whose label is required.
 */
abstract public String getLabel(Object userObject);
/**
 * Override this method to return the parent object
 * for the given userObject (child). This method is (only)
 * required in case operations (e.g. setSelection) are being
 * performed on a node which has not yet been uploaded to the
 * UI side.
 * By default, the Root will be assumed to be the parent. (!)
 * Note: As of ULC R3.1 getParentFor is deprecated. getParent
 *		 must be implemented to return the correct parent.
 *		 In the next release this abstract method will be deleted
 *		 and no dummy parent (eg root) will be returned. It will
 *		 be up to the programmer to specify such a behavior.
 *
 * @param child	The domain object whose parent is required.
 */
public Object getParent(Object child) {
	return getParentFor(child);
}
/**
 * Override this method to return the requested value for the 
 * specified cell.
 * This method needs to be overridden by subclasses if they
 * have added columns to the receiver's tree.
 *
 * @param colId	The key of the attribute to be retrieved.
 * @return Object 
 */
public Object getValue(String columnId, Object node) {
	if (columnId.equals(TREETABLE_FIRST_COLUMN_ATTRIBUTE_NAME)) {
		ULCLabel label = new ULCLabel();
		label.setLabel(getLabel(node));
		label.setIcon(getIcon(node));
		return label;
	} else
		return getValueAt(columnId, node);
}
/**
 * Override this method to return the requested value for the 
 * specified cell.
 * This method needs to be overridden by subclasses if they
 * have added columns to the receiver's tree.
 *
 * @param colId	The key of the attribute to be retrieved.
 * @return Object 
 */
public Object getValueAt(String columnId, Object node) {
	return "No value provided";
}
/**
 * Notify the receiver that the deletion of the given children
 * has happened. This is an internal method.
 *
 * @see #deleteChildren(Vector).
 */
protected void notifyDeletion(Vector removedChildren) {
	Object[] objects = new Object[removedChildren.size()];
	removedChildren.copyInto(objects);
	notifyDelete(objects);
}
/**
 * Notify the receiver that an adding of the given children
 * has happened for the given parent. This is an internal method.
 * Note that only the count of the number of children inserted, is
 * sent to the UI at this stage. The actual children will be loaded
 * lazily, on request from the UI.
 *
 * @see #addChildren(Object, Vector).
 */
protected void notifyInsertion(Object parent, Vector newlyAddedChildren, Vector indices) {
	// Note that the parent as well as the new children have
	//  been registered earlier. @see addChildren(parent, children).

	Anything a = new Anything();
	a.put("parent", getIdFor(parent));

	//How many NEW children have been added. This is *not* the new childCount
	//of the node. Just the increment on the existing count.
	a.put("howManyNewChildren", newlyAddedChildren.size());
	Anything in = new Anything();
	for (int i = 0; i < indices.size(); i++) {
		Anything aa = new Anything(((Integer)indices.elementAt(i)).intValue());
		in.append(aa);
	}
	a.put("indices", in);
	sendUI("insertChildren", a);
}
/**
 * Notify UI TreeModel that "everything" should be
 * reset: This would be done in order to change the root, say.
 * This is an internal method.
 *
 * @see #changed
 */
protected void reset() {
	getTreeItemList().setRoot(getRoot());
}
/**
 * Set the given icon for this object.
 *
 * @param newIcon 		ULCIcon The new icon.
 * @param userObject	The domain object whose icon is to be set.
 */
public void setIcon(ULCIcon newIcon, Object userObject) {
	notifyChange(TREETABLE_FIRST_COLUMN_ATTRIBUTE_NAME, userObject);
}
/**
 * Set the given label for this object.
 *
 * @param newLabel 		String 		The new label.
 * @param userObject	The domain object whose label is to be set.
 */
public void setLabel(String newLabel, Object userObject) {
	notifyChange(TREETABLE_FIRST_COLUMN_ATTRIBUTE_NAME, userObject);
}
/**
 * Override this method to store the given value in the specified cell.
 * The default implementation prints the given argument on the console.
 *
 * @param attributeName The key of the attribute that has changed.
 * @param rowIndex		The index of the row that has changed.
 * @param value 		The new <code>Object</code> value to be set.
 */
public void setValueAt(String attributeName, int rowIndex, Object value) {
	System.out.println("ULCTreeTableModel.setValueAt(" + attributeName + ", " + rowIndex + "): " + value.toString());
}
}
