package com.ibm.ulc.application;

/*
 * (c) Object Technology International 1997,1998.
 */
import java.util.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.UlcHashtable;
import com.ibm.ulc.comm.ORBConnection;

/**
 * The 'adapter' which maps the nodes in a tree to the underlying 
 * domain object(s) (implementing the hierarchical relationship).
 * Clients have to subclass from ULCTreeTableModel to at least implement the 
 * abstract methods defined in this class.
 * Typically subclasses do not storing the data themselves, but are adapters which 
 * redirect these calls to some other model or data structure.
 *
 * @see ULCTreeTable
 */
abstract public class ULCTreeTableModel extends ULCAbstractTableModel implements IHierarchyAdapter, IHierarchicalTableModel, IRowAdapter {
	/**
	 * The hierarchical itemList to be used by default
	 */
	protected ULCDefaultTreeItemList fHierarchicalItemList;

	/**
	 * The attribute name used for accessing the children of a node in the
	 * receiver
	 */
	protected String fChildrenAttribute;

	/**
	 * The attribute name used for accessing the parent of a node in the
	 * receiver
	 */	
	protected String fParentAttribute;
/**
 * Constructs a ULCTreeTableModel.
 */
public ULCTreeTableModel() {
}
/**
 * Constructs a TreeTableModel with the specifiend number of prefetch rows.
 *
 * @param prefetchCount 	determines the maximum number of children of a node 
 * 							that are transmitted in a single request.
 */
public ULCTreeTableModel(int prefetchCount) {
	super(prefetchCount);
}
/**
 * Add a new itemList to be served by the receiver. Only Hierarchical itemLists may
 * be used for the receiver.
 *
 * @param itemList	IItemList	The itemList to be added to the receiver's set of itemLists.
 * @see IHierarchicalItemList
 */
public void addItemList(IItemList itemList) {
	com.ibm.ulc.util.Assert.isTrue(itemList instanceof IHierarchicalItemList);
	super.addItemList(itemList);
}
/**
 * Answer an array with all the receiver's hierachical itemLists, including the default 
 * itemList, which is not contained in the <code>fItemLists</code>.
 */
protected IHierarchicalItemList[] allHierachicalItemLists() {
	Vector lists = new Vector();
	allItemLists(lists);
	IHierarchicalItemList[] answer = new IHierarchicalItemList[lists.size()];
	lists.copyInto(answer);
	return answer;
}
/**
 * Notify the UI TreeModel that the given objects (nodes) have
 * changed. This will cause these objects to be re-fetched from
 * the ULC to the UI. By default, the children of this node will
 * not be re-fetched.
 *
 * @params	userObjects 			Vector		the domain objects that have changed.
 *			shouldReloadChildren 	boolean 	should the children also be re-fetched
 *
 */
public void changed(Object[] userObjects, boolean shouldReloadChildren) {
	if (userObjects != null) {
		if (shouldReloadChildren) {
			int[] rowIds = getRowIdsFor(userObjects);
			startBatchingRequests();
			updateHierarchyCaches(rowIds, new String[] {getChildrenAttribute()});
			sendRowsChanged(rowIds, new String[0]);
			stopBatchingRequests();
		} else
			internalNotifyRowsChanged(getRowIdsFor(userObjects), new String[0]);
	}
}
/**
 * Notify the UI TreeModel that the given object (node) has
 * changed. This will cause this object to be re-fetched from
 * the ULC to the UI.
 *
 * @params	userObject 				Object		the domain object that has changed.
 *			shouldReloadChildren 	boolean 	should the children also be re-fetched
 *
 */
public void changed(Object userObject, boolean shouldReloadChildren) {
	changed(new Object[] {userObject}, shouldReloadChildren);
}
/**
 * Override this method to return the number of children
 * for the given userObject (parent).
 *
 * @param userObject	The domain object whose childCount is required.
 */
abstract public int getChildCount(Object userObject);
/**
 * Override this method to return the index of the given child object,
 * in the given parent bject. 
 * This method is (only) required in case operations (e.g. setSelection) 
 * are being performed on a node which has not yet been uploaded to the
 * UI side.
 *
 * By default, the child will be assumed to be the first child of this parent.
 *
 * @param child	The domain object whose parent is required.
 */
public int getChildIndexFor(Object child) {
	return getChildIndexFor(child, getParent(child));
}
/**
 * Override this method to return the index of the given child object,
 * in the given parent bject. 
 * This method is (only) required in case operations (e.g. setSelection) 
 * are being performed on a node which has not yet been uploaded to the
 * UI side.
 *
 * By default, the child will be assumed to be the first child of this parent.
 *
 * @param parent The domain object containing the given child.
 * @param child	The domain object whose parent is required.
 */
public int getChildIndexFor(Object child, Object parent) {
	trouble("getChildIndexFor(child, parent)", " needs to be over-written.");
	return 0;
}
/**
 * Override this method to return the collection of children
 * for the given userObject (parent).
 *
 * @param parent	The domain object whose children are required.
 */
abstract public Vector getChildren(Object parent);
/**
 * Answer the attribute name to be used for accessing the children
 * for any node in the receiver.
 */
public String getChildrenAttribute() {
	return fChildrenAttribute;
}
/*
 * Answer the receiver's default children attribute. Subclasses may want to 
 * reimplement this method.
 *
 * @see isChildrenAttribute
 */
protected String getDefaultChildrenAttribute() {
	return "defaultChildrenAttribute";
}
/**
 * Answer the receiver's default hierarchical itemList. It is not configured with 
 * parent / children attributes.
 */
protected IHierarchicalItemList getDefaultHierarchicalItemList() {
	return new ULCDefaultTreeItemList();
}
/*
 * Answer the receiver's default parent attribute. Subclasses may want to reimplement 
 * this method.
 *
 * @see isParentAttribute
 */
 protected String getDefaultParentAttribute() {
	return "defaultParentAttribute";
}
/**
 * Implementing the ULCTreeModel API:
 * Answer the root (node) for the receiver.
 */
public Object getDefaultRoot() {
	return getRoot();
}
/**
 * Answer the receiver's current hierarchy-based itemList. 
 */
public IHierarchicalItemList getHierarchicalItemList() {
	return fHierarchicalItemList;
}
/**
 * Answer the receiver's current index based itemList. This API is used to provide a default itemlist
 * for list widgets and combobox. The receiver does not support index based access to its data. Answer
 * null. Subclasses that do provide a default indexed itemList should reimplement this method.
 *
 * This receiver works only on hierarchical itemlists, has no indexed itemlists.
 * Hence this method should never get called.
 *
 */
public IIndexedItemList getIndexedItemList() {
	noIndexedAccessSupported("getIndexedItemList()");
	return null;
}
/**
 * Answer the attribute name to be used for accessing the parent
 * for any node in the receiver.
 */
public java.lang.String getParentAttribute() {
	return fParentAttribute;
}
/**
 * Override this method to return the parent object
 * for the given userObject (child). This method is (only)
 * required in case operations (e.g. setSelection) are being
 * performed on a node which has not yet been uploaded to the
 * UI side.
 * By default, the Root will be assumed to be the parent. (!)
 *
 * @param child	The domain object whose parent is required.
 * @deprecated	As of ULC R3.1, use getParent(Object) instead
 *				and implement the correct behavior (the framework
 *				will no longer return the root as default in the
 *				next release).
 * @see	#getParent(Object)
 */
public Object getParentFor(Object child) {
	System.out.println("ULCTreeTableModel.getParentFor(child) needs to be over-written.");
	return getRoot();
}
/**
 * Override this method to return the root object of the tree
 */
abstract public Object getRoot();
/**
 * This internal method is called when accessing the root for the
 * first time. Register the root and then answer it.
 */
protected Object getRootNode() {
	return getRoot();
}
/*
 * Answer the adapter for the receiver's rows. It is used to access the values of the rows.
 */
protected IRowAdapter getRowAdapter() {
	return this;
}
/**
 * When the receiver is used as TreeModel only, answer zero. Subclasses should reimplement
 * this method if they also serve data for other list widgets. In that case they should answer
 * the total number of rows available in the data base.
 */
public int getRowCount() {
	return 0;
}
/**
 * Override this method to return the index for the requested oid. This is an abstract method and must be reimplemented by subclasses
 * if index based access to nodes is needed..
 *
 * @param oid		the oid of the required node
 */
public int getRowIndexForOid(int oid) {
	trouble("ULCTreeTableModel#getRowIndexForOid", "is an abstract method. If access by indes is required in the application, it must be implemented by subclasses");
	return -1;
}
/**
 * Answer the receiver's default itemList for the receiver's tree.
 *
 * The receiver's <code>ULCDefaultTreeItemList</code> is responsible for the order in which
 * the receiver's nodes are displayed.
 *
 */
protected ULCDefaultTreeItemList getTreeItemList() {
	return fHierarchicalItemList;
}
/**
 * Override this method to return the requested value for the 
 * specified cell.
 * This method needs to be overridden by subclasses if they
 * have added columns to the receiver's tree.
 *
 * @param colId	The key of the attribute to be retrieved.
 * @param node	The object whose value at the above attribute is required.
 * @return Object 
 */
abstract public Object getValue(String columnId, Object node);
/**
 * Answer the value of the value for attributeName of the row specified by rowIndex.
 *
 * @param attributeName	The key of the attribute to be retrieved.
 * @param rowIndex		The index of the row being accessed.
 */
public Object getValueAt(String attributeName, int rowIndex) {
	noIndexedAccessSupported("getValueAt(String, int)");
	return null;
}
/**
 * Initialize the receiver
 */
protected void initialize() {
	super.initialize();
	resetDefaultHierarchicalItemList(null);
	setParentAttribute(getDefaultParentAttribute());
	setChildrenAttribute(getDefaultChildrenAttribute());
}
/**
 * Notify the UI that the rows identified by rowIds have changed.
 *
 * Only those rows whose uploaded attributes includes <code>attributeName</code> are uploaded.
 * If <code>attributeName</code> is null, the row will be refreshed completely, if it had previously
 * uploaded at least one attribute.
 *
 * @param	rowIds			int[]		The oid's of the rows which were added
 * @param	attributeNames	String[]	The attributeNames which need to be uploaded
 */
protected void internalNotifyRowsChanged(int[] rowIds, String[] attributeNames) {
	startBatchingRequests();
	updateHierarchyCaches(rowIds, attributeNames);
	sendRowsChanged(rowIds, attributeNames);
	stopBatchingRequests();
}
/**
 * Answer true if the specified attributeName is the attribute
 * defining the children of a node.
 */
public boolean isChildrenAttribute(java.lang.String attributeName) {
	return attributeName.equals(getChildrenAttribute());
}
/**
 * Answer true if the specified attributeName is the attribute
 * defining the parentage of a node.
 */
public boolean isParentAttribute(java.lang.String attributeName) {
	return attributeName.equals(getParentAttribute());
}
private void noIndexedAccessSupported(String string) {
	String className = getClass().getName();
	System.out.println("Warning " + className + "." + string + ": " + "does not support indexed access");
}
/**
 * Reset the receiver's default itemList.
 */
protected void resetDefaultHierarchicalItemList(IHierarchicalItemList itemList) {
	if (itemList == null || itemList == getTreeItemList()) {
		fHierarchicalItemList = (ULCDefaultTreeItemList) getDefaultHierarchicalItemList();
		getTreeItemList().setModel(this);
	}
}
/**
 * Reset the receiver's default item list. This is implemented by the superclass as abstract, so this class needs
 * an implementation. The message is however ignored because it should not reach the receiver.
 *
 */
protected void resetDefaultItemList(IIndexedItemList list) {
	noIndexedAccessSupported("resetDefaultItemList(IIndexedItemList)");
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveItemLists(Anything a) {
}
/**
 * Answer the attribute name to be used for accessing the children
 * for any node in the receiver.
 *
 * @param newChildrenAttribute 	String	: the new attribute name
 */
public void setChildrenAttribute(java.lang.String newChildrenAttribute) {
	fChildrenAttribute = newChildrenAttribute;
}
/**
 * Answer the attribute name to be used for accessing the children
 * for any node in the receiver.
 *
 * @param newParentAttribute 	String	: the new attribute name
 */
public void setParentAttribute(String newParentAttribute) {
	fParentAttribute = newParentAttribute;
}
/**
 * Set the receiver's RowCache to it's default value.
 */
protected void setRowCache() {
	setRowCache(new ULCHierarchicalRowCache(this));
}
/**
 * Override this method to store the given value in the specified cell.
 * The default implementation prints the given argument on the console.
 *
 * @param attributeName The key of the attribute that has changed.
 * @param rowIndex		The index of the row that has changed.
 * @param value 		The new <code>Object</code> value to be set.
 */
public void setValue(String attributeName, Object object, Object value) {
	System.out.println("ULCTreeTableModel.setValueAt(" + attributeName + ", " + object.toString() + "): " + value.toString());
}
/**
 * The rows specified by rowIds have changed in the attributes specified.
 * If the attributes indicate that children of the specified nodes have changed,
 * clean out all cached child nodes from both the itemLists concerned and the
 * UI ItemCache. The item cache is touched to force a reload / redefinition of
 * nodes that may have been moved in the hierarchy.
 *
 * @param	rowIds			int[]		The oid's of the rows to be updated
 * @param	attributeNames	String[]	The attributeNames which need to be updated
 */
protected void updateHierarchyCaches(int[] rowIds, String[] attributeNames) {
	IHierarchicalItemList[] lists = allHierachicalItemLists();
	UlcHashtable removedOids = new UlcHashtable();
	for (int i = 0; i < lists.length; i++) {
		lists[i].collectChildOids(removedOids, rowIds, attributeNames);
	}
	if (removedOids.size() > 0) {
		Enumeration keys = removedOids.keys();
		int[] oidArray = new int[removedOids.size()];
		int counter = 0;
		while (keys.hasMoreElements()) {
			oidArray[counter++] = ((Integer) keys.nextElement()).intValue();
		}
		Anything args = new Anything();
		args.put("rowids", new Anything(oidArray));
		sendUI("removeRows", args);
		for (int i = 0; i < oidArray.length; i++)
			getRowCache().resetRowId(oidArray[i]);
	}
	for (int i = 0; i < lists.length; i++) {
		lists[i].notifyRowsChanged(rowIds, attributeNames);
	}
}
}
