package com.ibm.ulc.comm;

/*
 * Copyright (c) 1998 Object Technology International Inc.
 */
import java.io.*;
import java.net.*;
import com.ibm.ulc.base.IDefaults;


/**
 * An implemention for ULC transports using sockets.
 *
 * @see UlcTransport
 */
public class UlcSocketTransport extends UlcTransport {
	private Socket fSocket= null;
	private OutputStream fOutStream= null;
	private InputStream fInStream= null;
	private int SOCKET_TIMEOUT= Integer.parseInt(getProperties().getProperty("UlcSocketTransport.timeout", "100"));
/**
 * UlcSocketTransport constructor.
 */
public UlcSocketTransport() {
	if (fDebug)
		System.out.println("UlcSocketTransport().constructor");
}
/**
 * UlcSocketTransport constructor.
 * Will use the passed in URL to create a socket connection.
 * @param urlString java.lang.String
 */
public UlcSocketTransport(String urlString) {
	super(urlString);
	if (fDebug)
		System.out.println("UlcSocketTransport(String).constructor");		
}
/**
 * UlcSocketTransport constructor.
 * Will use the passed in socket for connection.
 */
public UlcSocketTransport(Socket socket) {
	if (fDebug)
		System.out.println("UlcSocketTransport(Socket).constructor");	
	fSocket= socket;
	int port= fSocket.getLocalPort();
	String host= UlcConnection.getLocalHost();
	setUrlString("ulc://" + host + ":" + port);
}
/**
 * Factory method for creating sockets
 */
public Socket createSocket(String urlString)
throws UnknownHostException, IOException {
/*
 Once the SocketFactory stuff has been deprecated we can use the following code:
 -------------------------------------------------------------------------------
	// make sure hostnames work for applets:
	// convert real name to "localhost"
	String host= mapHost(UlcConnection.getHostFromUrlString(urlString));
	int port= UlcConnection.getPortFromUrlString(urlString);
	if (port == UlcConnection.INVALIDPORT)
		port= com.ibm.ulc.orb.Common.SERVER_PORT;

	return new Socket(host,port);
*/

	return Server.getSocketFactory().createSocket(urlString);
}
/**
 * During the shutdown of the connection any pending requests in tbe
 * output queue should be flushed before the connection is taken down.
 * Subclasses should override this method to implement any logic necessary to flush the
 * output queue
 */
protected void flushWriter() {
	while ((fOutStream != null) && (fWriterQueue != null) && (fWriterQueue.length() > 0)) {
		try {
			Thread.sleep(250); // wait here to give the writer thread a chance to write all pending requests.
		} catch (Exception e) {
		}
	}
	// now that the queue is empty we can flush 
	// the queue safely in this thread since 
	// the writer thread will be idle.
	sendDirect(null,true);
}
public String getLocalUrlString() {
	if (fSocket != null) {
		String host= null;
		InetAddress localAddr= fSocket.getLocalAddress();
		if (UlcConnection.fgIPAddr)
			host= localAddr.getHostAddress();
		else
			host= localAddr.getHostName();
		return
			"ulc://"
			+ host
			+ ":"
			+ fSocket.getPort();
	}
	return null;
}
public String getRemoteUrlString() {
	if (fSocket != null) {
		String host= null;
		InetAddress localAddr= fSocket.getInetAddress();
		if (UlcConnection.fgIPAddr)
			host= localAddr.getHostAddress();
		else
			host= localAddr.getHostName();
		return
			"ulc://"
			+ host
			+ ":"
			+ fSocket.getPort();
	}
	return null;
}
public Socket getSocket() {
	return fSocket;
}
/**
 * Initialize connection to peer if necessary.
 */
protected void initPeerConnection() throws UlcTransportException {
	if (fDebug)
		System.out.println("<UlcSocketTransport.initPeerConnection(). url=" + getUrlString());

	if (fOutStream == null) {
		try {
			if (fSocket == null)
				fSocket= createSocket(getUrlString());
			fOutStream= new BufferedOutputStream(fSocket.getOutputStream());
		}
		catch (UnknownHostException e) {
			if (fDebug)
				System.out.println("UlcSocketTransport.initPeerConnection(). Unknown host");
			fOutStream= null;
			UlcTransportException ex=
				new UlcTransportException(TRANSPORT_EX_INIT_CONNECTION_EXCEPTION,e);
			throw ex;
		}
		catch (IOException e) {
			if (fDebug)
				System.out.println("UlcSocketTransport.initPeerConnection(). IO Exception");
			fOutStream= null;
			UlcTransportException ex=
				new UlcTransportException(TRANSPORT_EX_INIT_CONNECTION_EXCEPTION,e);
			throw ex;
		}
		catch (UnknownError e) {
			if (fDebug)
				System.out.println("UlcSocketTransport.initPeerConnection(). unknown trouble");
			fOutStream= null;
			UlcTransportException ex=
				new UlcTransportException(TRANSPORT_EX_INIT_CONNECTION_ERROR,e);
			throw ex;
		}
	}

	if (fInStream == null) {
		try {
			fInStream= new BufferedInputStream(fSocket.getInputStream());
		}
		catch (IOException e) {
			if (fDebug)
				System.out.println("UlcSocketTransport.initPeerConnection(). IO Exception");
			fInStream= null;
			fOutStream= null;
			UlcTransportException ex=
				new UlcTransportException(TRANSPORT_EX_INIT_CONNECTION_EXCEPTION,e);
			throw ex;
		}
	}
	
	if (fDebug)
		System.out.println(">UlcSocketTransport.initPeerConnection().");
}
protected void readRequests() {
	UlcTransportException exception = null;
	int port = fSocket.getLocalPort();
	if (fDebug)
		System.out.println("Connection started for port " + port);
	try {
		while (!fTerminated) {
			Request r = fConnection.createRequest();
			try {
				fSocket.setSoTimeout(SOCKET_TIMEOUT);
				while (!fTerminated && !r.isAvailable(fInStream)) {};
				fSocket.setSoTimeout(0);
				if (fTerminated)
					return;
				boolean successful= r.readEx(fInStream);
				if (fTerminated)
					return;
				if (!successful) {
					String msg = "UlcSocketTransport. read error";
					if (fDebug)
						System.out.println(msg);
					exception = new UlcTransportException(msg, TRANSPORT_EX_CONNECTION_INTERRUPTED, (Exception) null);
					break;
				}
				// everything ok -> receive read request
				receive(r);
			} catch (IOException e) {
				if (fTerminated)
					// could have been set while blocked in read
					return;
				String msg = "UlcSocketTransport. read IO Exception";
				if (fDebug) {
					System.out.println(msg);
					e.printStackTrace();
				}
				exception = new UlcTransportException(TRANSPORT_EX_CONNECTION_INTERRUPTED, e);
				break;
			}
		}
	} catch (Exception e) {
		String msg = "UlcSocketTransport. got exception while reading";
		if (fDebug) {
			System.out.println(msg);
			e.printStackTrace();
		}
		exception = new UlcTransportException(TRANSPORT_EX_CONNECTION_INTERRUPTED, e);
	}
	if (!fTerminated) { // we decided here to stop processing!
		// close connection (by posting request)
		if (fConnection != null)
			fConnection.postRequest(new CloseRequest(fConnection, exception));
	}
	if (fDebug)
		System.out.println("Connection for port " + port + " terminated");
}
/**
 * Does the ULC transport need an asynchronous reader?
 * If it does then this class must override readRequests().
 *
 * Note: Normally this is needed if the LOW LEVEL transport itself
 *       is not synchronous (e.g. SocketTranport).
 */
public boolean readsAsynch() {
	return true;
}
protected void sendDirect(Request r, boolean flush) {
	if (fOutStream != null) {
		if (r != null)
			r.write(fOutStream);
		if (flush) {
			try {
				fOutStream.flush();
			}
			catch (InterruptedIOException ex) {
				// We've been interrupted.  Make sure we're still interrupted.
				Thread.currentThread().interrupt();
			}
			catch (IOException e) {
				System.out.println("UlcSocketTransport.sendDirect(). IOException");
				e.printStackTrace();
			}
		}
	}
}
protected void termPeerConnection() {
	if (fSocket != null) {
		try {
			fSocket.close();
		} catch (IOException e) {
			System.out.println("UlcSocketTransport.termConnection(). IOException while closing socket");
		}
		fSocket= null;
	}
	fOutStream= null;
}
protected void writeRequests() {
	while (fOutStream != null) {
		if (fWriterQueue == null)
			break;
		Request r = (Request) fWriterQueue.take(0);
		if (fTerminated)
			break;
		if (r != null) {
			sendDirect(r,false);
			if (fWriterQueue == null)
				break;
			if (fWriterQueue.length() <= 0)   // queue empty?
				sendDirect(null,true);   // just flush
		}
		if (fTerminated && (fWriterQueue == null || fWriterQueue.length() <= 0))
			break;
	}
	if (fDebug)
		System.out.println("UlcSocketTransport.writeRequests(). At end");
}
}
