package com.ibm.ulc.comm;

/*
 * Copyright (c) 1998 Object Technology International Inc.
 */
import java.net.*;
import java.io.*;
import java.util.*;
import com.ibm.ulc.util.UnboundedBuffer;
import com.ibm.ulc.base.IDefaults;

/**
 * An abstraction for ULC transports.
 *
 * @see UlcConnection
 */
public abstract class UlcTransport implements Serializable, IDefaults {
	public static boolean fgDebug= false;
	private static Properties fgProperties = null;

	protected boolean fDebug= false;
	
	/* Corresponding connection (each connection has one transport and vice versa) */
	protected UlcConnection fConnection = null;

	/* Adressing */
	private String fUrlString = null;
	// caches for URL content:
	private String fUrlHost = null;
	private int fUrlPort = UlcConnection.INVALIDPORT;
	private String fUrlFile = null;
	private String fUrlReference = null;

	/* Multi-threading */
	private Thread fReaderThread = null;
	private Thread fWriterThread = null;
	protected UnboundedBuffer fWriterQueue = null;
	protected boolean fTerminated = true;
/**
 * UlcTransport constructor 
 */
protected UlcTransport() {
	fDebug= fgDebug;
}
/**
 * UlcTransport constructor
 *
 * @param urlString java.lang.String
 */
public UlcTransport(String urlString) {
	this();
	fUrlString= urlString;
}
public void close() {
	flushWriter(); // this must be done before the fTerminated is set to true

	fTerminated = true;
	if (fReaderThread != null) {
		stopReader();
		fReaderThread = null;
	}
	if (fWriterThread != null) {
		stopWriter();
		fWriterThread = null;
		fWriterQueue = null;
	}
	termPeerConnection();
	fConnection = null;
}
/**
 * The factory method to create a concrete transport depending on the
 * protocol specified in the URL. The name of the concrete transport class
 * is read in from a property file.
 *
 * @param urlString java.lang.String
 */
public static UlcTransport create(String urlString) throws UlcTransportException {
	UlcTransport transport= null;
	String transportClass= null;
	String protocol= UlcConnection.getProtocolFromUrlString(urlString);

	Properties prop= getProperties();
	
	// get class name of concrete transport for desired protocol
	transportClass=
		prop.getProperty("UlcTransport.classForProtocol." + protocol);

	if (transportClass != null) {   // to be dynamically created
		try {
			Class cl= Class.forName(transportClass);
			if (cl != null) {
				transport= (UlcTransport) cl.newInstance();
				transport.setUrlString(urlString);
			}
			else {
				String msg= "UlcTransport.create(): could not find " + transportClass;
				if (fgDebug)
					System.out.println(msg);
				UlcTransportException ex=
					new UlcTransportException(msg,TRANSPORT_EX_INIT_CONNECTION_EXCEPTION,(Exception)null);
				throw ex;
			}
		}
		catch (ClassNotFoundException e) {
			if (fgDebug)
				System.out.println("UlcTransport.create(): ClassNotFoundException for " + transportClass + " - " + e.toString());
			UlcTransportException ex=
				new UlcTransportException(TRANSPORT_EX_INIT_CONNECTION_EXCEPTION,e);
			throw ex;
		}
		catch (InstantiationException e) {
			if (fgDebug)
				System.out.println("UlcTransport.create(): InstantiationException for " + transportClass + " - " + e.toString());
			UlcTransportException ex=
				new UlcTransportException(TRANSPORT_EX_INIT_CONNECTION_EXCEPTION,e);
			throw ex;
		}
		catch (IllegalAccessException e) {
			if (fgDebug)
				System.out.println("UlcTransport.create(): IllegalAccessException for " + transportClass + " - " + e.toString());
			UlcTransportException ex=
				new UlcTransportException(TRANSPORT_EX_INIT_CONNECTION_EXCEPTION,e);
			throw ex;
		}
		catch (ClassCastException e) {
			if (fgDebug)
				System.out.println("UlcTransport.create(): ClassCastException for " + transportClass + " - " + e.toString());
			UlcTransportException ex=
				new UlcTransportException(TRANSPORT_EX_INIT_CONNECTION_EXCEPTION,e);
			throw ex;
		}
	}
	
	return transport;
}
static Properties defaultProperties() {
	Properties prop = new Properties();

	// *** default "ulc" transport protocol	
	prop.put("UlcTransport.classForProtocol.ulc", 
			"com.ibm.ulc.comm.UlcSocketTransport");

	// *** default "iiop" transport protocol	
	prop.put("UlcTransport.classForProtocol.iiop",
			"com.ibm.ulc.corba.UlcCorbaTransport");

	// *** default "ejb" transport protocol
	prop.put("UlcTransport.classForProtocol.ejb",
			"com.ibm.ulc.ejb.UlcEJBTransport");
	
	// *** default "http" transport protocol
	prop.put("UlcTransport.classForProtocol.http",
			"com.ibm.ulc.http.UlcHttpTransport");

	// *** default "socket" server transport protocol	
	prop.put("UlcTransportServer.classForProtocol.ulc", 
			"com.ibm.ulc.comm.UlcSocketTransportServer");

	// *** default "iiop" server transport protocol
	prop.put("UlcTransportServer.classForProtocol.iiop",
			"com.ibm.ulc.corba.UlcCorbaTransportServer");

	// *** default "ejb" server transport protocol
	prop.put("UlcTransportServer.classForProtocol.ejb",
			"com.ibm.ulc.ejb.UlcEJBTransportServer");

	// *** default "http" server transport protocol
	prop.put("UlcTransportServer.classForProtocol.http",
			"com.ibm.ulc.http.UlcHttpTransportServer");

	return prop;
}
/**
 * During the shutdown of the connection any pending requests in tbe
 * output queue should be flushed before the connection is taken down.
 * Subclasses should override this method to implement any logic necessary to flush the
 * output queue
 */
protected void flushWriter() {
}
public UlcConnection getConnection() {
	return fConnection;
}
public String getHost() {
	if (fUrlHost == null) {
		fUrlHost= UlcConnection.getHostFromUrlString(fUrlString);
		if (fUrlHost == null)
			fUrlHost= "***unknown***";
	}
	return fUrlHost;
}
/**
 * Return an URL String describing this transports local endpoint.
 */
public abstract String getLocalUrlString();
public int getPort() {
	if (fUrlPort == UlcConnection.INVALIDPORT)
		fUrlPort= UlcConnection.getPortFromUrlString(fUrlString);
	return fUrlPort;
}
static public Properties getProperties() {
	if (fgProperties == null)
		loadProperties();
	return fgProperties;
}
/**
 * Return an URL String describing this transports remote endpoint.
 * Default implementation is to return the localUrlString.
 */
public String getRemoteUrlString() {
	return getLocalUrlString();
}
public String getUrlFile() {
	if (fUrlFile == null) {
		fUrlFile= UlcConnection.getFileFromUrlString(fUrlString);
		if (fUrlFile == null)
			fUrlFile= "***unknown***";
	}
	return fUrlFile;
}
/**
 * Return an URL String describing this transport's remote endpoint.
 */
public String getUrlString() {
	return fUrlString;
}
/**
 *  Method to perform connection initialization with peer.
 */
protected abstract void initPeerConnection() throws UlcTransportException;
/**
 * Is logging of debug information enabled?
 */
public boolean isDebugModeOn() {
	return fDebug;
}
private static synchronized boolean loadProperties() {
	if (fgProperties == null) {
		// read property file
		fgProperties = new Properties(defaultProperties());
		try { // we first search the root path to allow customers to override the default properties.
			InputStream propIS = UlcTransport.class.getResourceAsStream("/UlcTransport.properties");
			if (propIS == null) {
				propIS = UlcTransport.class.getResourceAsStream("UlcTransport.properties");
			}
			if (propIS == null) {
				System.out.println("UlcTransport: cannot load properties using defaults!");
				return false;
			}
			fgProperties.load(propIS);
		} catch (IOException e) {
			System.out.println("UlcTransport: cannot load properties using defaults! " + e);
			return false;
		}
		if (fgDebug)
			fgProperties.list(System.out);
	}
	return true;
}
String mapHost(String hostToMap) {
	// Convert real name to something acceptable to Web browsers if necessary,
	// i.e. if host is addressing the local machine.
	// (acceptable to Web browsers: "localhost", "127.0.0.1")
	String host= hostToMap;
	try {
		InetAddress localAddr= InetAddress.getLocalHost();
		if (hostToMap.equals(localAddr.getHostName())
			 || hostToMap.equals(localAddr.getHostAddress()))
			host= UlcConnection.getLocalHost();   // ip address or host name 
	} catch (UnknownHostException uhe) {
		System.out.println("@@@@@ DefaultSocketFactory.mapHost(): cannot get local host!?");
		host = "127.0.0.1";
	}

	return host;
}
/**
 * Read loop. Data read from Transport must be passed as requests
 * to associated request processor for execution.
 */
protected void readRequests() {
	if (fReaderThread != null)
		System.out.println("UlcTransport.readRequests(). MUST be overridden for synchronous transports!");
}
/**
 * Does the ULC transport need an asynchronous reader?
 * If it does then this class must override readRequests().
 *
 * Note: Normally this is needed if the LOW LEVEL transport itself
 *       is not synchronous (e.g. SocketTranport).
 */
public abstract boolean readsAsynch();
public final void receive(Request r) {
	if (fDebug)
		System.out.println("UlcTransport.Recv: " + r.getName() + " contextID: " + ((ORBConnection) this.getConnection()).getContextId());
	fConnection.receive(r);
}
/**
 * Sends a request. Requests are sent asynchronously.
 */
public final void send(Request r) {
	if (fDebug)
		System.out.println("UlcTransport.send: " + r.getName() + " contextID: " + ((ORBConnection) this.getConnection()).getContextId());	
	if (fWriterQueue != null)
		fWriterQueue.put(r);
	else
		sendDirect(r,true);
}
/**
 * Sends a request directly (without using WriterThread)
 */
protected abstract void sendDirect(Request r, boolean flush);
public void setConnection(UlcConnection connection) {
	fConnection= connection;
}
protected void setUrlString(String urlString) {
	fUrlString= urlString;
}
public final void start() throws UlcTransportException {

	if (!fTerminated)   // already started
		return;
	fTerminated= false;

	// initiate connection to peer
	try {
		initPeerConnection();
	}
	catch(UlcTransportException ex) {
		fTerminated= true;
		throw ex;
	}
	
	// start writer thread
	// (gets requests from writer queue and writes it to transport)
	if (writesAsynch()) {
		fWriterQueue= new UnboundedBuffer();
		startAsynchWriter();
	}
	// start reader thread if necessary
	// (reads data from transport and receives it as requests)
	if (readsAsynch())
		startAsynchReader();
	
}
private void startAsynchReader() {
	fReaderThread=
		new Reader(this, "UlcTransport Reader (URL: " + fUrlString + ")");
	fReaderThread.start();
}
private void startAsynchWriter() {
	fWriterThread=
		new Writer(this, "UlcTransport Writer (URL: " + fUrlString + ")");
	fWriterThread.start();
}
private void stopReader() {
	if (fDebug)
		System.out.println("UlcTransport.stopReader()");
	// wait for the thread to die
	try {
		fReaderThread.join();
	} catch (InterruptedException e) {
		System.out.println("WARNING: UlcTransport.stopReader(). Thread interrupted during join");
		e.printStackTrace();
	}
	if (fDebug)
		System.out.println("UlcTransport.stopReader(). Reader thread died");
}
private void stopWriter() {
	if (fDebug)
		System.out.println("UlcTransport.stopWriter()");
	// tell the thread to stop
	send(new ExitWriterRequest());
	// wait for it to die
	try {
		fWriterThread.join(5);
	} catch (Exception e) {
		System.out.println("UlcTransport.stopWriter(). Exception during join");
		e.printStackTrace();
	}
	if (fDebug)
		System.out.println("UlcTransport.stopWriter(). Writer thread died");
}
/**
 *  Method to perform connection termination with peer.
 */
protected abstract void termPeerConnection();
/**
 * Write loop. Requests that are coming in from associated write-request
 * queue must be written to outgoing transport.
 *
 * This is normally used if the transport is synchronous i.e. waiting to
 * be processed by the receiver (e.g. CORBA, HTTP).
 */
protected void writeRequests() {
	if (fWriterThread != null)
		System.out.println("UlcTransport.writeRequests(). MUST be overridden for asynchronous ULC transports!");
}
/**
 * Does the ULC transport force an asynchronous write by posting
 * to an internal write request queue?
 * If it does then this class must override writeRequests().
 *
 * Note: Normally this is needed if the LOW LEVEL transport itself
 *       is synchronous i.e. waits for the receiver to process the
 *       request (e.g. CORBA, HTTP) or if we want to guaranteed async
 *		 sends. For ULC we currently want to do that, therefore the
 *		 default is <code>true</code>
 */
public boolean writesAsynch() {
	return true;
}
}
