package com.ibm.ulc.examples.ManagementStructure;

import java.util.Vector;
import com.ibm.ulc.comm.Request;
import com.ibm.ulc.base.IDefaults;
import com.ibm.ulc.application.*;

/**
 * This class is responsible for the editing of the Management Structure
 * application. Editing includes building up the whole UI using the ULC
 * widgets, and setting up all the adapters, callbacks etc. which are 
 * required for the actual editing of the application.
 *
 * @see 		#edit(ULCApplication)
 * @see 		#build()
 *
 */
public class ManagementStructureEditor {
	
	/**
	 * The Application on which the editor opens
	 */
	protected 	ManagementStructure			fManagementStructure;

	/**
	 * The shell into which all the widgets are built
	 */
	protected 	ULCShell 					fShell;

	/**
	 * Adapter for the actual employee tree (hierarchy)
	 */
	protected 	ManagementStructureAdapter 	fTreeModel;

	/**
	 * Adapter for the selected employee details
	 */
	protected 	EmployeeAdapter 			fSelectedEmployeeAdapter;	

	/**
	 * ULCTreeRowModel to track the tree node selection without
	 * going to the ULC side.
	 */
	 protected ULCTreeRowModel 		fSelectedEmployeeRowModel;
	 
	/**
	 * Adapter for a new employee details
	 */
	protected 	EmployeeAdapter 			fNewEmployeeAdapter;

	/**
	 * The label for the aplication status
	 */
	protected 	ULCLabel 					fStatusLabel;

	/**
	 * The tree is cached since it is required for 
	 * enabling/disabling operations.
	 */
	protected 	ULCTree 					fEmployeesTree; 

	/**
	 * Field showing the selected node(s).
	 */
	protected ULCField 	fSelectedNodesField;

	/**
	 * Boolean determining whether a node gets reloaded everytime
	 * it is expanded.
	 */
	 protected boolean fReloadOnExpansion;
	 
	/**
	 * The width and height for the application shell
	 */
	protected static final int SHELL_WIDTH= 800; 
	protected static final int SHELL_HEIGHT= 500; 
public ManagementStructureEditor(ManagementStructure mgmtStructure) {
	fManagementStructure= mgmtStructure;
	fTreeModel= new ManagementStructureAdapter(this);
	fSelectedEmployeeAdapter= new EmployeeAdapter();
	fNewEmployeeAdapter= new EmployeeAdapter();
}
/**
 * Add numerous new employees to the currently selected employee.
 * Test that these get inserted at the right indices.
 */
public void addEmployeesAtPreSpecifiedIndices() {
	Employee currentBoss = getSelectedEmployee();
	Employee newEmployee = null;
	Vector children = new Vector();
	for (int jj = 0; jj < 5; jj++) {
		newEmployee = new Employee("Inserted: " + jj, "InsertTitle: " + jj);
		if (jj == 0)	newEmployee.setBoss(currentBoss, 2);
		if (jj == 1)	newEmployee.setBoss(currentBoss, 4);
		if (jj == 2)	newEmployee.setBoss(currentBoss, 6);
		if (jj == 3)	newEmployee.setBoss(currentBoss, 8);
		if (jj == 4)	newEmployee.setBoss(currentBoss, 10);
		children.addElement(newEmployee);
	}
	Vector indices = new Vector();
	indices.addElement(new Integer(2));
	indices.addElement(new Integer(4));
	indices.addElement(new Integer(6));
	indices.addElement(new Integer(8));
	indices.addElement(new Integer(10));
	fTreeModel.addChildren(currentBoss, children, indices);
	// Refresh the boss of the added employee, since his childCount has increased.
	currentBoss.setTitle("ChangedBoss");
	fTreeModel.changed(currentBoss, false);
	employeeAdditionPostProcessing(newEmployee);
}
/**
 * Add a new employee to the currently selected employee,
 * at the specified index. Note that if this index is invalid,
 * the UI will add the new node at the end of the current list
 * of children for the selected employee.
 */
public void addNewEmployee(int index) {
	Employee currentBoss= getSelectedEmployee();
	fNewEmployeeAdapter.saveInput(); // flush the user-data for the new employee
	Employee newEmployee= fNewEmployeeAdapter.getEmployee();
	
	// Now add the new employee at the right position, else when the
	// UI asks for specific positions, we'll go haywire !
	newEmployee.setBoss(currentBoss, index);

	// Inform the treeModel that this child is now to be added on the UI as well.
	fTreeModel.addChild(currentBoss, newEmployee, index);
	// Refresh the boss of the added employee, since his childCount has increased.
	currentBoss.setTitle("ChangedBoss"); //Testing that the changed label goes to UI.
	fTreeModel.changed(currentBoss, false);

	// This is a convenient place to test the setIcon API
	testIconSetting(currentBoss);
	
	employeeAdditionPostProcessing(newEmployee);
}
/**
 * Add a new employee to the parent of the currently selected employee,
 * at a position immediately after that of the currently selected employee.
 */
public void addNewEmployeeAfter(Employee selectedObj) {
	Employee boss= selectedObj.getBoss();
	fNewEmployeeAdapter.saveInput(); // flush the user-data for the new employee
	Employee newEmployee= fNewEmployeeAdapter.getEmployee();
	
	// Now add the new employee at the right position, else when the
	// UI asks for specific positions, we'll go haywire !
	newEmployee.setBossAfter(boss, selectedObj);

	// Inform the treeModel that this child is now to be added on the UI as well.
	fTreeModel.addChildAfter(boss, newEmployee, selectedObj);
	// Refresh the boss of the added employee, since his childCount has increased.
	boss.setTitle("BossChanged");
	fTreeModel.changed(boss, false);

	employeeAdditionPostProcessing(newEmployee);
}
/**
 * Add a new employee to the parent of the currently selected employee,
 * at a position immediately before that of the currently selected employee.
 */
public void addNewEmployeeBefore(Employee selectedObj) {
	Employee boss= selectedObj.getBoss();
	fNewEmployeeAdapter.saveInput(); // flush the user-data for the new employee
	Employee newEmployee= fNewEmployeeAdapter.getEmployee();
	
	// Now add the new employee at the right position, else when the
	// UI asks for specific positions, we'll go haywire !
	newEmployee.setBossBefore(boss, selectedObj);

	// Inform the treeModel that this child is now to be added on the UI as well.
	fTreeModel.addChildBefore(boss, newEmployee, selectedObj);
	// Refresh the boss of the added employee, since his childCount has increased.
	boss.setTitle("BossChanged");
	fTreeModel.changed(boss, false);

	employeeAdditionPostProcessing(newEmployee);
}
/**
 * Add numerous new employees to the currently selected employee.
 * Test that these get loaded 'lazily'.
 */
public void addNumerousEmployees() {
	Employee currentBoss= getSelectedEmployee();
	Employee newEmployee= null;
	
	Vector children= new Vector();
	for (int jj= 0; jj < 100; jj ++) {
		newEmployee= new Employee("Name: " + jj, "Title: " + jj);
		newEmployee.setBoss(currentBoss);
		children.addElement(newEmployee);
	}
	fTreeModel.addChildren(currentBoss, children);
	// Refresh the boss of the added employee, since his childCount has increased.
	fTreeModel.changed(currentBoss, false);
	
	if (newEmployee != null) employeeAdditionPostProcessing(newEmployee);
}
/**
 * Add attributes that will be preloaded for each node of the tree, even if
 * it is not visible in any of the columns. This could be used for dynamic
 * popup menus.
 *
 * In this example, the attribute to track is the 'PreLoadedAttribute', and the menu will be displayed
 * (dynamically) when the value of the 'PreLoadedAttribute' is 'Preloaded for: Dave Thomas'. Note 
 * that the 'PreLoadedAttribute' is uploaded using the addPreloadAttribute method on the TreeModel.
 *
 * @see ManagementStructureAdapter#getValueAt(String, Object)
 * @see #createDynamicPopupMenu(ULCComponent)
 */
public void addPreloadAttributes() {
	fTreeModel.addPreloadAttribute("PreloadedAttribute");
}
/**
 * Columns can be added to a ULCTree, in the same manner as done for
 * the ULCTable. Customized renderers can also be provided for rendering
 * specific columns.
 * The First Column of a tree is 'special' in that it displays the icon
 * and the label which act as identifiers of a node. Hence this firstColumn
 * is created by the tree internally. You can only change it's label using
 * the API provided (below).
 */
public void addTreeColumns() {
	//fEmployeesTree.setFirstColumnLabel("First-Column"); //By default: blank label
	//addColumn api not valid anymore for the ulctree.
	/*fEmployeesTree.addColumn("numOfReports", "Num", 18);
	fEmployeesTree.addColumn("hasReports", "Bool", 20, true, new ULCCheckbox());	
	fEmployeesTree.addColumn("BossName", "Boss", 38);*/
	fEmployeesTree.setAutoResize(IDefaults.TABLE_AUTO_RESIZE_ALL_COLUMNS);
}
/**
 * Add the listeners to listen for various events on the tree
 */
public void addTreeListeners() {
	fEmployeesTree.addTreeSelectionChangedListener(new TreeNodeAction(this));
	fEmployeesTree.addTreeDefaultActionListener(new TreeNodeAction(this));
	fEmployeesTree.addTreeExpandActionListener(new TreeNodeAction(this));
	fEmployeesTree.addTreeCollapseActionListener(new TreeNodeAction(this));
}
/**
 * Build the application widgets into the receiver's shell.
 */
public void build() {
	fShell.setSize(SHELL_WIDTH, SHELL_HEIGHT);
	fShell.addWindowClosingListener(new WindowAction(this));
	ULCBox box= new ULCBox(2,2);
	
	box.add("ee", buildTreePart());
	box.add("ee", buildEmployeeDetailsPart());
	box.span(2, "eb", buildStatusPart());
	
	fShell.add(box);
}
/**
 * Build and answer the part that tests the handling of columns
 * for the tree sample
 * R3.0 changes : The addColumn, removeColumn API has been deprecated for ULCTree.
 * (ULCTreeTable should be used for trees with columns). Hence only the last part
 * of this method (testing setTreeColumnAttributeName()) is still valid.
 */
public ULCComponent buildColumnsHandlingPart() {
	String layout = "et";
	ULCBorder attributesBorder = new ULCBorder("Attribute (columns)");
	ULCBox attributesBox = new ULCBox(3, 2);

	/*
	//Label and Field to enter the Identifier of the column to be added/removed
	attributesBox.add(layout, new ULCLabel("Identifier"));
	final ULCField colIdField = createColumnIdentifierField();
	attributesBox.add(layout, colIdField);

	//Label and Field to enter the Label of the column to be added
	attributesBox.add(layout, new ULCLabel("Label"));
	final ULCField colLabelField = createColumnLabelField();
	attributesBox.add(layout, colLabelField);

	//Label and Field to enter the Width of the column to be added
	attributesBox.add(layout, new ULCLabel("Width"));
	final ULCField colWidthField = createColumnWidthField();
	attributesBox.add(layout, colWidthField);

	//A button to add a new ULCColumn to the table. This ULCColumn is
	//created based on the user inputs provided above.
	//Note: This implementation of the sample provides a convenience
	//hook to be able to add columns easily for testing: the treeModel
	//answers the identifier of a column if it is unable to find the
	//right object.
	//@see ManagementStructureAdapter.getValueAt(columnId, userObject).

	ULCButton addButton = new ULCButton("Add Column");
	addButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			String colId = (String) colIdField.getValue();
			String colLabel = (String) colLabelField.getValue();
			String widthStr = (String) colWidthField.getValue();
			try {
				Integer widthInt = new Integer(widthStr);
				//Add the column based on the given parameters.
				fEmployeesTree.addColumn(colId, colLabel, widthInt.intValue());
			} catch (NumberFormatException e1) {
				//Use the default column width
				fEmployeesTree.addColumn(colId, colLabel);
			}
		}
	});
	attributesBox.add("eb", addButton);

	//A button to add dummy columns to the table. This shows the usage
	//of the addColumns() API of the tree.

	ULCButton addDummyColsButton = new ULCButton("AddDummyColumns");
	addDummyColsButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			//Adding the dummy columns to the tree
			fEmployeesTree.addColumns(createDummyColumns());
		}
	});
	attributesBox.add("eb", addDummyColsButton);

	//A button to remove an existing ULCColumn from the table.
	//Note that removing a column from a tree has to be done only
	//via the removeColumn(aULCColumn) API. In order to facilitate
	//this, a getColumns() method is provided, which can be used in
	//order to determine the precise column(s) to be removed.

	ULCButton removeButton = new ULCButton("Remove Column");
	removeButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			String colId = (String) colIdField.getValue();
			//Get all the columns of the tree (ULCColumn objects)
			Vector columns = fEmployeesTree.getColumns();
			//Application logic to select the columns to be deleted.
			//In our example, we delete all columns identified by the
			//identifier in our colId field.
			for (int i = 0; i < columns.size(); i++) {
				ULCColumn column = (ULCColumn) columns.elementAt(i);
				if (column.getAttributeName().equals(colId))
					
					//Remove this ULCColumn from the tree
					fEmployeesTree.removeColumn(column);
			}
		}
	});
	attributesBox.add("eb", removeButton);

	//A button to remove all columns from the table. This shows the usage
	//of the removeColumns() API of the tree.
	//Note that removing columns from a tree has to be done only
	//via the removeColumns(aULCColumnVector) API. In order to facilitate
	//this, a getColumns() method is provided, which can be used in
	//order to determine the precise column(s) to be removed.	
	//Note also that the FirstColumn (which displays the icon & label of 
	//each node) is never deleted.

	ULCButton removeAllColButton = new ULCButton("removeAllColumns");
	removeAllColButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			//Get all the columns of the tree (ULCColumn objects)			
			Vector columns = fEmployeesTree.getColumns();
			//Remove these ULCColumns from the tree			
			fEmployeesTree.removeColumns(columns);
		}
	});
	attributesBox.add("eb", removeAllColButton);
	*/
	//
	final ULCField newColumnIdField = createColumnLabelField();
	attributesBox.add("eb", newColumnIdField);

	//Button to set the new label for the selected employee
	ULCButton columnIdChanger = new ULCButton("Set Tree ColumnId");
	columnIdChanger.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			String newLabel = (String) newColumnIdField.getValue();
			fEmployeesTree.setTreeColumnAttributeName((String) newColumnIdField.getValue());
		}
	});
	attributesBox.add("eb", columnIdChanger);

	//
	attributesBorder.add(attributesBox);
	return attributesBorder;
}
/**
 * Build and answer the part that tests the handling of details
 * for the employees in the management structure
 */
public ULCComponent buildEmployeeDetailsPart() {
	ULCNotebook notebook = new ULCNotebook();

	// Page 0 : An empty page to ensure lazy-loading of everything else :
	ULCBorder emptyBorder = new ULCBorder("Empty Page");
	ULCBox emptyBox = new ULCVBox(1);
	emptyBox.add("ee", new ULCLabel("Nothing here."));
	emptyBorder.add(emptyBox);
	notebook.addTab("Empty", emptyBorder);

	// Page 1 : Track the employee selection from the tree.
	ULCBox selectedEmployeeDetailsBox = new ULCVBox(2);
	selectedEmployeeDetailsBox.add("ee", buildSelectedEmployeeDetailsPart());
	selectedEmployeeDetailsBox.add("ee", buildRowModelBasedDetailsPart());
	notebook.addTab("Selected Employee Details", selectedEmployeeDetailsBox);

	// Page 2 : Set the possibility for various opreations on the tree...
	notebook.addTab("General Operations", buildGeneralOperationsPart());

	// Page 3 : Now for the new employee...	
	notebook.addTab("New Employee", buildNewEmployeePart());

	// Page 4 : Now the actions on the selected Employee
	notebook.addTab("Selected Employee Operations", buildSelectedEmployeeOperationsPart());

	// Page 5 : Now for the column adding/removing...
	notebook.addTab("Column Handling", buildColumnsHandlingPart());

	//Page 6 : Doing some special, not-so-common stuff
	notebook.addTab("Special Operations", buildSpecialOperationsPart());
	return notebook;
}
/**
 * Build the part for showing some of the 'general' operations
 * that can be done for a tree.
 */
public ULCComponent buildGeneralOperationsPart() {
	ULCBorder generalOptionsBorder = new ULCBorder("General operations");
	ULCBox box1 = new ULCBox(4, 2);

	//Radio-button to set the selection mode to single or multi-select
	box1.add("lt", createMultiSelectButton());
	box1.add("lt", createRootVisibleCheckBox());

	//Field to enter the new rowHeight to be set for the tree
	final ULCField rowHeightField = createSetRowHeightField();
	box1.add("lt", rowHeightField);

	//Button to set the new label for the selected employee
	ULCButton rowHeightChanger = new ULCButton("Set Row Height");
	rowHeightChanger.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			int newRowHeight = new Integer((String) rowHeightField.getValue()).intValue();
			fEmployeesTree.setRowHeight(newRowHeight);
		}
	});
	box1.add("lt", rowHeightChanger);

	// The field which will show the currently selected nodes
	box1.add(new ULCLabel("Selected Nodes: "));
	box1.add("et", createSelectedNodesField());
		
	// The button to set a new treeModel for this tree
	box1.add("et", createTreeModelChangeButton());

	// The button to set the treeModel for this tree to the original one
	box1.add("et", createTreeModelRevertButton());

	//Now the numerous selection options
	ULCBorder generalSelectionBorder = new ULCBorder("Selection operations");
	ULCBox selectionBox = new ULCVBox(6);
	generalSelectionBorder.add(selectionBox);

	// The button which will set the selection to an already uploaded
	// node.
	selectionBox.add("et", createAlreadyUploadedNodeSelectionButton());

	// The button which will set the selection to an existing but not-yet-uploaded
	// node. 
	selectionBox.add("et", createExistingNotYetUploadedNodeSelectionButton());

	// The button which will set the selection to a not-yet-uploaded
	// node.
	selectionBox.add("et", createNotYetUploadedNodeSelectionButton());

	// The button which will set the selection to a not-yet-uploaded
	// node.
	selectionBox.add("et", createNotYetUploadedNodeSelectionButton2());

	// The button which will set the selection to another not-yet-uploaded
	// node.
	selectionBox.add("et", createNotYetUploadedNodeSelectionButton3());

	// The button which will de-select all nodes in the tree.
	selectionBox.add("et", createDeselectionButton());

	ULCBox generalOptionsBox= new ULCVBox(2);
	generalOptionsBox.add("ee", box1);
	generalOptionsBox.add("ee", selectionBox);
	//
	generalOptionsBorder.add(generalOptionsBox);
	return generalOptionsBorder;
}
/**
 * Build and answer the part that tests the handling of new
 * employee addition in the tree sample
 */
public ULCComponent buildNewEmployeePart() {
	String layout = "et";
	ULCBorder newEmployeeDetailsBorder = new ULCBorder("New Employee Details");
	ULCBox newEmployeeDetailsBox1 = new ULCBox(3, 2);

	//Label & Field to enter the Name of the new employee
	newEmployeeDetailsBox1.add(layout, new ULCLabel("Name"));
	ULCField newNameField = new ULCField(fNewEmployeeAdapter, "Name", 10);
	newEmployeeDetailsBox1.add(layout, newNameField);

	//Label & Field to enter the Title of the new employee
	newEmployeeDetailsBox1.add(layout, new ULCLabel("Title"));
	ULCField newTitleField = new ULCField(fNewEmployeeAdapter, "Title", 10);
	newEmployeeDetailsBox1.add(layout, newTitleField);

	//Label & Field to enter the index of the new employee. That is, the
	//index in this employee's boss' list of direct-reports. By default,
	//the new employee will be added as the last child of his parent.
	newEmployeeDetailsBox1.add(layout, new ULCLabel("Index"));
	final ULCField indexField = new ULCField(10);
	indexField.addValueChangedListener(new IValueChangedListener() {
		public void valueChanged(ULCValueChangedEvent e) {
			indexField.setValue(e.getValue());
		}
	});
	newEmployeeDetailsBox1.add(layout, indexField);
	//
	//Now a new box to hold the buttons in one column.
	//
	ULCBox newEmployeeDetailsBox2 = new ULCVBox(5);
	//Button to add a new employee at the specified index,
	//as a child of the currently selected node (the boss
	//for this employee).
	ULCButton newButton = new ULCButton("New Employee");
	newButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			String indexStr = (String) indexField.getValue();
			Integer indexInt = null;
			try {
				indexInt = new Integer(indexStr);
			}
			catch (NumberFormatException e1) {
				indexInt = new Integer(-1); // default: add at the end.
			}
			//Now add a new employee at this index
			addNewEmployee(indexInt.intValue());
		}
	});
	newButton.setEnabled(false);
	newButton.setEnabler(fEmployeesTree);
	newEmployeeDetailsBox2.add(newButton);

	//Button to add a new employee immediately after the
	//selected one, as a new child of the same parent.
	ULCButton siblingButton = new ULCButton("New Sibling After Selected");
	siblingButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			Employee selectedObj = getSelectedEmployee();
			addNewEmployeeAfter(selectedObj);
		}
	});
	siblingButton.setEnabled(false);
	siblingButton.setEnabler(fEmployeesTree);
	newEmployeeDetailsBox2.add(siblingButton);

	//Button to add a new employee immediately before the
	//selected one, as a new child of the same parent.
	ULCButton siblingButton2 = new ULCButton("New Sibling Before Selected");
	siblingButton2.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			Employee selectedObj = getSelectedEmployee();
			addNewEmployeeBefore(selectedObj);
		}
	});
	siblingButton2.setEnabled(false);
	siblingButton2.setEnabler(fEmployeesTree);
	newEmployeeDetailsBox2.add(siblingButton2);

	//Button to add 100 new employees at the specified index,
	//as childern of the currently selected node (the boss
	//for this employee).
	ULCButton manyNewButton = new ULCButton("Add Many Employees");
	manyNewButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			addNumerousEmployees();
		}
	});
	manyNewButton.setEnabled(false);
	manyNewButton.setEnabler(fEmployeesTree);
	newEmployeeDetailsBox2.add(manyNewButton);

	//Button to add 100 new employees at the specified index,
	//as childern of the currently selected node (the boss
	//for this employee).
	ULCButton insertNewButton = new ULCButton("Insert New Employees");
	insertNewButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			addEmployeesAtPreSpecifiedIndices();
		}
	});
	insertNewButton.setEnabled(false);
	insertNewButton.setEnabler(fEmployeesTree);
	newEmployeeDetailsBox2.add(insertNewButton);
	//
	ULCBox box = new ULCVBox(2);
	box.add(newEmployeeDetailsBox1);
	box.add(newEmployeeDetailsBox2);
	newEmployeeDetailsBorder.add(box);
	return newEmployeeDetailsBorder;
}
/**
 * Build and answer the part that tests the selection of
 * employees in the tree sample, using the treeRowModel.
 */
public ULCComponent buildRowModelBasedDetailsPart() {
	String layout= "et";
	
	ULCBorder selectedEmployeeDetailsBorder= new ULCBorder("Selected Employee Details (Based on a Row Model)");
	ULCBox selectedEmployeeDetailsBox= new ULCBox(5,2);

	//Label and Field to display the Number of direct reports of the currently 
	//selected employee.
	//Note that the name (attributeName) for this field should be exactly the same as that used for
	//the attributes of the node. @see addTreeColumns.
	selectedEmployeeDetailsBox.add(layout, new ULCLabel("Num. Direct Reports"));
	ULCField numReportsField= new ULCField(fSelectedEmployeeRowModel, "numOfReports", 10);
	selectedEmployeeDetailsBox.add(layout, numReportsField);

	//Label and Field to display whether the selected employee has direct reports.
	selectedEmployeeDetailsBox.add(layout, new ULCLabel("Has Direct Reports"));
	ULCField hasReportsField= new ULCField(fSelectedEmployeeRowModel, "hasReports", 10);
	selectedEmployeeDetailsBox.add(layout, hasReportsField);
		
	//Label and Field to display the Boss of the currently selected employee
	selectedEmployeeDetailsBox.add(layout, new ULCLabel("Boss"));
	ULCField bossField= new ULCField(fSelectedEmployeeRowModel, "BossName", 10);
	selectedEmployeeDetailsBox.add(layout, bossField);
	
	selectedEmployeeDetailsBorder.add(selectedEmployeeDetailsBox);
		
	return selectedEmployeeDetailsBorder;
}
/**
 * Build and answer the part that tests the selection of
 * employees in the tree sample
 */
public ULCComponent buildSelectedEmployeeDetailsPart() {
	String layout= "et";
	
	ULCBorder selectedEmployeeDetailsBorder= new ULCBorder("Selected Employee Details");
	ULCBox selectedEmployeeDetailsBox= new ULCBox(5,2);

	//Label and Field to display the Name of the currently selected employee
	selectedEmployeeDetailsBox.add(layout, new ULCLabel("Name"));
	ULCField nameField= new ULCField(fSelectedEmployeeAdapter, "Name", 10);
	selectedEmployeeDetailsBox.add(layout, nameField);

	//Label and Field to display the Title of the currently selected employee	
	selectedEmployeeDetailsBox.add(layout, new ULCLabel("Title"));
	ULCField titleField= new ULCField(fSelectedEmployeeAdapter, "Title", 10);
	selectedEmployeeDetailsBox.add(layout, titleField);
	
	//Label and Field to display the Number of direct reports of the currently 
	//selected employee
	selectedEmployeeDetailsBox.add(layout, new ULCLabel("Num. Direct Reports"));
	ULCField numReportsField= new ULCField(fSelectedEmployeeAdapter, "NumDirectReports", 10);
	selectedEmployeeDetailsBox.add(layout, numReportsField);
	
	//Label and Field to display the Boss of the currently selected employee
	selectedEmployeeDetailsBox.add(layout, new ULCLabel("Boss"));
	ULCField bossField= new ULCField(fSelectedEmployeeAdapter, "Boss", 10);
	selectedEmployeeDetailsBox.add(layout, bossField);
	
	selectedEmployeeDetailsBorder.add(selectedEmployeeDetailsBox);
		
	return selectedEmployeeDetailsBorder;
}
/**
 * Build and answer the part that tests the handling of various
 * operations performed on the employee(s) in the management
 * structure
 */
public ULCComponent buildSelectedEmployeeOperationsPart() {
	String layout= "et";
	
	ULCBorder selectedEmployeeBorder= new ULCBorder("Selected Employee Operations");
	ULCBox selectedEmployeeBox= new ULCBox(4,2);

	//Button to delete the selected employee
	selectedEmployeeBox.add(layout, createDeleteEmployeeButton());

	//Button to set the selected emploee as the root of the tree
	selectedEmployeeBox.add(layout, createRootChangerButton());

	//Field to enter the new label to be set for selected employee
	final ULCField labelField= createSetLabelField();
	selectedEmployeeBox.add(layout, labelField);

	//Button to set the new label for the selected employee
	ULCButton titleChanger= new ULCButton("Set Label");
	titleChanger.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			String newLabel= (String) labelField.getValue();
			setLabel(newLabel, getSelectedEmployee());	
		}
	});
	titleChanger.setEnabled(false);
	titleChanger.setEnabler(fEmployeesTree);	
	selectedEmployeeBox.add(layout, titleChanger);

	// The button to set the tree selection to the selected node's parent.
	selectedEmployeeBox.add(layout, createSelectParentButton());

	// The button to ensure that the parent of the current selection, 
	// is visbile.
	selectedEmployeeBox.add(layout, createRevealParentButton());	

	// The button to reload the selected node (& it's children).
	// Done after adding a new child to the selected node.
	selectedEmployeeBox.add(layout, createReloadEmployeeButton());
	
	// Yet another button to reload the selected node (& it's children).
	// This time, all existing children of this node are removed.
	selectedEmployeeBox.add(layout, createAnotherReloadEmployeeButton());

	selectedEmployeeBorder.add(selectedEmployeeBox);
	return selectedEmployeeBorder;
}
/**
 * Build the part for showing some of the 'special' operations
 * that can be done for a tree.
 */
public ULCComponent buildSpecialOperationsPart() {
	ULCBorder specialOperationsBorder = new ULCBorder("Special operations");
	ULCBox box = new ULCBox(1, 1);

	//Radio-button to perform a 'reload" when a node is expanded for the first time.
	//This is a 'special' operation because it can lead to timing problems : on expansion
	//the 'sendSpecifiedChildren' request comes to ULCTreeMode to paint the node's
	//children. And if we make this 'reload' call at the same time, it gets very tricky !
	box.add("lt", createReloadOnExpansionCheckBox());
	
	specialOperationsBorder.add(box);
	return specialOperationsBorder;
}
/**
 * Build and answer the part that displays the status of the
 * application. This status will b updated when specific events
 * occur in the application (for example, selection of a node,
 * expanding or collapsing of a node etc.).
 */
public ULCComponent buildStatusPart() {
	String layout= "lt";
	ULCBorder statusBorder= new ULCBorder("Application Status");
	ULCBox statusBox= new ULCHBox();
	fStatusLabel= new ULCLabel("Current status shown here");
	statusBox.add("ec", fStatusLabel);
	statusBorder.add(statusBox);
	return statusBorder;
}
/**
 * Build and answer the tree part of the sample
 */
public ULCComponent buildTreePart() {
	ULCHBox box = new ULCHBox();
	fEmployeesTree = new ULCTree(fTreeModel);
	addTreeColumns();
	addTreeListeners();
	createTreeRowModel();
	createDynamicPopupMenu(fEmployeesTree);
	addPreloadAttributes();
	box.add("ee", fEmployeesTree);
	return box;
}
/**
 * Create and answer a Button which will be used for selecting an
 * employee which has definitely been uploaded already. 
 */
protected ULCButton createAlreadyUploadedNodeSelectionButton() {
	ULCButton button = new ULCButton("Select an Existing Uploaded Node");
	button.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			//fEmployeesTree.setSelectedNode(((Employee) fTreeModel.getRoot()).getChildren().firstElement());
			//Use setSelectedItem instead of setSelectedNode :
			fEmployeesTree.setSelectedItem(((Employee) fTreeModel.getRoot()).getChildren().firstElement());
		}
	});
	return button;
}
/**
 * Create and answer a Button which will be used for reloading the
 * currently selected employee from the ULC side.
 */
protected ULCButton createAnotherReloadEmployeeButton() {
	ULCButton reloadButton = new ULCButton("Reload2");
	reloadButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			final Employee employee = getSelectedEmployee();
			employee.setLabel("Reloaded");
			employee.deleteAllImmediateReports();
			new Employee("NewForReload", "NewForReloadTitle").setBoss(employee);
			reload(employee);
		}
	});
	reloadButton.setEnabled(false);
	reloadButton.setEnabler(fEmployeesTree);
	return reloadButton;
}
/**
 * Create and answer a field where the identifier for a new
 * ULCColumn is entered by the user
 */
protected ULCField createColumnIdentifierField() {
	final ULCField colIdField= new ULCField(10);
	colIdField.addValueChangedListener(new IValueChangedListener() {
		public void valueChanged(ULCValueChangedEvent e) {
			colIdField.setValue(e.getValue());
		}
	});
	return colIdField;
}
/**
 * Create and answer a field where the Label for a new
 * ULCColumn is entered by the user
 */
protected ULCField createColumnLabelField() {
	final ULCField colLabelField= new ULCField(10);
	colLabelField.addValueChangedListener(new IValueChangedListener() {
		public void valueChanged(ULCValueChangedEvent e) {
			colLabelField.setValue(e.getValue());
		}
	});
	return colLabelField;
}
/**
 * Create and answer a field where the Width for a new
 * ULCColumn is entered by the user
 */
protected ULCField createColumnWidthField() {
	final ULCField colWidthField= new ULCField(10);
	colWidthField.addValueChangedListener(new IValueChangedListener() {
		public void valueChanged(ULCValueChangedEvent e) {
			colWidthField.setValue(e.getValue());
		}
	});
	return colWidthField;
}
/**
 * Create and answer a Button which will be used for deleting the
 * currently selected employee
 */
protected ULCButton createDeleteEmployeeButton() {
	ULCButton deleteButton= new ULCButton("Delete");
	deleteButton.addActionListener(new DeleteEmployeeAction(this));	
	deleteButton.setEnabled(false);
	deleteButton.setEnabler(fEmployeesTree);
	return deleteButton;
}
protected ULCMenuItem createDeleteEmployeeMenuItem() {
	ULCMenuItem item= new ULCMenuItem("Delete Employee", new DeleteEmployeeAction(this));
	item.setEnabler(fEmployeesTree);
	return item;
}
/**
 * Create and answer a Button which will be used for deselecting
 * all nodes in the management tree.
 */
protected ULCButton createDeselectionButton() {
	ULCButton button = new ULCButton("Deselect all Nodes");
	button.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			//Use setSelectedItems instead of setSelectedNodes :
			//fEmployeesTree.setSelectedNodes(null);
			fEmployeesTree.setSelectedItems(null);
		}
	});
	return button;
}
/**
 * Create dummy ULCColumn objects that are to be added to the
 * tree.
 */
protected Vector createDummyColumns() {
	Vector v = new Vector();
	v.addElement(new ULCColumn("DummyId1", "DummLabel1"));
	v.addElement(new ULCColumn("DummyId2", "DummLabel2"));
	return v;
}
/** 
 * The popup menu can include 'static' as well as 'dynamic' parts. The dynamic parts work
 * based on the TreeRowModel (fSelectedEmployeeRowModel) defined earlier. You need to
 * define the attributeName from the TreeRowModel, which needs to be tracked by the dynamic
 * menu. And you need to provide the attributeValue for this attributeName, so that when the
 * TreeRowModel has that value for that attribute, the dynamic menu will be displayed.
 *
 * In this example, the attribute to track is the 'PreLoadedAttribute', and the menu will be displayed
 * (dynamically) when the value of the 'PreLoadedAttribute' is 'Preloaded for: Dave Thomas'. 
 * Similarly for the 'Finance Incharge' node.
 *
 * Note  that the 'PreLoadedAttribute' is uploaded using the addPreloadAttribute method on the TreeModel.
 *
 * @see #addPreloadAttributes()
 * @see ManagementStructureAdapter#getValueAt(String, Object)
 */
public void createDynamicPopupMenu(ULCComponent component) {
	ULCMenu dynamicMenu = new ULCMenu("Dynamic Popup Menu");
	
	//Add a Dynamic menu based on the formModel.
	ULCMenu m = new ULCMenu("The OTI Boss");
	m.add(new ULCMenuItem("Dynamic menu for the CEO"));
	m.setFormModel(fSelectedEmployeeRowModel);
	m.setFormAttributeName("PreloadedAttribute");
	m.setFormAttributeValue("Preloaded for: Dave Thomas");
	dynamicMenu.add(m);

	//Add another Dynamic menu based on the formModel.
	m = new ULCMenu("The OTI Finance Incharge");
	m.add(new ULCMenuItem("Dynamic menu for the Finance Incharge"));
	m.setFormModel(fSelectedEmployeeRowModel);
	m.setFormAttributeName("PreloadedAttribute");
	m.setFormAttributeValue("Preloaded for: Mr. Finance");
	dynamicMenu.add(m);
	
	//Add Static menus (which will always be shown).
	dynamicMenu.add(createDeleteEmployeeMenuItem());

	component.setPopupMenu(dynamicMenu);
}
/**
 * Create and answer a Button which will be used for selecting an
 * *existing* employee which has *not yet* been uploaded. 
 */
protected ULCButton createExistingNotYetUploadedNodeSelectionButton() {
	ULCButton button = new ULCButton("Select an Existing Not-Yet-Uploaded Node");
	button.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			Employee biggestBoss = (Employee) fTreeModel.getRoot();
			// Select the first child of the 'Technical Boss'.
			Employee toSelect= biggestBoss.getChildAtIndex(2).getChildAtIndex(0);
			//Use setSelectedItem instead of setSelectedNode :
			//fEmployeesTree.setSelectedNode(toSelect);
			fEmployeesTree.setSelectedItem(toSelect);
		}
	});
	return button;
}
/**
 * Create and answer a radioButton which will be used for determining
 * the selection policy for the tree - as to whether it allows single
 * selection or multiple selection.
 */
protected ULCRadioButton createMultiSelectButton() {
	ULCRadioButton radioButton= new ULCRadioButton("Multi-Selection ?", false);
	radioButton.addValueChangedListener(new IValueChangedListener() {
		public void valueChanged(ULCValueChangedEvent e) {
			valueChanged(e.getValue());
		}
		public void valueChanged(Object newValue) {
			if (((Boolean) newValue).booleanValue())
				//Multiple selection mode
				fEmployeesTree.setSelectionMode(IDefaults.TREE_DISCONTIGUOUS_SELECTION);
			else
				//Single selection mode
				fEmployeesTree.setSelectionMode(IDefaults.TREE_SINGLE_SELECTION);
		}
	});	
	return radioButton;
}
/**
 * Create and answer a Button which will be used for selecting an
 * employee which has *not yet* been uploaded. 
 */
protected ULCButton createNotYetUploadedNodeSelectionButton() {
	ULCButton button = new ULCButton("Select a New Not-Yet-Uploaded Node (Level1)");
	button.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			Employee biggestBoss = (Employee) fTreeModel.getRoot();
			Employee newEmployee = new Employee("NotUploadedEmployee ", "NotUploadedTitle ");
			newEmployee.setBoss(biggestBoss);
			//Use setSelectedItem instead of setSelectedNode :
			//fEmployeesTree.setSelectedNode(newEmployee);
			fEmployeesTree.setSelectedItem(newEmployee);
			//Now refresh the parent node, since we want to display the (new) child count :
			fTreeModel.changed(biggestBoss, false); //no reload of children required.
		}
	});
	return button;
}
/**
 * Create and answer a Button which will be used for selecting an
 * employee which has *not yet* been uploaded. 
 */
protected ULCButton createNotYetUploadedNodeSelectionButton2() {
	ULCButton button = new ULCButton("Select a New Not-Yet-Uploaded Node (Level2)");
	button.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			Employee biggestBoss = (Employee) fTreeModel.getRoot();
			Employee newEmployee = new Employee("NotUploadedEmployee1 ", "NotUploadedTitle1 ");
			newEmployee.setBoss(biggestBoss); //New parent

			Employee newEmployee2 = new Employee("NotUploadedEmployee2 ", "NotUploadedTitle2 ");
			newEmployee2.setBoss(newEmployee); //Child1

			Employee newEmployee3 = new Employee("NotUploadedEmployee3 ", "NotUploadedTitle3 ");
			newEmployee3.setBoss(newEmployee);
			//Use setSelectedItem instead of setSelectedNode :
			//fEmployeesTree.setSelectedNode(newEmployee3); //Child2= selection.
			fEmployeesTree.setSelectedItem(newEmployee3); //Child2= selection.

			//Now refresh the parent node, since we want to display the (new) child count :
			fTreeModel.changed(biggestBoss, false); //no reload of children required.
		}
	});
	return button;
}
/**
 * Create and answer a Button which will be used for selecting an
 * employee which has *not yet* been uploaded. 
 */
protected ULCButton createNotYetUploadedNodeSelectionButton3() {
	ULCButton button = new ULCButton("Select a New Not-Yet-Uploaded Node (Level3)");
	button.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			Employee biggestBoss = (Employee) fTreeModel.getRoot();
			Employee newEmployee = new Employee("NotUploadedEmployee1 ", "NotUploadedTitle1 ");
			newEmployee.setBoss(biggestBoss); //New parent

			Employee newEmployee2 = new Employee("NotUploadedEmployee2 ", "NotUploadedTitle2 ");
			newEmployee2.setBoss(newEmployee); //Child1

			Employee newEmployee3 = new Employee("NotUploadedEmployee3 ", "NotUploadedTitle3 ");
			newEmployee3.setBoss(newEmployee); //Child2 = next level parent

			Employee newEmployee4 = new Employee("NotUploadedEmployee4 ", "NotUploadedTitle4 ");
			newEmployee4.setBoss(newEmployee3); //Child of Child2.

			//Use setSelectedItem instead of setSelectedNode :
			//fEmployeesTree.setSelectedNode(newEmployee4);
			fEmployeesTree.setSelectedItem(newEmployee4);

			//Now refresh the parent node, since we want to display the (new) child count :
			fTreeModel.changed(biggestBoss, false); //no reload of children required.
		}
	});
	return button;
}
/**
 * Create and answer a Button which will be used for reloading the
 * currently selected employee from the ULC side.
 */
protected ULCButton createReloadEmployeeButton() {
	ULCButton reloadButton = new ULCButton("Reload");
	reloadButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			final Employee employee= getSelectedEmployee();
			employee.setLabel("Reloaded");
			new Employee("NewForReload", "NewForReloadTitle").setBoss(employee);
			reload(employee);
		}
	});
	reloadButton.setEnabled(false);
	reloadButton.setEnabler(fEmployeesTree);
	return reloadButton;
}
/**
 * Create and answer a checkBox which will be used for determining
 * whether a node (& it's children) should be reloaded when this node
 * is expanded.
 */
protected ULCComponent createReloadOnExpansionCheckBox() {
	final ULCCheckbox checkbox = new ULCCheckbox("Reload on Expansion ?", false);
	checkbox.addValueChangedListener(new IValueChangedListener() {
		public void valueChanged(ULCValueChangedEvent e) {
			fReloadOnExpansion = checkbox.isSelected();
		}
	});
	return checkbox;
}
/**
 * Create and answer a button which on being pressed, ensures that
 * the parent of the currently selected node is visible. Scrolling
 * is performed, if necessary
 */
protected ULCButton createRevealParentButton() {
	ULCButton revealButton= new ULCButton("Reveal Parent");
	revealButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			fEmployeesTree.ensureNodeIsVisible(getSelectedEmployee().getBoss());
		}
	});
	revealButton.setEnabler(fSelectedNodesField);
	return revealButton;
}
/**
 * Create and answer a Button which will be used for changing the
 * root of the tree. The new root of the tree will be set to the
 * currently selected node of the tree.
 */
protected ULCButton createRootChangerButton() {
	ULCButton rootChanger= new ULCButton("Set As Root");
	rootChanger.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			//Set the new root for the application
			fTreeModel.setNewCEO(getSelectedEmployee());
			//Signal that the whole tree has changed, which will
			//start by asking the new root for it's children
			fTreeModel.changed();
		}
	});
	rootChanger.setEnabled(false);
	rootChanger.setEnabler(fEmployeesTree);
	return rootChanger;
}
/**
 * Create and answer a checkBox which will be used for determining
 * the rootVisible policy for the tree
 */
protected ULCComponent createRootVisibleCheckBox() {
	final ULCCheckbox checkbox = new ULCCheckbox("Root Visible?", true);
	checkbox.addValueChangedListener(new IValueChangedListener() {
		public void valueChanged(ULCValueChangedEvent e) {
			fEmployeesTree.setRootVisible(checkbox.isSelected());
		}
	});
	return checkbox;
}
/**
 * Create and answer a field where the list of selected nodes
 * is displayed
 */
protected ULCField createSelectedNodesField() {
	fSelectedNodesField= new ULCField(10);
	fSelectedNodesField.addValueChangedListener(new IValueChangedListener() {
		public void valueChanged(ULCValueChangedEvent e) {
			fSelectedNodesField.setValue(e.getValue());;
		}
	});
	return fSelectedNodesField;
}
/**
 * Create and answer a button which on being pressed, sets the
 * selection of the tree from the currently selected employee to
 * his boss
 */
protected ULCButton createSelectParentButton() {
	ULCButton changeSelButton= new ULCButton("Select parent");
	changeSelButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			//Use setSelectedItem instead of setSelectedNode :
			//fEmployeesTree.setSelectedNode(getSelectedEmployee().getBoss());
			fEmployeesTree.setSelectedItem(getSelectedEmployee().getBoss());
			return;
		}
	});
	changeSelButton.setEnabler(fEmployeesTree);
	return changeSelButton;
}
/**
 * Create and answer a field where a new Label for the
 * selected employee is entered by the user
 */
protected ULCField createSetLabelField() {
	final ULCField labelField= new ULCField("New Label", 10);
	labelField.addValueChangedListener(new IValueChangedListener() {
		public void valueChanged(ULCValueChangedEvent e) {
			labelField.setValue(e.getValue());
		}
	});
	return labelField;
}
/**
 * Create and answer a field where a new row height for the
 * tree is entered by the user
 */
protected ULCField createSetRowHeightField() {
	final ULCField rowHeightField= new ULCField(2);
	rowHeightField.addValueChangedListener(new IValueChangedListener() {
		public void valueChanged(ULCValueChangedEvent e) {
			rowHeightField.setValue(e.getValue());
		}
	});
	return rowHeightField;
}
/**
 * Create and answer the button which, when pressed,
 * results in a new treeModel being hooked on as the
 * model (dataSource) for the tree.
 */
protected ULCButton createTreeModelChangeButton() {
	ULCButton changeModelButton= new ULCButton("Change Tree Model");
	changeModelButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			//Set a new tree model
			fEmployeesTree.setTreeModel(getNewTreeModel());
			return;
		}
	});
	return changeModelButton;
}
/**
 * Create and answer the button which, when pressed,
 * results in a new treeModel being hooked on as the
 * model (dataSource) for the tree: This will be the
 * 'complete' model as available at the start of the
 * sample
 */
protected ULCButton createTreeModelRevertButton() {
	ULCButton revertModelButton= new ULCButton("Revert Tree Model");
	revertModelButton.addActionListener(new IActionListener() {
		public void actionPerformed(ULCActionEvent e) {
			//Set a new tree model
			fEmployeesTree.setTreeModel(getOriginalTreeModel());
			return;
		}
	});
	return revertModelButton;
}
/**
 * Associate the tree and it's model with a 'TreeRowModel', so that the TreeRowModel will
 * automatically track the tree selections. It is essentially a FormModel, so it can then be
 * used for updating fields associated with it, without traffic going to ULC.
 *
 * @see #buildRowModelBasedDetailsPart
 *
 */
protected void createTreeRowModel() {
	fSelectedEmployeeRowModel = new ULCTreeRowModel(fTreeModel, fEmployeesTree);
}
/**
 * Delete the selected employee.
 */
public void deleteSelectedEmployee() {
	Employee employeeToDelete = getSelectedEmployee();

	// First ensure that the deletion is done successfully
	// inside the application
	boolean hasBeenDeleted = true;
	if (employeeToDelete == fTreeModel.getRoot()) {
		fTreeModel.setNewCEO(null);
	}
	else {
		hasBeenDeleted = employeeToDelete.delete();
	}
	if (hasBeenDeleted) {
		//Inform the treeModel to delete this node
		fTreeModel.deleteChild(employeeToDelete);

		//Refresh the boss for the deleted employee,
		// since his childCount would have changed
		fTreeModel.changed(employeeToDelete.fBoss, false);

		//And some internal updates
		updateStatus("Deleted Employee: ", employeeToDelete);
		fSelectedEmployeeAdapter.setDummyEmployee();
	}
	else
		updateStatus("Unable to delete (probably has children): ", employeeToDelete);
}
/**
 * Display the list of selected nodes in the appropriate
 * field of the receiver
 */
public void displaySelectedNodes(Vector selectedNodes) {
	String display= "";
	for (int i= 0; i < selectedNodes.size(); i++) {
		display= display + ((Employee) selectedNodes.elementAt(i)).getName() + ",";
	}

	fSelectedNodesField.setValue(display);
}
/**
 * Build and display the widgets for the given application
 */
public void edit(ULCApplication app) {
	fShell= new ULCShell("Management Structure", true); // Create the shell
	app.add(fShell);				// This shell is part of the overall application
	build();						// Build all the widgets into this shell
	fShell.setVisible(true);				// Now show the result.
}
/**
 * Do the required refreshings in order to get the application in synch
 * with the addition of the new employee
 */
protected void employeeAdditionPostProcessing(Employee newlyAddedEmployee) {
	// Update the status to show this new action
	updateStatus("Inserted new item: ", newlyAddedEmployee);

	// Reset the adapter, but keep the values for convenience
	fNewEmployeeAdapter.setEmployee(new Employee(
			newlyAddedEmployee.getName(), newlyAddedEmployee.getTitle()));

	// The childCount is also displayed in the selectedEmployeeAdapter,
	// which therefore needs to be updated
	fSelectedEmployeeAdapter.notifyChange();
}
/**
 * Answer a new ULCTreeModel. This is used in order to test the
 * setTreeModel API, through which it is possible to dynamically
 * change the treeModel (the 'dataSource') for an existing tree.
 *
 * Any preloadAttributes present on the original treeModel have to
 * be added here as well.
 *
 */
public ULCTreeModel getNewTreeModel() {
	fTreeModel= new AnotherMgmtStructAdapter(this);
	addPreloadAttributes();
	return fTreeModel;
}
/**
 * Answer a new ULCTreeModel. This is used in order to test the
 * setTreeModel API, through which it is possible to dynamically
 * change the treeModel (the 'dataSource') for an existing tree.
 *
 * Any preloadAttributes present on the original treeModel have to
 * be added here as well.
 */
public ULCTreeModel getOriginalTreeModel() {
	fTreeModel= new ManagementStructureAdapter(this);
	addPreloadAttributes();
	return fTreeModel;
}
/**
 * Answer the currently selected employee
 */
public Employee getSelectedEmployee() {
	return fSelectedEmployeeAdapter.getEmployee();
}
/**
 * The given node of the tree has been collapsed
 */
public void performCollapseActionFor(Object selectedNode) {
	Employee selectedEmployee= (Employee) selectedNode;
	updateStatus("Collapse Action performed on: ", selectedEmployee);
}
/**
 * Default action (typically double-click) has been performed on one or more
 * nodes of the tree
 */
public void performDefaultActionFor(Object firstSelectedNode, Vector selectedNodes) {
	Employee selectedEmployee= (Employee) firstSelectedNode;
	updateStatus("Default action done on: " + selectedNodes.size() + 
		" nodes. Firstly: ", selectedEmployee);
}
/**
 * The given node of the tree has been expanded
 */
public void performExpandActionFor(Object selectedNode) {
	Employee selectedEmployee = (Employee) selectedNode;
	if (shouldReloadOnExpansion(selectedEmployee)) {
		postDeferredReloadOperationFor(selectedEmployee);
		selectedEmployee.setHasBeenExpandedOnce(true);
	}
	updateStatus("Expand Action performed on: ", selectedEmployee);
}
/**
 * If we need to do a 'reload' during an 'expand' event, this could lead to serious
 * timing conflicts and problems. The correct way is to 'post' a deferred request
 * which will get queued up correctly, and hence not interfere with the current
 * operations.
 */
public void postDeferredReloadOperationFor(Employee selectedEmployee) {
	final Employee employee = selectedEmployee;
	//Request r = new Request() {
	//	public void dispatch() {
			//First change the bo a bit
			employee.setLabel("Reloaded on Expansion");
			employee.deleteAllImmediateReports();
			new Employee("ReloadedOnExpansion", "NewForReloadOnExpansion").setBoss(employee);
			new Employee("Reloaded2OnExpansion", "NewForReload2OnExpansion").setBoss(employee);			
			//And now fire the reload operation.
			reload(employee);
//		}
//	};
//	fManagementStructure.postRequest(r);
}
/**
 * Quit this application - after getting a confirmation
 * from the user.
 */
public void quit() {
	fManagementStructure.beep();
	ULCAlert alert= new ULCAlert(fShell, "Please Confirm",
					"Do you really want to quit?", "Yes", "No");
	if (alert.ask().equals("Yes")) fManagementStructure.quit();
}
/**
 * Reload the given employee (and it's children).
 */
protected void reload(Employee employee) {
	fTreeModel.changed(employee, true);
}
/**
 * Set the given node as the selected employee
 */
public void setEmployeeSelection(Object selectedNode) {
	Employee selectedEmployee= (Employee) selectedNode;
	fSelectedEmployeeAdapter.setEmployee(selectedEmployee);
	updateStatus("Selected new item: ", selectedEmployee);
}
/**
 * Set the label of the given employee to the newLabel.
 * Note : Swing has a bug due to which the *root* node does
 * not refresh correctly after the setLabel.
 */
public void setLabel(String newLabel, Employee selectedEmployee) {
	//First set the label in the application
	String oldLabel= selectedEmployee.getLabel();
	selectedEmployee.setLabel(newLabel);

	//Inform the treeModel to update the UI accordingly	
	fTreeModel.setLabel(newLabel, selectedEmployee);

	//Update the rest of the application for this change
	fSelectedEmployeeAdapter.notifyChange();
	updateStatus("Label changed (" + oldLabel + " -> " + newLabel + ") for: ", selectedEmployee);
}
/**
 * Answer boolean whether a node should be reloaded
 * when it is expanded.
 */
public boolean shouldReloadOnExpansion() {
	return fReloadOnExpansion;
}
/**
 * Answer boolean whether the given employee should be reloaded
 * when it is expanded. This depends on the checkBox set in the
 * selectedEmployees page, as well as on whether the given employee
 * has already been reloaded (during expansion) once.
 */
public boolean shouldReloadOnExpansion(Employee selectedEmployee) {
	return (fReloadOnExpansion && !selectedEmployee.hasBeenExpandedOnce());
}
/**
 * Test the setIcon API : If the given node has more than
 * 5 children, we set a different icon for it !
 * Note : Swing has a bug due to which the *root* node does
 * not refresh correctly after the setIcon.
 */
public void testIconSetting(Employee node) {
	if (node.getChildCount() > 5) {
		node.setIconName("green-ball.gif");
		fTreeModel.setIcon(new ULCIcon(getClass().getResource("green-ball.gif")), node);
	}
}
/**
 * Update the receiver's status with the given string,
 * for the given employee
 */
public void updateStatus(String newStatus, Employee employee) {
	if (employee != null) 
		fStatusLabel.setLabel(newStatus + employee.getName());
	else
		fStatusLabel.setLabel(newStatus);
}
}
