package com.ibm.ulc.examples.carRental.model;

import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Hashtable;

/**
 * This class mixes two responsibilities:
 * - simulating a database accessor
 * - providing dummy data for the sample application.
 *
 * @author: Markus Marzetta (CS-ZRH)
 */
public class DataAccessor extends CarRentalObject {
	/**
	 * The data accessor singleton.
	 */
	 protected static DataAccessor fgInstance= null;

	 static {
		 fgInstance= new DataAccessor();
	 }
	 
	/**
	 * Hold the collection of all card companies in an instance variable.
	 */
	 protected CardCompany[] fCardCompanies = new CardCompany[0];
	/**
	 * Hold the collection of all car classes in an instance variable.
	 */
	 protected CarClass[] fCarClasses = new CarClass[0];
	/**
	 * Hold the collection of all countries in an instance variable.
	 */
	 protected Country[] fCountries = new Country[0];
	/**
	 * Hold the collection of all locations in an instance variable.
	 */
	 protected Location[] fLocations = new Location[0];
	/**
	 * Hold the collection of all rateSchemes in an instance variable.
	 */
	 protected RateScheme[] fRateSchemes = new RateScheme[0];

	/**
	 * The maximum for the example data parameter.
	 */
	 public static int MAX_EXAMPLE_SIZE= 5;

/**
 * DataAccessor constructor comment.
 */
public DataAccessor() {
	super();
}
/**
 * Answer a card company named name.
 *
 * Creation date: (07.01.00 12:17:45)
 * @return com.ibm.ulc.examples.carRental.model.CardCompany
 * @param name java.lang.String
 */
public CardCompany cardCompanyNamed(String name) {
	for (int i=0; i<fCardCompanies.length; i++)
		if (fCardCompanies[i].getName().equals(name))
			return fCardCompanies[i];
	return null;
}
/**
 * Answer a rate scheme named name.
 *
 * Creation date: (07.01.00 12:17:45)
 * @return com.ibm.ulc.examples.carRental.model.RateScheme
 * @param name java.lang.String
 */
public Country countryNamed(String name) {
	for (int i=0; i<fCountries.length; i++)
		if (fCountries[i].getName().equals(name))
			return fCountries[i];
	return null;
}
/**
 * Answer the country with the code given in the parameter (or null).
 * 
 * Creation date: (13.12.99 13:24:55)
 * @return com.ibm.ulc.examples.carRental.model.Country
 * @param code java.lang.String
 */
protected Country countryWithId(String id) {
	for (int i= 0; i<fCountries.length; i++) {
		if (fCountries[i].getId().equals(id)) {
			return fCountries[i];
		}
	}
	return null;
}
/**
 * Create a new CarClass from 2 Strings.
 * 
 * Creation date: (13.12.99 13:00:52)
 * @return com.ibm.ulc.examples.carRental.model.CarClass
 */
protected CarClass createCarClass(String code, String name) {
	return new CarClass(code, name);
}
/**
 * Create a new CardCompany from 2 Strings.
 * 
 * Creation date: (13.12.99 13:00:52)
 * @return com.ibm.ulc.examples.carRental.model.CardCompany
 */
protected CardCompany createCardCompany(String code, String name) {
	return new CardCompany(code, name);
}
/**
 * Create a new Country from 2 Strings.
 * 
 * Creation date: (13.12.99 13:00:52)
 * @return com.ibm.ulc.examples.carRental.model.Country
 */
protected Country createCountry(String code, String name) {
	return new Country(code, name);
}
/**
 * Create a new location from a bunch of Strings.
 * 
 * Creation date: (14.12.99 09:48:55)
 * @return com.ibm.ulc.examples.carRental.model.Location
 * @param data java.lang.String[]
 */
protected Location createLocation(String[] data) {
	if (data.length<10) {
		throw new Error("I'm broken: this method was called with an insufficient number of aprameters");
	}
	Location location= new Location(data[0],data[1]);
	location.setAddress(data[2]);
	location.setCity(data[3]);
	location.setState(data[4]);
	location.setCountry(countryWithId(data[5]));
	location.setFax(data[6]);
	location.setPhone(data[7]);
	location.setTelex(data[8]);
	location.setTimeOffset((int)Double.valueOf(data[9]).doubleValue()*3600*1000);
	
	location.setOpeningSchedule(createOpeningSchedule(
				RandomDataFactory.booleanTrue(0.05d),
				RandomDataFactory.intBelow(4),
				RandomDataFactory.intBelow(2),
				RandomDataFactory.intBelow(2),
				RandomDataFactory.intBelow(5),
				RandomDataFactory.booleanTrue(0.9d),
				RandomDataFactory.booleanTrue(0.01d)));

	String[] mapKeys= new String[3];
	mapKeys[0]= location.getCity();
	mapKeys[1]= location.getState();
	mapKeys[2]= location.getCountry().getName();
	location.setMapKey(mapKeys);
	return location;
}
/**
 * Create an OpeningSchedule from a couple of ints andd booleans.
 * 
 * Creation date: (14.12.99 10:12:50)
 * @return com.ibm.ulc.examples.carRental.model.OpeningSchedule
 * @param sevenTwentyFour boolean
 * @param morning0 int
 * @param morning1 int
 * @param pause int
 * @param afternoon1 int
 * @param noonPause boolean
 * @param saturdayClosed boolean
 */
protected OpeningSchedule createOpeningSchedule(boolean sevenTwentyFour, int morning0, int morning1, int pause, int afternoon1, boolean noonPause, boolean saturdayClosed) {
	try {
		if (sevenTwentyFour) {
			// 7 x 24 hours of operation
			IntInterval[] dayRanges= new IntInterval[1];
			dayRanges[0]= new IntInterval(1,7);
			DateInterval[][] timeRanges= new DateInterval[1][1];
			timeRanges[0][0]= new DateInterval(DateExtensions.newTime(0,0,0), DateExtensions.newTime(23,59,59));
			
			return new OpeningSchedule(dayRanges,timeRanges);
		}
	} catch (Exception e) {
		e.printStackTrace();
	}
	try {
		Date morning= DateExtensions.newTime(6 + morning0,0,0);
		Date noon= DateExtensions.newTime(12 + morning1,0,0);
		Date afterNoon= new Date(noon.getTime()+3600*1000*(1 + pause));
		Date evening= DateExtensions.newTime(18 + afternoon1,0,0);

		IntInterval[] dayRanges= new IntInterval[3];
		dayRanges[0]= new IntInterval(2,6);
		dayRanges[1]= new IntInterval(7,7);
		dayRanges[2]= new IntInterval(1,1);

		DateInterval[][] timeRanges= new DateInterval[3][];
		if (noonPause) {
			timeRanges[0]= new DateInterval[2];
			timeRanges[0][0]= new DateInterval(morning,noon);
			timeRanges[0][1]= new DateInterval(afterNoon,evening);
		} else {
			timeRanges[0]= new DateInterval[1];
			timeRanges[0][0]= new DateInterval(morning,evening);
		}			
		if (saturdayClosed) {
			timeRanges[1]= new DateInterval[0];
		} else {
			timeRanges[1]= new DateInterval[1];
			timeRanges[1][0]= new DateInterval(morning,noon);
		}			
		timeRanges[2]= new DateInterval[0];
		return new OpeningSchedule(dayRanges,timeRanges);
	} catch (Exception e) {
		e.printStackTrace();
	}
		
	return null;
}
/**
 * Create a new RateScheme from 2 Strings.
 * 
 * Creation date: (13.12.99 13:00:52)
 * @return com.ibm.ulc.examples.carRental.model.RateScheme
 */
protected RateScheme createRateScheme(String code, String name) {
	return new RateScheme(code, name);
}
/**
 * Answer an array of CarClass test data.
 * 
 * Creation date: (13.12.99 13:00:52)
 * @return String[][]
 */
protected String[][] dataForCarClasses() {
	String[][] data= {
	{"ECO","Economy car"},
	{"CO4","Compact 4WD"},
	{"MED","Intermediate car"},
	{"STD","Standard car"},
	{"ST4","Standard 4WD"},
	{"FUL","Fullsize car"},
	{"FU4","Fullsize 4WD"},
	{"PRE","Premium car"},
	{"PR4","Premium 4WD"},
	{"LX4","Luxury 4WD"},
	{"VAF","Fullsize Van"},
	{"VAS","Mini Van"},
	};
	return data;
}
/**
 * Answer an array of CardCompany test data.
 * 
 * Creation date: (13.12.99 13:00:52)
 * @return String[][]
 */

protected String[][] dataForCardCompanies() {
	String[][] data= {
	{"MC","Mastercard"},
	{"VI","Visa"},
	{"AE","American express"}
	};
	return data;
}
/**
 * Answer an array of Country test data.
 * 
 * Creation date: (13.12.99 13:00:52)
 * @return String[][]
 */
protected String[][] dataForCountries() {
	String[][] data= {
	{"US","United States"},
	{"DE","Germany"},
	{"JP","Japan"},
	{"FR","France"},
	{"GB","Great Britain"},
	{"CA","Canada"},
	{"IT","Italy"},
	{"RU","Russian Federation"},
	{"HU","Hungary"},
	{"IS","Iceland"},
	{"IN","India"},
	{"ID","Indonesia"},
	{"IR","Iran"},
	{"CH","Switzerland"},
	{"SY","Syrian Arab Rep."},
	{"TW","Taiwan"},
	{"TJ","Tajikistan"},
	{"TZ","Tanzania"}};
	return data;
}
/**
 * Answer an array of Location test data.
 * 
 * Creation date: (13.12.99 13:00:52)
 * @return String[][]
 */
protected String[][] dataForLocations() {
	String[][] data= {
	{"NYM","New York Manhattan",
		"1234 E, 80th Street", "New York","NY","US",
		"212-93947629","212-93947630","263432",
		"-5.0"},
	{"NYJ","New York JFK Airport",
		"General Aviation", "New York", "NY", "US",
		"345-56785678", "345-56785677","263433",
		"-5.0"},
	{"MUF","Munich FJS Airport",
		"Terminal B, FJS Flughafen", "Munich", "BA", "DE",
		"089-12345678","089-12345677","947631",
		"1.0"},
	{"MUL","Munich Leopoldstrasse",
		"Leopoldstrasse 674","Munich","BA","DE",
		"089-1235555","089-1235556","7624245",
		"1.0"},
	{"TOC","Tokyo Chuo-ku",
		"Tokuma Fuji 7","Tokyo","T","JP",
		"011-4726492","011-4728403","66234",
		"9.0"},
	{"TON","Tokyo Nerima-ku",
		"Nerima-ku Blvd. 77","Tokyo","T","JP",
		"011-4563666","011-4563667","669924",
		"9.0"},
	{"PAI","Paris Invalides",
		"34, rue des Invalildes","Paris","0","FR",
		"01 23 45 67","01 23 55 67","888244",
		"1.0"},
	{"PAG","Paris CdG Airport",
		"22, Aeroport Charles de Gaulle","Paris","0","FR",
		"01 82 44 82","01 82 44 83","888245",
		"1.0"},
	{"LOV","London Victoria",
		"12, Victoria Street","London","LO","GB",
		"0134-23454567","0134-23454500","756235",
		"0.0"},
	{"LOH","London Heathrow",
		"Terminal A, Heathrow Airport","London","LO","GB",
		"0134-296723","0134-296737","759335",
		"0.0"},
	{"OTA","Ottawa International Airport",
		"204, Vilkamla Drive","Ottawa","ON","CA",
		"503-5649351","503-8725019","41095",
		"-7.0"},
	{"OTL","Ottawa St. Laurent Blvd.",
		"27, Zeggeqje Drive","Ottawa","ON","CA",
		"510-5663557","510-9486471","93022",
		"-7.0"},
	{"MIM","Milan Malpensa Airport",
		"9, Aeroporto Forlanini","Milan","MI","IT",
		"02-8426329","02-4427433","90083",
		"1.0"},
	{"MIP","Milan Piazza Fontana",
		"12, Piazza Fontana","Milan","MI","IT",
		"02-8490831","02-7652261","55331",
		"1.0"},
	{"MOA","Moscow International Airport",
		"108, Heylaazxav Drive","Moscow","RS62","RU",
		"476-4839441","476-7154822","25254",
		"3.0"},
	{"MOR","Moscow Red Square",
		"259, Red Square","Moscow","RS62","RU",
		"884-8173184","884-6726075","56599",
		"3.0"},
	{"BUA","Budapest Ferihegy Airport",
		"35, Yidriijkog Street","Budapest","BU","HU",
		"730-6547246","730-8809274","84439",
		"1.0"},
	{"BUD","Budapest Downtown",
		"239, Sojlusexza Street","Budapest","BU","HU",
		"265-9174314","265-5549791","33367",
		"1.0"},
	{"TEA","Teheran Airport",
		"202, Jikow Street","Teheran","IR19","IR",
		"277-2195399","277-2261032","77237",
		"3.0"},
	{"ZRA","Zuerich Airport",
		"135, Flughafen","Kloten","ZH","CH",
		"01-8527816","01-4710851","74571",
		"1.0"}
	};
	return data;
}
/**
 * Answer an array of RateScheme test data.
 * 
 * Creation date: (13.12.99 13:00:52)
 * @return String[][]
 */
protected String[][] dataForRateSchemes() {
	String[][] data= {
	{"ST0","General"},
	{"GO3","ULC's Gold Card Club"},
	{"SI1","ULC's Silver Card Club"},
	{"CA5","Corporate rate A"},
	{"CB3","Corporate rate B"}
	};
	return data;
}
/**
 * Answer an array of AvailableCarSpecPattern for the exchange date
 * given in the parameter.
 * (In the current implementation, just a random choice, independent from the parameter,
 * with at least a transmission choice and at least an air choice.)
 *
 * Creation date: (10.01.00 10:51:50)
 * @return com.ibm.ulc.examples.carRental.model.AvailableCarSpecPattern[]
 * @param exchangeDate com.ibm.ulc.examples.carRental.model.ExchangeDate
 */
public AvailableCarSpecPattern[] getAvailableCarSpecsPatternsFor(ExchangeDate exchangeDate) {
	if (fCarClasses.length<2) return new AvailableCarSpecPattern[0];
	
	// We have at least 2 car classes now!
	boolean[] takeTheClass= new boolean[fCarClasses.length];
	takeTheClass[0]= true;
	takeTheClass[1]= true;
	int count= 2;
	for (int i=2; i<takeTheClass.length; i++) {
		takeTheClass[i]= RandomDataFactory.booleanTrue(0.5d);
		if(takeTheClass[i]) count++;
	}
	// Create an array of AvailableCarSpecPatterns, and fill it
	AvailableCarSpecPattern[] answer= new AvailableCarSpecPattern[count];
	AvailableCarSpecPattern newInstance= null;
	// The first instance
	newInstance= new AvailableCarSpecPattern();
	newInstance.setCarClass(fCarClasses[0]);
	setTransmission(newInstance, 2);
	setAir(newInstance, 0);
	answer[0]= newInstance;
	// The second instance
	newInstance= new AvailableCarSpecPattern();
	newInstance.setCarClass(fCarClasses[1]);
	setTransmission(newInstance, 0);
	setAir(newInstance, 2);
	answer[1]= newInstance;
	// All the others
	int c= 2;
	for (int i=2; i<takeTheClass.length; i++) {
		if (takeTheClass[i]) {
			newInstance= new AvailableCarSpecPattern();
			newInstance.setCarClass(fCarClasses[i]);
			setTransmission(newInstance, RandomDataFactory.intBelow(3));
			setAir(newInstance, RandomDataFactory.intBelow(3));
			answer[c]= newInstance;
			c++; // That's what it is!
		}
	}
	return answer;
}
/**
 * Getter for fCarClasses
 * 
 * Creation date: (13.12.99 10:18:45)
 * @return com.ibm.ulc.examples.carRental.model.CarClass[]
 */
public CarClass[] getCarClasses() {
	CarClass[] answer = new CarClass[fCarClasses.length];
	System.arraycopy(fCarClasses, 0, answer, 0, fCarClasses.length);
	return answer;
}
/**
 * Getter for fCardCompanies
 * 
 * Creation date: (13.12.99 10:18:45)
 * @return com.ibm.ulc.examples.carRental.model.CardCompany[]
 */
public CardCompany[] getCardCompanies() {
	CardCompany[] answer = new CardCompany[fCardCompanies.length];
	System.arraycopy(fCardCompanies, 0, answer, 0, fCardCompanies.length);
	return answer;
}
/**
 * Getter for fCountries
 * 
 * Creation date: (13.12.99 10:18:45)
 * @return com.ibm.ulc.examples.carRental.model.Country[]
 */
public Country[] getCountries() {
	Country[] answer = new Country[fCountries.length];
	System.arraycopy(fCountries, 0, answer, 0, fCountries.length);
	return answer;
}
/**
 * Get all countries sorted by name.
 * 
 * Creation date: (13.12.99 10:18:45)
 * @return com.ibm.ulc.examples.carRental.model.Country[]
 */
public Country[] getCountriesSortedByName() {
	Country[] countries= getCountries();
	ArrayExtensions.sort(countries, new Comparator() {
		public int compare(Object s1, Object s2) {
			return ((FinalPersistentObject)s1).getName().compareTo(((FinalPersistentObject)s2).getName());
		}
	});	
	return countries;
}
/**
 * Get the first RateScheme or null
 * 
 * Creation date: (13.12.99 10:18:45)
 * @return com.ibm.ulc.examples.carRental.model.RateScheme
 */
public RateScheme getFirstRateScheme() {
	return fRateSchemes.length==0 ? null : fRateSchemes[0];
}
/**
 * Get all countries that have locations, sorted by name.
 * 
 * Creation date: (13.12.99 10:18:45)
 * @return com.ibm.ulc.examples.carRental.model.Country[]
 */
public Country[] getLocationCountriesSortedByName() {
	// find out the size of the array to create.
	Boolean dummy= new Boolean(true);
	Hashtable countriesSet= new Hashtable(fLocations.length);
	for (int i= 0; i<fLocations.length; i++) {
		countriesSet.put(fLocations[i].getCountry(),dummy);
	}

	Country[] countries= new Country[countriesSet.size()];
	int alreadyInserted= 0;
	// use a HashSet to check whether a particular country has already been added.
	countriesSet= new Hashtable(fLocations.length);
	for (int i= 0; i<fLocations.length; i++) {
		Country thisCountry= fLocations[i].getCountry();
		if (!countriesSet.containsKey(thisCountry)) {
			countriesSet.put(thisCountry,dummy);
			countries[alreadyInserted]= thisCountry;
			alreadyInserted++;
		}
	}
	ArrayExtensions.sort(countries,new Comparator() {
		public int compare(Object s1, Object s2) {
			return ((FinalPersistentObject)s1).getName().compareTo(((FinalPersistentObject)s2).getName());
		}
	});
	return countries;
}
/**
 * Getter for fLocations
 * 
 * Creation date: (13.12.99 10:18:45)
 * @return com.ibm.ulc.examples.carRental.model.Location[]
 */
public Location[] getLocations() {
	Location[] answer = new Location[fLocations.length];
	System.arraycopy(fLocations, 0, answer, 0, fLocations.length);
	return answer;
}
/**
 * Get all locations in the given country.
 * 
 * Creation date: (13.12.99 10:18:45)
 * @return com.ibm.ulc.examples.carRental.model.Location[]
 */
public Location[] getLocationsInCountry(Country country) {
	Location[] selected = new Location[fLocations.length];
	int alreadySelected= 0;

	for (int i= 0; i<fLocations.length; i++) {
		Location thisLocation= fLocations[i];
		if (thisLocation.getCountry().equals(country)) {
			selected[alreadySelected]= thisLocation;
			alreadySelected++;
		}
	}
	Location[] answer= new Location[alreadySelected];
	System.arraycopy(selected, 0, answer, 0, alreadySelected);
	return answer;
}
/**
 * Get all locations lying in the given country and in the given state.
 * 
 * Creation date: (13.12.99 10:18:45)
 * @return com.ibm.ulc.examples.carRental.model.Location[]
 */
public Location[] getLocationsInCountryAndState(Country country, String state) {
	Location[] selected = new Location[fLocations.length];
	int alreadySelected= 0;

	for (int i= 0; i<fLocations.length; i++) {
		Location thisLocation= fLocations[i];
		if (thisLocation.getCountry().equals(country)
					&& thisLocation.getState().equals(state)) {
			selected[alreadySelected]= thisLocation;
			alreadySelected++;
		}
	}
	Location[] answer= new Location[alreadySelected];
	System.arraycopy(selected, 0, answer, 0, alreadySelected);
	return answer;
}
/**
 * Getter for fRateSchemes
 * 
 * Creation date: (13.12.99 10:18:45)
 * @return com.ibm.ulc.examples.carRental.model.RateScheme[]
 */
public RateScheme[] getRateSchemes() {
	RateScheme[] answer = new RateScheme[fRateSchemes.length];
	System.arraycopy(fRateSchemes, 0, answer, 0, fRateSchemes.length);
	return answer;
}
/**
 * Answer a test instance.
 * Creation date: (10.01.00 22:42:33)
 * @return com.ibm.ulc.examples.carRental.model.ContractSpec
 */
public static ContractSpec getTestInstance1() {
	DataAccessor da= instance();

	// create the instance
	ContractSpec instance= new ContractSpec();

	// customer
	CustomerSpec customer= instance.getCustomerSpec();
	customer.setResidenceCountry(da.fCountries[0]);
	customer.setRateScheme(da.fRateSchemes[0]);
	customer.setCustomerId("123-456789-0");

	// pickup
	ExchangeDate pickup= instance.getPickupDetails();
	pickup.setLocation(da.fLocations[0]);
	pickup.setDate((new GregorianCalendar(2001,06,14, 0, 0, 0)).getTime());
	pickup.setTime((new GregorianCalendar(1900,00,01, 8,30, 0)).getTime());

	// return
	ExchangeDate returnD= instance.getReturnDetails();
	returnD.setLocation(da.fLocations[0]);
	returnD.setDate((new GregorianCalendar(2001,07, 1, 0, 0, 0)).getTime());
	returnD.setTime((new GregorianCalendar(1900,00,01,13,45, 0)).getTime());
	
	instance.setIsReturnSameAsPickup(true);
	
	// car
	CarSpec car= instance.getCarSpec();
	car.setCarClass(da.fCarClasses[0]);
	car.setAirConditioning(true);
	car.setTransmission(CarSpec.MANUAL_TRANSMISSION);

	// options
	OptionsSpec options= instance.getOptionsSpec();
	options.setChildSeats(0);
	options.setMobile(true);
	options.setNavigation(false);
	options.setLossIns(true);
	options.setPropertyIns(false);

	// guarantee
	Guarantee guarantee= instance.getGuarantee();
	guarantee.setCardCompany(da.fCardCompanies[0]);
	guarantee.setCardNumber("1234-5678-9012-3456");
	guarantee.setExpiration((new GregorianCalendar(2003,11,31,24, 0, 0)).getTime());
	
	return instance;
}
/**
 * Initialize the CarClasses of the receiver with example size given in the parameter.
 * 
 * Creation date: (14.12.99 12:13:40)
 * @param exampleSize int
 */
protected void initializeCarClasses(int exampleSize) {
	int[] instCount= {0, 12, 24, 48, 192,  3072};

	int nInstances= instCount[exampleSize];
	fCarClasses= new CarClass[nInstances];
	
	String[][] data= dataForCarClasses();
	int nSamples= data.length;
	int iSamples= 0;
	String suffix= "";
	
	for (int i=0; i<nInstances; i++) {

		fCarClasses[i]= createCarClass(data[iSamples][0]+suffix,data[iSamples][1]+suffix);
		iSamples++;
		if (iSamples>=nSamples)	{
			iSamples= 0;
			suffix= Integer.toString(i / nSamples);
		}
	}
}
/**
 * Initialize the CardCompanies of the receiver with example size given in the parameter.
 * 
 * Creation date: (14.12.99 12:13:40)
 * @param exampleSize int
 */
protected void initializeCardCompanies(int exampleSize) {
	int[] instCount=  {0,3,6,12,48,768};

	int nInstances= instCount[exampleSize];
	fCardCompanies= new CardCompany[nInstances];
	
	String[][] data= dataForCardCompanies();
	int nSamples= data.length;
	int iSamples= 0;
	String suffix= "";
	
	for (int i=0; i<nInstances; i++) {

		fCardCompanies[i]= createCardCompany(data[iSamples][0]+suffix,data[iSamples][1]+suffix);
		iSamples++;
		if (iSamples>=nSamples)	{
			iSamples= 0;
			suffix= Integer.toString(i / nSamples);
		}
	}
}
/**
 * Initialize the Countries of the receiver with example size given in the parameter.
 * 
 * Creation date: (14.12.99 12:13:40)
 * @param exampleSize int
 */
protected void initializeCountries(int exampleSize) {
	int[] instCount= {0,18,36,72,288,4608};

	int nInstances= instCount[exampleSize];
	fCountries= new Country[nInstances];
	
	String[][] data= dataForCountries();
	int nSamples= data.length;
	int iSamples= 0;
	String suffix= "";
	
	for (int i=0; i<nInstances; i++) {

		fCountries[i]= createCountry(data[iSamples][0]+suffix,data[iSamples][1]+suffix);
		iSamples++;
		if (iSamples>=nSamples)	{
			iSamples= 0;
			suffix= Integer.toString(i / nSamples);
		}
	}
}
/**
 * Initiaize the sample data with example size @exampleSize.
 * 
 * Creation date: (14.12.99 12:28:34)
 * @param exampleSize int
 */
public void initializeData(int exampleSize) {
	initializeCardCompanies(exampleSize);
	initializeCarClasses(exampleSize);
	initializeCountries(exampleSize);
	initializeLocations(exampleSize);
	initializeRateSchemes(exampleSize);
}
/**
 * Initialize the Locations of the receiver with example size given in the parameter.
 * 
 * Creation date: (14.12.99 12:13:40)
 * @param exampleSize int
 */
protected void initializeLocations(int exampleSize) {
	int[] instCount= {0, 20, 100, 1000, 10000, 100000};

	int nInstances= instCount[exampleSize];
	fLocations= new Location[nInstances];
	
	String[][] data= dataForLocations();
	int nSamples= data.length;
	int iSamples= 0;
	String suffix= "";
	
	for (int i=0; i<nInstances; i++) {
		String[] thisData= new String[10];
		System.arraycopy(data[iSamples],0,thisData,0,10);
		thisData[0]= data[iSamples][0]+suffix;
		thisData[1]= data[iSamples][1]+suffix;
		fLocations[i]= createLocation(thisData);
		iSamples++;
		if (iSamples>=nSamples)	{
			iSamples= 0;
			suffix= Integer.toString(i / nSamples);
		}
	}
}
/**
 * Initialize the RateSchemes of the receiver with example size given in the parameter.
 * 
 * Creation date: (14.12.99 12:13:40)
 * @param exampleSize int
 */
protected void initializeRateSchemes(int exampleSize) {
	int[] instCount= {0,5,10,20,80,1280};

	int nInstances= instCount[exampleSize];
	fRateSchemes= new RateScheme[nInstances];
	
	String[][] data= dataForRateSchemes();
	int nSamples= data.length;
	int iSamples= 0;
	String suffix= "";
	
	for (int i=0; i<nInstances; i++) {

		fRateSchemes[i]= createRateScheme(data[iSamples][0]+suffix,data[iSamples][1]+suffix);
		iSamples++;
		if (iSamples>=nSamples)	{
			iSamples= 0;
			suffix= Integer.toString(i / nSamples);
		}
	}
}
/**
 * Answer the canonical instance.
 * 
 * Creation date: (14.12.99 12:06:17)
 * @return com.ibm.ulc.examples.carRental.model.DataAccessor
 */
public static DataAccessor instance() {
	return fgInstance;
}
/**
 * Answer a rate scheme named name.
 *
 * Creation date: (07.01.00 12:17:45)
 * @return com.ibm.ulc.examples.carRental.model.RateScheme
 * @param name java.lang.String
 */
public RateScheme rateSchemeNamed(String name) {
	for (int i=0; i<fRateSchemes.length; i++)
		if (fRateSchemes[i].getName().equals(name))
			return fRateSchemes[i];
	return null;
}
/**
 * Set the 'airConditioning' attribute of the pattern according to parameter:
 *	0: 	{true}
 *	1: 	{false}
 *	2: 	{true,false}
 *
 * Creation date: (10.01.00 11:19:41)
 * @param pattern com.ibm.ulc.examples.carRental.model.AvailableCarSpecPattern
 * @param parameter int
 */
protected void setAir(AvailableCarSpecPattern pattern, int parameter) {
	int length= parameter==2 ? 2 : 1;
	boolean[] setting= new boolean[length];
	
	int i= 0;
	if (parameter % 2 == 0) {
		setting[i]= true;
		i++;
	}
	if (parameter > 0) {
		setting[i]= false;
	}
	pattern.setAirConditioning(setting);
}
/**
 * Set the 'transmission' attribute of the pattern according to parameter:
 *	0: 	{automatic}
 *	1: 	{manual}
 *	2: 	{automatic,manual}
 *
 * Creation date: (10.01.00 11:19:41)
 * @param pattern com.ibm.ulc.examples.carRental.model.AvailableCarSpecPattern
 * @param parameter int
 */
protected void setTransmission(AvailableCarSpecPattern pattern, int parameter) {
	int length= parameter==2 ? 2 : 1;
	String[] setting= new String[length];
	
	int i= 0;
	if (parameter % 2 == 0) {
		setting[i]= CarSpec.AUTOMATIC_TRANSMISSION;
		i++;
	}
	if (parameter > 0) {
		setting[i]= CarSpec.MANUAL_TRANSMISSION;
	}
	pattern.setTransmission(setting);
}
}
