package com.ibm.ulc.examples.carRental.model;

import java.util.Date;
import java.util.GregorianCalendar;
import java.text.DateFormatSymbols;

/**
 * Instances describe the opening hours of a shop.
 *
 * A typical schedule looks as follows:
 *   Monday-Friday: 8-12, 14-20;
 *   Saturday:      8-13;
 *   Sunday:        closed
 *
 * So, an opening schedudle has the structure of a dictionary
 * mapping a range of days (of the week) 
 * to a collection of ranges of time (of the day).
 *
 * Because Hashtables do not support iterating on both
 * keys and values simultaneously, we did not use this class to model it;
 * instead, we carry two arrays fDayWRanges and fTimeRanges,
 * with the convention that the i-th element of the first array
 * maps to the i-th element of the second.
 *
 * @author: Markus Marzetta (CS-ZRH)
 */
public class OpeningSchedule extends CarRentalObject {
	/**
	 * Day of week ranges.
	 * These are ClosedIntervals of ints, whose meaning is given is Calendar
	 * (1=SUNDAY,...,7=SATURDAY)
	 */
	protected IntInterval[] fDayWRanges= null;
	/**
	 * Time ranges, i.e.
	 * Arrays of ClosedIntervals of Dates representing the opening hours
	 * on the days of the corresponding interval of fDayWRanges.
	 *
	 * INTEGRITY Condition: fDayWRanges and fTimeRanges must have the same length.
	 */
	protected DateInterval[][] fTimeRanges= null;
	
/**
 * OpeningSchedule constructor comment.
 */
public OpeningSchedule() {
	super();
}
/**
 * Insert the method's description here.
 * 
 * Creation date: (07.12.99 17:44:03)
 * @param dayWRanges com.ibm.ulc.examples.carRental.model.ClosedInterval[]
 * @param timeRanges com.ibm.ulc.examples.carRental.model.ClosedInterval[][]
 */
public OpeningSchedule(IntInterval[] dayWRanges, DateInterval[][] timeRanges) {
	fDayWRanges= dayWRanges;
	fTimeRanges= timeRanges;
	}
/**
 * Answer true, if the time parameter lies in one of the closed intervals
 * timeRanges.
 * 
 * Creation date: (07.12.99 17:29:24)
 * @return boolean
 * @param timeRanges com.ibm.ulc.examples.carRental.model.ClosedInterval[]
 * @param time java.util.Date
 */
protected boolean doIntervalsInclude(DateInterval[] timeRanges, Date time) {
	for (int i= 0; i<timeRanges.length; i++) {
		if (timeRanges[i].includes(time)) {
			return true;
		}
	}
	return false;
}
/**
 * Answer the index of the day range containing the parameter.
 * Answer -1 if no range was found.
 * 
 * Creation date: (07.12.99 17:08:45)
 * @return int
 * @param dayW int
 */
 
protected int indexOfDayRangeContaining(int dayW) {
	for (int i= 0; i<fDayWRanges.length; i++) {
		if (fDayWRanges[i].includes(dayW)) {
			return i;
		}
	}
	return -1;
}
/**
 * Answer true, if the schedule requires opening at the time given in the parameter.
 * 
 * Creation date: (07.12.99 16:35:25)
 * @return boolean
 * @param date java.util.Date
 */
public boolean isOpenAt(Date date) {

	// get the time schedule for the given day	
	int dayOfWeek= DateExtensions.getDayOfWeek(date);
	int i= indexOfDayRangeContaining(dayOfWeek);
	if (i<0) {
		return false;
	}
	DateInterval[] timeRanges= fTimeRanges[i];

	// now check, whether the given time lies in an open range
	if (doIntervalsInclude(timeRanges, DateExtensions.normalizeTime(date)))
		return true;
	else
		return false;
}
/**
 * Insert the method's description here.
 * 
 * Creation date: (09.12.99 10:42:48)
 * @param timeRanges com.ibm.ulc.examples.carRental.model.DateInterval
 * @param buffer java.lang.StringBuffer
 */
protected void printTimeRange(DateInterval timeRange, StringBuffer buffer) {
	GregorianCalendar startCal= new GregorianCalendar();
	startCal.setTime(timeRange.getMin());
	GregorianCalendar stopCal= new GregorianCalendar();
	stopCal.setTime(timeRange.getMax());

	int startH= startCal.get(GregorianCalendar.HOUR_OF_DAY);
	int startM= startCal.get(GregorianCalendar.MINUTE);
	int stopH= stopCal.get(GregorianCalendar.HOUR_OF_DAY);
	int stopM= stopCal.get(GregorianCalendar.MINUTE);

	buffer.append(startH);
	if (startM>0)
		buffer.append(":").append(startM);
	buffer.append("-");
	buffer.append(stopH);
	if (stopM>0)
		buffer.append(":").append(stopM);
	}
/**
 * Print the opening schedule
 * 
 * Creation date: (07.12.99 17:51:16)
 * @return java.lang.String
 */
public String toString() {
	String daySep= "; ";
	StringBuffer buffer= new StringBuffer(100);
	String[] weekDays= (new DateFormatSymbols()).getWeekdays();

	boolean firstRangePrinted= false;
	for (int i= 0; i<fDayWRanges.length; i++) {
		IntInterval dayRange= fDayWRanges[i];
		DateInterval[] timeRanges= fTimeRanges[i];

		if (firstRangePrinted) {
			buffer.append(daySep);
		}
		buffer.append(weekDays[dayRange.getMin()]);
		if (dayRange.size()>1) {
			buffer.append("-");
			buffer.append(weekDays[dayRange.getMax()]);
		}
		buffer.append(": ");
		if (timeRanges.length>0) {
			printTimeRange(timeRanges[0],buffer);
			for (int j=1; j<timeRanges.length; j++) {
				buffer.append(", ");
				printTimeRange(timeRanges[j],buffer);
			}
		} else {
			buffer.append("closed");
		}
		firstRangePrinted= true;
	}

	return buffer.toString();
}
}
