package com.ibm.ulc.ui;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.io.*;
import java.util.*;
import java.net.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.peer.*;
import javax.swing.*;
import com.ibm.ulc.base.IDefaults;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.*;
import com.ibm.ulc.ui.base.*;
import javax.swing.ToolTipManager;
public class UI extends Server implements ICallable, IConnectionController, IRequestListener, IDefaults {
	protected UlcHashtable fConnections = new UlcHashtable(); // List of active outgoing connections
	protected Vector fActiveConnections = new Vector(); // List of active connections
	protected static long fgConnectionId = 0;
	protected Hashtable fConnectionsInProgress = new Hashtable();
	protected Vector fConnectionListeners = null;
	protected boolean fServerMode = true;
	protected boolean fTest = false;
	protected Registry fRegistry = null;
	protected IRequestProcessor fRequestProcessor;
	protected ULCOrbMonitor fMonitor = null;
	protected boolean fIsApplet = false;
	protected Object fApplet = null;
	protected String fLookAndFeelClassName = null;
	protected boolean fUseDoubleBuffering = true;
	protected Point fTopLeft = new Point(20, 20);
	protected ILookAndFeel fLookAndFeel = null;
	protected IClassLookUp fClassLookUp = null;
	public static boolean fgExitOnLastConnectionClose = true;
	public static boolean fgReconnectOnException = true;

	/**
	 * @deprecated	As of ULC R3.1, use fUILocale instead
	 * @see #fUILocale
	 */
	public static Locale fgUILocale= null;
	/**
	 * @deprecated	As of ULC R3.1, use fDebug instead
	 * @see #fDebug
	 */
	public static boolean fgDebug= false;
	protected Locale fUILocale= null;
	protected boolean fDebug= false;	
	protected String fLogFileName = "ulcui.log";
	protected ILogStream fLogStream = null;
	protected boolean fLogEnabled = false;
	protected String fUserParameter = "";
	protected ResourceBundle fResourceBundle = null;
	protected UIResourceCacheAbstract fResourceCache = null;
	protected String fUrlString= "";
	protected int fRunnerCount=4;
	private Vector fErrorMessageViews= new Vector();
	/**
	 * <code>true</code> if the UI in client mode will attempt to reconnect to the ULC server when the connection goes down.
	 * @serial
	 */
	private boolean fAutoReconnect= false;
	/**
	 * <code>true</code> if the ULC server is in client reconnect mode
	 * @serial
	 */
	private boolean fServerInClientReconnectMode= false;		
public UI(String[] args, IRequestProcessor requestProcessor) {
	this(args,requestProcessor,null);
}
/**
 * This method is called either from the main or from the UIApplet
 * Note: By default the SwingRequestProcessor is used. During debugging
 * it is convenient to use the AwtRequestProcessor which allows the debugger
 * not to block the UI thread.
 */
public UI(String[] args, IRequestProcessor requestProcessor, Object applet) {
	super("UI Engine");
	boolean argsOK = true;
	boolean useMonitor = false;
	boolean winConsole = false;
	boolean memoryWatcher = false;
	int serverPort = Common.SERVER_PORT;
	fUrlString = "ulc://" + UlcConnection.getLocalHost() + ":" + serverPort;
	//String clientUrl = null;
	fApplet = applet;
	ToolTipManager toolTipManager = ToolTipManager.sharedInstance();
	toolTipManager.setDismissDelay(1000000);
	addConnectionController(this);
	if (Arrays.asList(args).contains("-debug")) {
		fDebug= true;
		System.out.println("UI constructor called. fgConnectionId= " + String.valueOf(fgConnectionId));
		UlcConnection.fgDebug= true;
		UlcTransport.fgDebug= true;
		UlcTransportServer.fgDebug= true;
	}
	else {
		UlcConnection.fgDebug= false;
		UlcTransport.fgDebug= false;
		UlcTransportServer.fgDebug= false;		
	}
	for (int i = 0; i < args.length; i++) {
		String arg = args[i];
		if (arg.equals("-applet")) {
			fIsApplet = true;
			continue;
		}
		if (arg.equals("-ipaddr")) {
			UlcConnection.fgIPAddr = true;
			continue;
		}
		if (arg.equals("-test")) {
			fTest = true;
			if (i < (args.length - 1)) {
				i++;
				String testCount = args[i];
				if (!testCount.startsWith("-")) {
					if (i < args.length) {
						try {
							fRunnerCount = (new Integer(testCount)).intValue();
						} catch (Exception e) {
							i--;
						};
					}
				} else
					i--;
			}
			continue;
		}
		if (arg.equals("-m")) {
			useMonitor = true;
			continue;
		}
		if (arg.equals("-memory")) {
			startMemoryWatcher();
			continue;
		}
		if (arg.equals("-doublebuffering")) {
			fUseDoubleBuffering = !fUseDoubleBuffering;
			continue;
		}
		if (arg.equals("-look")) {
			i++;
			if (i < args.length) {
				fLookAndFeelClassName = args[i];
			} else {
				System.out.println(getResourceString("MsgWarnExpectingLookName"));
			}
			continue;
		}
		if (arg.equals("-logfilename")) {
			i++;
			if (i < args.length) {
				fLogFileName = args[i];
			} else {
				System.out.println(getResourceString("MsgWarnLogFileName"));
			}
			continue;
		}
		if (arg.equals("-enablelog")) {
			setLoggingEnabled(true);
			continue;
		}
		if (arg.equals("-locale")) {
			i++;
			if (i < args.length) {
				String localeString = args[i];
				int ndx = localeString.indexOf('_');
				if ((ndx > 0) && (localeString.length() > ndx)) {
					String language = localeString.substring(0, ndx);
					String country = localeString.substring(ndx + 1);
					setUILocale(new Locale(language.toLowerCase(), country.toUpperCase()));
				}
			} else {
				System.out.println(getResourceString("MsgWarnExpectingLocale"));
			}
			continue;
		}
		if (arg.equals("-wincon")) {
			winConsole = true;
			continue;
		}
		if (arg.equals("-url")) {
			i++;
			if (i < args.length) {
				fUrlString = args[i];
				setServerMode(false);
			} else {
				System.out.println(getResourceString("MsgWarnExpectingUrl"));
				argsOK = false;
			}
			continue;
		}
		if (arg.equals("-indirectUrl")) {
			i++;
			if (i < args.length) {
				fUrlString = UI.parseIndirectArguments(args[i], fDebug);
				if (fUrlString != null)
					setServerMode(false);
				else
					argsOK = false;
			}
			continue;
		}
		if (arg.equals("-genServerUrlFile")) {
			i++;
			if (i < args.length) {
				setServerUrlFileName(args[i]);
			} else {
				System.out.println(getResourceString("MsgWarnExpectingGenServerFileName"));
				argsOK = false;
			}
			continue;
		}
		if (arg.equals("-userParameter")) {
			i++;
			if (i < args.length) {
				fUserParameter = args[i];
			} else {
				System.out.println(getResourceString("MsgWarnExpectingUserParameter"));
			}
			continue;
		}
		if (arg.equals("-autoReconnect")) {
			setAutoReconnect(true);
			continue;
		}
		if (arg.equals("-server")) {
			i++;
			if (i < args.length) {
				try {
					// argument could be a port number
					serverPort = Integer.parseInt(args[i]);
					if (serverPort <= 0) {
						System.out.println(getResourceString("MsgWarnInvalidPortNumber"));
						argsOK = false;
					} else {
						fUrlString = "ulc://" + UlcConnection.getLocalHost() + ":" + serverPort;
					}
				} catch (NumberFormatException e) {
					// no port number -> argument must be a URL
					fUrlString = args[i];
					// check if valid
					if (UlcConnection.getHostFromUrlString(fUrlString) == null) {
						System.out.println(getResourceString("MsgInvalidUrlOption"));
						i--;
						argsOK = false;
					} else {
						serverPort = UlcConnection.getPortFromUrlString(fUrlString);
					}
				}
			} else {
				System.out.println(getResourceString("MsgWarnExpectingPortNumber"));
				argsOK = false;
			}
			continue;
		}
	}
	if (!argsOK)
		return;
	fClassLookUp = createClassLookUp();
	fResourceCache = createUIResourceCache();
	fLookAndFeel = createLookAndFeel();
	fRegistry = new Registry(null);
	fRegistry.register(Common.FACTORY_ID, this);
	fRequestProcessor = requestProcessor;

	/*
	 * for  some strange (and yet to be discovered) reasons
	 * the first window generated by the UI Engine appears behind all other windows.
	 * The problem doesn't occur if either the ORBMonitor or the Console window exists.
	 * We assume that this is a threading problem.
	 *
	 * A workaround is to create a window, show it and immediately hide it.
	 * But only if neither a ORBMonitor nor a Console window exists.
	 * The boolean flag fgNeedWorkaroundWindow tells us, whether we need the
	 * workaround window.
	 */
	boolean needWorkaroundWindow = true;
	if (winConsole) {
		SwingConsole.createConsole("UI Engine ", getNextWindowPosition(), this);
		needWorkaroundWindow = false;
	}
	Common.printULCVersionAndProtocolVersion();
	if (useMonitor) {
		fMonitor = new ULCOrbMonitor(this);
		Server.addConnectionController(fMonitor);
		fMonitor.setVisible(true);
		needWorkaroundWindow = false;
	}
	if (needWorkaroundWindow) {
		Frame f = new Frame("");
		f.setBounds(-100, -100, 1, 1); // move window outside of screen
		f.setVisible(true);
		f.dispose();
	}
}
public void addUIConnectionListener(ORBConnection conn) {
	if (fConnectionListeners == null)
		fConnectionListeners = new Vector();
	fConnectionListeners.addElement(conn);
}
/**
 * Reconnect to the ULC server.
 * This method may only be called when auto-reconnect is on
 * and the UI is in client mode.
 *
 * @param	connection	The connection which has been lost
 * @since	R3.1
 */
protected void autoReconnect(UlcConnection connection) {
	if (isAutoReconnectOn()) {
		String clientData= connection.getClientData();
		if (clientData.startsWith("connectTo:") || clientData.startsWith("mustStartNewContext"))
			clientData= "";	// clear client data from previous reconnect
		callApp(connection.getUrlString(), clientData, reconnectDelayInMillis());
	}
}
public static void beep() {
	Toolkit tk = Toolkit.getDefaultToolkit();
	if (tk != null)
		tk.beep();
}
public String callApp(String clientUrl, String clientData) {
	return callApp(clientUrl,clientData,0);
}
public String callApp(String clientUrl, String clientData, long waitMillis) {
	String connId = null;
	// create connection in a separate thread to avoid blocking the UI.
	// (when the connection succeeds the thread will post a request
	// to call our connectionSucceeded() method and then terminate)
	connId = nextConnectionId();
	ConnectionInProgress connInProg = (ConnectionInProgress) fConnectionsInProgress.get(connId);
	if (connInProg == null) {
		connInProg = new ConnectionInProgress(this, connId, clientData, clientUrl, fRequestProcessor, true, new Registry(fRegistry), waitMillis);
		// keep track of connections in progress
		fConnectionsInProgress.put(connId, connInProg);
		// start thread
		connInProg.start();
	}
	return connId;
}
/**
 * This method gets called when the UIApplet is being destroyed.
 */
public void cleanup() {
	boolean needsShutdown= fIsApplet && (fActiveConnections == null || fActiveConnections.isEmpty());
	if (fDebug)
		System.out.println("<UI.cleanup() with shutdown= " + needsShutdown);

	if (fActiveConnections != null) {
		Enumeration conns = fActiveConnections.elements();
		while (conns.hasMoreElements()) {
			try {
				final ORBConnection c = (ORBConnection) conns.nextElement();
				if (c.getRegistry().find(Common.FACTORY_ID) == this) {
					if (fDebug)
						System.out.println("posting close request to " + c);
					DeferredRequest r = new DeferredRequest(c) {
						public void safeDispatch() {
							c.close();
						}
					};
					c.postRequest(r);
				}
			} catch (ClassCastException e) {
			}
		}
	}
	// kill threads that were started for connections in progress
	if (fConnectionsInProgress != null) {
		for (Enumeration connsInProg = fConnectionsInProgress.elements(); connsInProg.hasMoreElements();) {
			ConnectionInProgress connInProg = (ConnectionInProgress) connsInProg.nextElement();
			if (connInProg.getUI() == this) {
				connInProg.stopMe();
				try {
					connInProg.join();
				} catch (InterruptedException e) {
					System.out.println("WARNING: UI.cleanup(): Thread interrupted during join");
					e.printStackTrace();
				}
			}
		}
		fConnectionsInProgress.clear();
	}
	if (needsShutdown)
		startShutDownThread();
	if (fDebug)
		System.out.println(">UI.cleanup()");	
}
public void connectionCreated(IConnection connection) {
	if (fDebug)
		System.out.println("UI.connectionCreated(). URL=" + connection.getUrlString());
	((ORBConnection) connection).addRequestListener(this);
	logPrintln("Conn Created: ");
}
public synchronized void connectionEnded(IConnection connection, UlcTransportException ex) {
	if (fDebug) {
		System.out.println("UI.connectionEnded(). URL=" + connection.getUrlString());
		if (ex == null)
			System.out.print("                        Exception is null");
		else
			System.out.print("                        Exception=" + ex);
		System.out.println();
	}
	if (ex == null)
		logPrintln("Conn Ended: ");
	else
		logPrintln("Conn Ended: " + ex.toString());

	// Display error message if requested by ULCContext on startup
	String msg= connection.getConnDownErrorMsg();	
	if (ex != null && msg != null) {
		if (isAutoReconnectOn())
			msg= msg + "\n\n" + getResourceString("MsgInformAboutReconnect");
		displayConnErrorMsg(msg);
	}
	ORBConnection conn = (ORBConnection) connection;

	// Reconnect to ULC server if desired
	if (isAutoReconnectOn() && ex != null)
		autoReconnect(conn);
		
	getActiveConnections().removeElement(conn);
	conn.removeRequestListener(this);
	String connId = conn.getId();
	if (connId == null) {
		// no outgoing connection
		return;
	}
	String host = conn.getHost();
	String clientData = conn.getClientData();
	int index = -1;
	if (fConnectionListeners != null) {
		index = fConnectionListeners.indexOf(conn);
		if (index >= 0)
			fConnectionListeners.removeElementAt(index);
	}
	if (fConnections != null) {
		if (fConnections.remove(connId) != null) {
			if (fConnectionListeners != null) {
				// only send notifications if connection was
				// created using a hostname
				if (host != null) {
					Enumeration e = fConnectionListeners.elements();
					while (e.hasMoreElements()) {
						ORBConnection c = (ORBConnection) e.nextElement();
						if (c != null) {
							Anything a = new Anything();
							a.put("host", conn.getHost());
							a.put("port", conn.getPort());
							a.put("connectionId", connId);
							a.put("clientData", clientData);
							c.send(Common.FACTORY_ID, "removedUIConnection", a);
						}
					}
				}
			}
		}
		if (fConnections.isEmpty() && fConnectionsInProgress.isEmpty() && getActiveConnections().isEmpty() && (ex == null || !isAutoReconnectOn()))
			startShutDownThread();
	}
}
public synchronized void connectionStarted(IConnection connection, UlcTransportException ex) {
	ORBConnection conn = (ORBConnection) connection;
	String connId = conn.getId();
	if (fDebug) {
		System.out.println("UI.connectionStarted(). URL=" + connection.getUrlString());
		if (ex == null)
			System.out.print("                      Exception is null");
		else
			System.out.print("                      Exception=" + ex);
		System.out.println();
	}
	if (ex == null)	{
		getActiveConnections().addElement(connection);
		logPrintln("Conn Started: ");
	} else
		logPrintln("Conn Started: " + ex.toString());

	if (connId == null || conn.find(Common.FACTORY_ID) != this)
		// no outgoing connection
		return;

	ConnectionInProgress connInProgress = null;
	// remove entry from connections-in-progress list and send init if no exception occurred
	if (fConnectionsInProgress != null) {
		connInProgress = (ConnectionInProgress) fConnectionsInProgress.get(connId);
		if (connInProgress != null) {
			if (connInProgress.getUI() != this)
				return; //its not from this instance of the UI
			if (ex == null) {
				fConnectionsInProgress.remove(connId);
				sendInit(conn);
				// remove from InProgress table (thread ends by itself)				
			}
		}
	}
	String host = conn.getHost();
	String clientData = conn.getClientData();
	if (ex == null) { // there was no exception
		// finish connection setup
		// add to list of outgoing connections
		fConnections.put(connId, conn);
		// tell connection listeners
		if (fConnectionListeners != null) {
			Enumeration e = fConnectionListeners.elements();
			while (e.hasMoreElements()) {
				ORBConnection c = (ORBConnection) e.nextElement();
				if (c != null) {
					Anything a = new Anything();
					if (host != null)
						a.put("host", host);
					a.put("port", conn.getPort());
					a.put("connectionId", connId);
					a.put("clientData", clientData);
					c.send(Common.FACTORY_ID, "addedUIConnection", a);
				}
			}
		}
	} else { // exception during startup of connection
		if (!isServerMode() && reconnectAfterException(conn, ex)) {
			if (connInProgress == null)
				callApp(conn.getUrlString(), clientData, reconnectDelayInMillis());
			else {
				retryConnectionUsing(connInProgress);
			}
		} else {
			// tell connection listeners
			if (fConnectionListeners != null) {
				Enumeration e = fConnectionListeners.elements();
				while (e.hasMoreElements()) {
					ORBConnection c = (ORBConnection) e.nextElement();
					if (c != null) {
						Anything a = new Anything();
						if (host != null)
							a.put("host", host);
						a.put("port", conn.getPort());
						a.put("connectionId", connId);
						a.put("clientData", clientData);
						c.send(Common.FACTORY_ID, "removedUIConnection", a);
					}
				}
			}
		}
	}
}
public void connectionSucceeded(ORBConnection conn) {
}
/**
 * Convenience method to convert an OrbRequest to a String for logging purposes
 */
protected String convertORBRequestToStringForLog(String prefix, ORBRequest o, String suffix) {
	return Common.convertORBRequestToStringForLog(prefix, o, suffix);
}
public IClassLookUp createClassLookUp() {
	return new UiClassLookUp();
}
public void createConnection(UlcTransport transport) {
	ORBConnection conn=
		new ORBConnection(transport,fRequestProcessor,true,new Registry(fRegistry));
	propagateConnectionCreated(conn);   // notify connection controllers
	try {
		conn.start();
		propagateConnectionStarted(conn,null);   // notify connection controllers
	}
	catch (UlcTransportException ex) {
		propagateConnectionStarted(conn,ex);   // notify connection controllers
	}
}
public ILogStream createLogStream(String logFileName) {
	return new UlcLog(logFileName);
}
/**
 * @return com.ibm.ulc.ui.ILookAndFeel
 */
public ILookAndFeel createLookAndFeel() {
	return new UILookAndFeel(getLookAndFeelClassName());
}
public UIResourceCacheAbstract createUIResourceCache() {
	if (fApplet == null)
		return new UIResourceFileCache();
	else
		return new UIAppletResourceCache();
}
/**
 * returns a default widget that is used in place of the widget 
 * requested but not found.
 */
protected UIProxy defaultWidgetFor(String type, Anything args) {
	UIProxy m = null;
	m = internalNewInstance(getClassNameFor("Label"));
	if (m != null) {
		UIComponent c = (UIComponent) m;
		c.setLabel(type);
	}
	return m;
}
protected void displayConnErrorMsg(String errorMessage) {
	updateErrMsgViewCollection(); // make sure the collection doesn't get too big	
	JOptionPane optionPane= new JOptionPane();
	optionPane.setMessageType(JOptionPane.ERROR_MESSAGE);
	optionPane.setIcon(new ImageIcon(getClass().getResource("alert.gif")));
	optionPane.setMessage(errorMessage);
	JDialog dialog= optionPane.createDialog(null, "ULC Error");
	dialog.setModal(false);
	dialog.show();
	fErrorMessageViews.add(dialog);	
}
/**
 * This method is called to start the server listening or connect to a remote server.
 * Prior to R3.0 this method was called during the instantiation of the UI. In R3.0
 * onwards this method must be called explicitly.
 */
public void establishConnection() {
	if (!isServerMode()) {
		if (fTest) {
			for (int i = 0; i < fRunnerCount; i++) {
				Runner t = new Runner("Runner-" + i, this, fUrlString);
				t.start();
			}
		} else
			callApp(fUrlString, "");
	} else {
		int ret = start(fUrlString); // start server on this port
		if (ret != 0) { // trouble! (server socket already open)
			UlcObject.trouble2Err("main", getResourceString("MsgCouldNotCreateSocket"));
			System.exit(-1); //FIXME: we shouldn't do that, but some threads keeps us alive...
		}
	}
}
public void free() {
	/*
	 * DO NOT SET fApplet to null because it
	 * will be used later to notify fApplet
	 * when the UI is totally destroyed.
 	 */
	getActiveConnections().clear();
	fConnections.clear();
	fRegistry.shutdown();
}
/**
 * Retrieve the list of active connections in this VM
 *
 */
public Vector getActiveConnections() {
	return fActiveConnections;
}
public Object getApplet() {
	return fApplet;
}
/**
 * returns a fully qualified class name.
 * If typeString contains '.' then it is assumed that
 * the typeString already represents a fully qualified 
 * class name,  otherwise the typeString is looked up
 * in the UIClassLookUp and if a mapping exists the result is returned.
 * If a mapping is not found then the class is assumed to be in the
 * com.ibm.ulc.ui package and the typeString is prefixed with 
 * the package name prefix;
 */
public String getClassNameFor(String typeString) {
	if (typeString.indexOf('.') != -1)
		return typeString;
	String className = fClassLookUp.classNameFor(typeString);
	if (className != null)
		return className;
	return "com.ibm.ulc.ui.UI" + typeString;
}
public boolean getDoubleBuffering() {
	return fUseDoubleBuffering;
}
/**
  * Returns true if the UI will exit when the last connection is closed
  *
  * @return boolean if false, the UI must be explicitly terminated by calling System.exit
  */
public static boolean getExitOnLastConnectionClosedFlag() {
	return fgExitOnLastConnectionClose;
}
/**
 * Returns the registered id for this proxy.
 * The UI always has the FACTORY_ID
 */
public int getId() {
	return Common.FACTORY_ID;
}
void getInstalledLookAndFeels(ORBConnection conn) {
	if (fLookAndFeel != null)
		fLookAndFeel.getInstalledLookAndFeels(conn);
}
/**
 * global window positioning strategy: staggering
 */
public ILogStream getLogStream() {
	if (fLogStream != null)
		return fLogStream;
	if (fLogEnabled) {
		if (fLogStream == null)
			fLogStream = createLogStream(fLogFileName);
		return fLogStream;
	}
	return null;
}
void getLook(ORBConnection conn) {
	if (fLookAndFeel != null)
		fLookAndFeel.getLook(conn);
}
public String getLookAndFeelClassName() {
	return fLookAndFeelClassName;
}
/**
 * global window positioning strategy: staggering
 */
public Point getNextWindowPosition() {
	fTopLeft.x += 20;
	if (fTopLeft.x > 550)
		fTopLeft.x = 20;
	fTopLeft.y += 20;
	if (fTopLeft.y > 350)
		fTopLeft.y = 20;
	return new Point(fTopLeft.x, fTopLeft.y);
}
public static boolean getReconnectAfterException() {
	return fgReconnectOnException;
}
/**
 * Send the requested reconnect token to the ULC side
 * which waits/blocks for the answer.
 *
 * This method is invoked when the server allows client
 * reconnection and when the server reconnection strategy
 * asks the UI for the token (which is not a must).
 *
 * @param	conn			The <code>ORBConnection</code> on which the reply should be sent.
 * @param	challenge		The <code>String</code> that can be used to encrypt the answer.
 * @param	strategyName	The typeString used to create an instance of the strategy.
 * @since	R3.1
 */
protected void getReconnectToken(ORBConnection conn, String challenge, String reconnectTokenStrategyType) {
	UIProxy strategy= internalNewInstance(getClassNameFor(reconnectTokenStrategyType));
	if (strategy == null || !(strategy instanceof IReconnectTokenStrategy)) {
		conn.send(Common.FACTORY_ID, "getReconnectToken", new Anything()); // wakup server
		conn.postRequest(new CloseRequest(conn, null));
		throw new Error("Illegal reconnect token strategy");
	}
	String value= ((IReconnectTokenStrategy) strategy).getToken(challenge);
	Anything a= null;
	if (value == null)
		a= new Anything();
	else
		a= new Anything(value);	
	conn.send(Common.FACTORY_ID, "getReconnectToken", a);
}
public IRequestProcessor getRequestProcessor() {
	return fRequestProcessor;
}
/**
 * Returns the Resource Bundle base file name.
 *
 * @return <code>String</code> or null if the subclass has not implemented this method.
 * @see getResourceBundleBaseFileName
 * @see getResourceString
 */
public String getResourceBundleBaseFileName() {
	return "ulcui";
}
/**
 * Returns the configured UIResourceCache object for this UI.
 */
public UIResourceCacheAbstract getResourceCache() {
	return fResourceCache;
}
/**
 * Returns the locale to be used for resource lookup.
 * The locale used will be by default the locale of the UI Engine.
 * If subclasses want to override the locale used they should
 * implement this method and return the locale to be used for the context.
 *
 * @param key <code>String</code> lookup key.
 * @return <code>String</code>
 * @see getResourceBundleBaseFileName
 * @see getResourceString
 */
public Locale getResourceLocale() {
	return getUILocale();
}
/**
 * Gets the resource object identified by the specified key from the specified ResourceBundle.
 * The resourceBundle is identified by the <code>getResourceBundleBaseFileName</code>
 * Subclasses must implement <code>getResourceBundleBaseFileName</code> if they
 * are using this method to retrieve the correct String for the specified locale.
 * The locale used will be by default the locale of the UI Engine.
 * If subclasses want to override the locale used they can implement 
 * the method <code>getResourceLocale</code>
 *
 * @param key <code>String</code> lookup key.
 * @return <code>Object</code>
 * @see getResourceBundleBaseFileName
 * @see getResourceLocale
 */
public Object getResourceObject(String key) {
	if (fResourceBundle == null) {
		try {
			String baseFileName = "com.ibm.ulc.ui." + getResourceBundleBaseFileName();
			if (baseFileName != null)
				fResourceBundle = PropertyResourceBundle.getBundle(baseFileName, getResourceLocale());
		} catch (MissingResourceException e) {
		}
	}
	Object value;
	if (fResourceBundle == null)
		return key;
	try {
		value = fResourceBundle.getString(key);
	} catch (MissingResourceException e) {
		value = key;
	}
	return value;
}
/**
 * Gets a string from the specified ResourceBundle.
 * The resourceBundle is identified by the <code>getResourceBundleBaseFileName</code>
 * Subclasses must implement <code>getResourceBundleBaseFileName</code> if they
 * are using this method to retrieve the correct String for the specified locale.
 * The locale used will be by default the locale of the UI Engine.
 * If subclasses want to override the locale used they can implement 
 * the method <code>getResourceLocale</code>
 *
 * @param key <code>String</code> lookup key.
 * @return <code>String</code>
 * @see getResourceBundleBaseFileName
 * @see getResourceLocale
 */
public String getResourceString(String key) {
	return (String) getResourceObject(key);
}
/**
 * return the string to indicate that this Server has started.
 */
public String getStartedString() {
	return getResourceString("LblUIEngineStarted");
}
/**
 * Send the available fonts to the ULC side which
 * waits/blocks for the answer.
 *
 * @since R3.1
 */
protected void getUIAvailableFontFamilyNames(ORBConnection conn) {
	Anything a= new Anything(
						GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames());
	conn.send(Common.FACTORY_ID, "getUIAvailableFontFamilyNames", a);
}
protected Locale getUILocale() {
	if (fUILocale == null) {
		fUILocale = Locale.getDefault();
		fgUILocale= fUILocale; // backward compatibility
	}
	return fUILocale;
}
protected Anything getUILocaleAnything() {
	Anything locale= new Anything();
	Locale l = getUILocale();
	locale.put("country", l.getCountry());
	locale.put("lang", l.getLanguage());
	locale.put("variant", l.getVariant());
	return locale;
}
/**
 * Send the screen resolution to the ULC side which
 * waits/blocks for the answer.
 *
 * @since R3.1
 */
protected void getUIScreenResolution(ORBConnection conn) {
	conn.send(Common.FACTORY_ID, "getUIScreenResolution", new Anything(Toolkit.getDefaultToolkit().getScreenResolution()));
}
/**
 * Send the screen size to the ULC side which
 * waits/blocks for the answer.
 *
 * @since R3.1
 */
protected void getUIScreenSize(ORBConnection conn) {
	Anything a= new Anything();
	Dimension size= Toolkit.getDefaultToolkit().getScreenSize();
	a.put("w", size.width);
	a.put("h", size.height);
	conn.send(Common.FACTORY_ID, "getUIScreenSize", a);
}
/**
 * Send the requested system properties to the ULC side which
 * waits/blocks for the answer.
 * SecurityException is mapped to a <code>null</code> value
 *
 * @since R3.1
 */
protected void getUISystemProperties(ORBConnection conn, Anything args) {
	Anything a= new Anything();
	for (Enumeration enum= args.values(); enum.hasMoreElements();) {
		String value;
		String key;
		key= ((Anything) enum.nextElement()).toString();
		try {
			value= System.getProperty(key);}
		catch (SecurityException e) {
			value= null;
		}
		if (value == null)
			a.append(new Anything());
		else
			a.append(new Anything(value));
	}
	conn.send(Common.FACTORY_ID, "getUISystemProperty", a);
}
/**
 * The ULC application has sent a collection of requests as a batched request to this object.
 * Do all processing necessary.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param args		Anything		The collection of requests arguments associated with this request.
 */
public void handleBatchedRequest(ORBConnection connection, Anything args) {
	for (int i = 0; i < args.size(); i++) {
		final Anything request = args.get(i);
		final String req = request.get("req", null);
		final int id = request.get("id", 0);
		final ORBConnection conn = connection;
		final ICallable c = conn.find(id);
		if (c != null && req != null) {
			Request r = new Request() {
				public void dispatch() {
					Anything a = request.get("a");
					c.handleRequest(conn, req, a);
				};
			};
			connection.postRequest(r);
		}
	}
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request write a message out on the console.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	String nullString= null;

	if (request.equals("connectToApp")) {
		String url = args.get("URL", "unknown");
		String clientData = args.get("clientData", "");
		callApp(url, clientData, reconnectDelayInMillis());
		return;
	}
	if (request.equals("disconnectFromApp")) {
		String url = args.get("URL", "unknown");
		String connId = args.get("connectionId", "*");
		killApp(url, connId);
		return;
	}
	if (request.equals("addUIConnectionListener")) {
		addUIConnectionListener(conn);
		return;
	}
	if (request.equals("new")) {
		newInstance(conn, args);
		return;
	}
	if (request.equals("freeAll")) {
		conn.disposeRegistry();
		return;
	}
	if (request.equals("echo")) {
		conn.send(Common.FACTORY_ID, request, args);
		return;
	}
	if (request.equals("beep")) {
		beep();
		return;
	}
	if (request.equals("kill")) {
		System.exit(1);
		return;
	}
	if (request.equals("synch")) {
		conn.send(Common.FACTORY_ID, request, args);
		return;
	}
	if (request.equals("init")) {
		if (isServerMode()) {
			setContextId(conn, args);
			conn.setConnDownErrorMsg(args.get("connDownErrorMsg", null));
			Anything a = new Anything();			
			if (Common.checkMajorVersion(args)) { // a major mismatch
				a.put("reason", "ULC protocol major version mismatch");
				conn.send(Common.FACTORY_ID, "reject", a); // let application kill itself
			}
			else if (Common.checkMinorVersion(args)) { // a minor mismatch
				a.put("reason", "ULC protocol minor version mismatch");
				conn.send(Common.FACTORY_ID, "accept", a); 
			}
		}
		else
			trouble("handleRequest", "init received while in client mode");
		return;
	}
	if (request.equals("reject")) {
		if (!isServerMode()) { // we are the initiator
			String reason = args.get("reason", "unknown");
			System.out.println("Problem between UI Engine and application server. Reason: " + reason + "; giving up!");
			conn.close();//System.exit(-1); // give up
		}
		return;
	}
	if (request.equals("accept")) {
		setContextId(conn, args);
		if (!isServerMode())
			conn.setConnDownErrorMsg(args.get("connDownErrorMsg", null));		
		String reason = args.get("reason", null);
		if (reason != null)
			System.out.println("Minor problem between UI Engine and application server. Reason: " + reason);
		return;
	}
	if (request.equals("setLook")) {
		setLook(args.toString());
		return;
	}
	if (request.equals("getLook")) {
		getLook(conn);
		return;
	}
	if (request.equals("getUISystemProperty")) {
		getUISystemProperties(conn, args);
		return;
	}
	if (request.equals("getInstalledLookAndFeels")) {
		getInstalledLookAndFeels(conn);
		return;
	}
	if (request.equals("getUILocale")) {
		Anything locale = new Anything();
		locale.put("locale", getUILocaleAnything());
		conn.send(Common.FACTORY_ID, "getUILocale", locale);
		return;
	}
	if (request.equals("getUIScreenResolution")) {
		getUIScreenResolution(conn);
		return;
	}
	if (request.equals("getUIScreenSize")) {
		getUIScreenSize(conn);
		return;
	}
	if (request.equals("getUIAvailableFontFamilyNames")) {
		getUIAvailableFontFamilyNames(conn);
		return;
	}
	if (request.equals("getUserParameter")) {
		Anything parameter = new Anything(fUserParameter);
		conn.send(Common.FACTORY_ID, "getUserParameter", parameter);
		return;
	}
	if (request.equals("getReconnectToken")) {
		fServerInClientReconnectMode= true;
		getReconnectToken(conn, args.get("challenge", nullString), args.get("reconnectTokenStrategy", ""));
		return;
	}
	if (request.equals("enableLog")) {
		setLoggingEnabled(args.asBoolean(false));
		return;
	}
	if (request.equals("handleBundledRequests")) {
		handleBatchedRequest(conn, args);
		return;
	}
	if (request.equals("setConnDownErrorMsg")) {
		conn.setConnDownErrorMsg(args.asString(nullString));
		return;
	}
	trouble("handleRequest", "unknown request: " + request);
}
UIProxy internalNewInstance(String type) {
	try {
		Class cl = Class.forName(type);
		if (cl != null)
			return (UIProxy) cl.newInstance();
		trouble2("UI.internalNewInstance", "couldn't find " + type);
	} catch (ClassNotFoundException e) {
		// this is ok...
	} catch (InstantiationException e) {
		trouble2("UI.internalNewInstance", "got InstantiationException");
	} catch (IllegalAccessException e) {
		trouble2("UI.internalNewInstance", "got IllegalAccessException");
	} catch (ClassCastException e) {
		trouble2("UI.internalNewInstance", "can't cast object to Managed");
	}
	return null;
}
/**
  * Is the UI running as Applet
  */
public boolean isApplet() {
	return fIsApplet;
}
/**
 * Return the reconnect mode. If <code>true</code> then the UI will retry the connection
 * to the ULC server if the connection ever goes down.
 *
 * @return	mode 		If <code>true</code> retry the failed connection.
 * @since	R3.1
 */
public boolean isAutoReconnectOn() {
	return fAutoReconnect && !isServerMode();
}
/**
 * Return the reconnect mode of the ULC server. If <code>true</code> then the ULC
 * ULC server allows client reconnect mode.
 *
 * @return	mode 		If <code>true</code> ULC server is in client reconnect mode.
 * @since	R3.1
 */
public boolean isServerInClientReconnectMode() {
	return fServerInClientReconnectMode && !isServerMode();
}
/**
 * If true then the UI Engine is waiting for connections from the Application.
 * This flag indicates the 'mode' in which the receiver is executing. 
 * In the 'client' mode, the UI-engine connects to a running ULC application.
 * In the 'server' mode the UI Engine is waiting for applications to connect to it.
 */
public  boolean isServerMode() {
	return fServerMode;
}
void killApp(String clientUrl, String connId) {
	ORBConnection conn = (ORBConnection) fConnections.get(connId);
	if (conn != null) {
		// This is the clean way to do it
		conn.send(Common.FACTORY_ID, "terminate", new Anything(conn.getContextId()));
		conn.postRequest(new CloseRequest(conn, null));
	}
	else {
		ConnectionInProgress connInProg = (ConnectionInProgress) fConnectionsInProgress.get(connId);
		if (connInProg == null) { // not found (client did not have connection id yet)
			if (connId.equals("*")) {
				// search through table based on url in this case
				for (Enumeration keys = fConnectionsInProgress.keys(); keys.hasMoreElements();) {
					String key = (String) keys.nextElement();
					ConnectionInProgress cip = (ConnectionInProgress) fConnectionsInProgress.get(key);
					if (cip.getUrlString().equals(clientUrl)) {
						connInProg = cip;
						connId = key;
						// let thread finish
						connInProg.stopMe();			
						try {
							connInProg.join();
						} catch (InterruptedException e) {
							System.out.println("WARNING: UI.killApp(String, String): Thread interrupted during join");
							e.printStackTrace();
						}
						// remove entry from list of connections in progress
						fConnectionsInProgress.remove(connId);	
					}
				}
			}
		}
	}
}
public void logPrint(String logEntry) {
	if (fLogEnabled) {
		ILogStream log = getLogStream();
		if (log != null)
			log.logPrint(logEntry);
	}
}
public void logPrintln(String logEntry) {
	if (fLogEnabled) {
		ILogStream log = getLogStream();
		if (log != null)
			log.logPrintln(logEntry);
	}
}
/**
 * This method is called first whenever the UIEngine is started as an application.
 *
 */
public static void main(String[] args) {
	String vers = System.getProperty("java.version");
	if (vers.compareTo(MIN_JVM_VERSION) < 0) {
		System.out.println("ERROR: UI Engine must be run with a " + MIN_JVM_VERSION + " or higher VM version!");
		return;
	}

	String[] newargs;
	//
	// assume that if there is only one argument and the argument does not
	// start with - it is a file containing the arguments.
	// or
	// if the ui was started using ulcui.exe if the argument count=2 
	// and the first argument is -wincon and the second  arg does not start with - 
	// then second argument is a fileName containing the arguments.
	//
	// Ensure that the ulcui.exe and UIApplet are kept in sync with any changes here
	//
	if (((args.length == 1) && (!args[0].startsWith("-"))) || ((args.length == 2) && ((args[0].equals("-wincon")) && (!args[1].startsWith("-"))))) {
		if (args.length == 2) {
			newargs = parseArgumentsFromFile(args[1], true);
			if (args[0].equals("-wincon")) {
				String[] tempArgs = new String[newargs.length + 1];
				for (int i = 0; i < newargs.length; i++)
					tempArgs[i] = newargs[i];
				tempArgs[tempArgs.length - 1] = args[0];
				newargs = tempArgs;
			}
		} else
			newargs = parseArgumentsFromFile(args[0], true);
	} else { // process packed arguments if any
		newargs = parsePackedArgs(args);
	}
	boolean useAwtProcessor = false;
	for (int i = 0; i < newargs.length; i++) {
		if (newargs[i].equals("-useAwtProcessor")) {
			useAwtProcessor = true;
			break;
		}
	}
	if (useAwtProcessor) {
		UI ui = new UI(newargs, new AWTRequestProcessor());
		ui.establishConnection();
	} else {
		UI ui = new UI(newargs, new SwingRequestProcessor());
		ui.establishConnection();
	}
}
/**
 * Defines how ULC managed objects are found.
 */
UIProxy newInstance(ORBConnection conn, Anything args) {
	UIProxy m = null;
	String type = args.get("type", null);
	if (type == null) {
		trouble("newInstance", "no type information received cannot create instance");
		return defaultWidgetFor(null, args);
	}
	String className = getClassNameFor(type);
	m = internalNewInstance(className);
	if (m == null) {
		trouble("newInstance", "couldn't create instance of " + className);
		m = defaultWidgetFor(type, args);
	} else {
		m.setConnectionAndId(conn, args);
		m.restoreState(conn, args);
	}
	return m;
}
synchronized String nextConnectionId() {
	fgConnectionId++;
	return Long.toString(fgConnectionId);
}
/**
 * Return a collection of arguments that are read from the first line in the file
 * specified by fileName.
 *
 */
protected static String[] parseArgumentsFromFile(String fileName, boolean debug) {
	String[] newargs = new String[0];
	try {
		BufferedReader reader = new BufferedReader(new FileReader(fileName));
		String clientUrl = reader.readLine();
		reader.close();
		reader = null;
		StringTokenizer st = new StringTokenizer(clientUrl, " ");
		int packlen = st.countTokens();
		newargs = new String[packlen];
		int newind = 0;
		while (st.hasMoreTokens()) {
			String token = st.nextToken();
			if (token.equals("-indirectUrl") && (newind < packlen)) {
				{
					clientUrl = parseIndirectArguments(st.nextToken(), debug);
					if (clientUrl != null) {
						newargs[newind++] = "-url";
						newargs[newind++] = clientUrl;
					}
				}
			} else
				newargs[newind++] = token;
		}
	} catch (IOException e2) {
		System.out.println("cannot open stream in parseArgumentsFromFile " + e2);
	}
	return newargs;
}
/**
 * Retrieve the specified URL and return its result as an argument string to be passed to the UI.
 *
 */
protected static String parseIndirectArguments(String stringUrl, boolean debug) {
	boolean argsOK = true;
	String clientUrl = null;
	try {
		URL indirUrl = new URL(stringUrl);
		try {
			BufferedReader reader = new BufferedReader(new InputStreamReader(indirUrl.openStream()));
			clientUrl = reader.readLine();
			if (debug)
				System.out.println("indirectUrl from " + stringUrl + " is: " + clientUrl);
			reader.close();
			reader = null;
		} catch (IOException e2) {
			System.out.println("cannot open stream on URL (" + stringUrl + ") for option -indirectUrl. " + e2);
			argsOK = false;
		}
	} catch (MalformedURLException e) {
		System.out.println("invalid URL for option -indirectUrl " + e);
		argsOK = false;
	}
	if (argsOK)
		return clientUrl;
	else
		return null;
}
/**
 * Parse the specified arguments that are in the packedargs format.
 *
 */
protected static String[] parsePackedArgs(String[] args) {
	String[] newargs = args;
	for (int i = 0; i < args.length; i++) {
		String arg = args[i];
		if (arg.equals("-packedargs")) {
			i++;
			if (i < args.length) {
				StringTokenizer st = new StringTokenizer(args[i], " ");
				int packlen = st.countTokens();
				newargs = new String[args.length - 2 + packlen];
				int newind = 0;
				for (int j = 0; j < args.length; j++) {
					if (args[j].equals("-packedargs")) {
						j++;
						continue;
					}
					newargs[newind++] = args[j];
				}
				while (st.hasMoreTokens()) {
					newargs[newind++] = st.nextToken();
				}
				args = newargs;
			} else {
				System.out.println("expecting value arg for option -packedargs");
			}
			break;
		}
	}
	return newargs;
}
public static void printStatistics(String msg) {
	Runtime.getRuntime().gc();
	UIProxy.printInstances(msg);
	DebugThread.printInstances();
}
public void printULCVersionAndProtocolVersion() {
	System.out.println(getResourceString("LblULCVersion") + Common.getUlcVersion() + getResourceString("LblProtocolVersion") + Common.getVersion());
}
void randomDelay(int minSeconds, int maxSeconds) {
	int range = maxSeconds - minSeconds;
	int s = (int) (Math.random() * range) + minSeconds;
	try {
		Thread.sleep(s * 1000);
	} catch (InterruptedException e) {
	}
}
public synchronized void receivedRequest(UlcConnection conn, Request request) {
	logPrintln(Common.convertORBRequestToStringForLog("Recv: ", (ORBRequest) request, null));
}
protected boolean reconnectAfterException(IConnection conn, UlcTransportException ex) {
	return getReconnectAfterException();
}
protected long reconnectDelayInMillis() {
	return 2000;
}
protected void retryConnectionUsing(ConnectionInProgress connInProgress) {
	connInProgress.retryConnection(true);
}
public void runApp(String clientUrl) {
	while (true) {
		String connId = callApp(clientUrl, "");
		randomDelay(40, 60);
		killApp(clientUrl, connId);
		randomDelay(5, 30);
	}
}
protected void sendInit(ORBConnection conn) {
	String connId = conn.getId();
	String clientData = conn.getClientData();
	Anything info = new Anything();
	info.put("majorVersion", Common.PROTOCOL_VERSION_MAJOR);
	info.put("minorVersion", Common.PROTOCOL_VERSION_MINOR);
	String appName = conn.getApplicationName();
	String peerLocation = conn.getLocalUrlString();
	if ((peerLocation != null) && (peerLocation.length() > 0))
		info.put("peerLocation", peerLocation);
	if (appName != null) {
		info.put("appName", appName);
		info.put("connectionId", connId);
		info.put("clientData", clientData);
	}
	info.put("initUrl", conn.getUrlString());
	info.put("locale", getUILocaleAnything());
	conn.send(Common.FACTORY_ID, "init", info);
}
public synchronized void sentRequest(UlcConnection conn, Request request) {
	logPrintln(Common.convertORBRequestToStringForLog("Sent: ", (ORBRequest) request, null));
}
/**
 * Set the reconnect mode. If <code>true</code> then the UI will retry the connection
 * to the ULC server if the connection ever goes down.
 *
 * @param	mode 		If <code>true</code> retry the failed connection
 * @since	R3.1
 */
public void setAutoReconnect(boolean mode) {
	fAutoReconnect= mode;
}
/**
 * Set the unique contextId in the connection.
 * This contextId once set will be sent on all requests from the UI Engine 
 * to the application.
 * It is the applications responsibility to generate this contextId
 *
 * @param conn 	The connection in which to set the contextId.
 * @param args 	The arguments in which the contextId is specified.
 */
protected void setContextId(ORBConnection conn, Anything args) {
	long contextId = args.get("contextId", -1L);
	if (contextId != -1)
		conn.setContextId(contextId);
}
/**
  * Set to true if the UI should exit when the last connection is closed.
  * If set to false the UI will remain active even after the last connection 
  * is closed and an explicit System.exit need to be called.
  *
  * @param exitOnLastConnectionClosedFlag If true UI will exit on last connection closed.
  * @see getExitOnLastConnectionClosedFlag
  */
public static void setExitOnLastConnectionClosedFlag(boolean exitOnLastConnectionClosedFlag) {
	fgExitOnLastConnectionClose = exitOnLastConnectionClosedFlag;
}
public void setLoggingEnabled(boolean enabled) {
	fLogEnabled = enabled;
}
void setLook(String lookClass) {
	fLookAndFeelClassName = lookClass;
	if (fLookAndFeel != null) {
		Enumeration conns = getActiveConnections().elements();
		while (conns.hasMoreElements()) {
			try {
				ORBConnection conn = (ORBConnection) conns.nextElement();
				fLookAndFeel.setLook(conn, lookClass);
			} catch (ClassCastException e) {
			}
		}
	}
}
public static void setReconnectAfterException(boolean reconnectOnException) {
	fgReconnectOnException = reconnectOnException;
}
/**
 * If true then the UI Engine is waiting for connections from the Application.
 * This flag indicates the 'mode' in which the receiver is executing. 
 * In the 'client' mode, the UI-engine connects to a running ULC application.
 * In the 'server' mode the UI Engine is waiting for applications to connect to it.
 */
public void setServerMode(boolean serverMode) {
	fServerMode = serverMode;
}
protected void setUILocale(Locale currentLocale) {
	fUILocale= currentLocale;
	fgUILocale= currentLocale; // backward compatibility
}
/**
  * This instance of the UI has no connections active. Do any cleanUp necessary.
  */
protected void shutDown() {
	if (fDebug)
		System.out.println("<UI.shutDown()");

	if (!isServerMode() && !fTest) {
		if ((!fIsApplet) && getExitOnLastConnectionClosedFlag()) {
			System.exit(0);
		}
	}
	if (fIsApplet) {
		Server.removeConnectionController(this);
		free();		
		if (fApplet != null) {
			if (fDebug)
				System.out.println("UI.shutDown(): start releasing lock...");
			synchronized (fApplet) {
				fApplet.notify();
			}
			if (fDebug)
				System.out.println("UI.shutDown(): lock released");			
			fApplet= null;
		}
	}
	if (fDebug)
		System.out.println(">UI.shutDown()");
}
void startMemoryWatcher() {
	(new MemoryWatcher()).start();
}
/**
 * If you want to change the shutDown behavior
 * you can overwrite shutDown()
 */
protected void startShutDownThread() {
	if (fDebug)
		System.out.println("<UI.startShutDownThread()");
	new Thread() {
		public void run() {
			if (fDebug)
				System.out.println("<UI.startShutDownThread().run()");
			// wait for error msg views to be closed
			while (fErrorMessageViews.size() > 0) {
				updateErrMsgViewCollection();
				try {Thread.currentThread().sleep(200);
				} catch (InterruptedException e) {
				}
			}
			shutDown();
			if (fDebug)
				System.out.println(">UI.startShutDownThread().run()");
		}
	}.start();
	if (fDebug)
		System.out.println(">UI.startShutDownThread()");	
}
public void stopServer() {
	cleanup();
	if (isServerMode()) {
		if (getTransportServer() != null)
			try {
			getTransportServer().stop();
		} catch (Exception e) {
		}
	}
}
protected void updateErrMsgViewCollection() {
	if (fErrorMessageViews.isEmpty())
		return;

	for (Enumeration enum= ((Vector) fErrorMessageViews.clone()).elements(); enum.hasMoreElements();) {
		JDialog elem= (JDialog) enum.nextElement();
		if (!elem.isVisible())
			fErrorMessageViews.remove(elem);
	};
}
}
