package com.ibm.ulc.ui;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.awt.*;
import java.net.*;
import java.awt.event.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.*;
import com.ibm.ulc.ui.base.*;
import com.ibm.ulc.ui.lists.*;
import com.ibm.ulc.ui.dataTypes.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.Border;
/**
 * This class is the root of the hierarchy for all visible UI-side widgets.
 * Subclasses typically implement wrappers for the various Swing widgets, and
 * also provide the capability to communicate to and from the application side.
 * In addition, this class provides functionality that is common to all widgets,
 * such as colouring, mouse-handling, enabling etc.
 */
public abstract class UIComponent extends UIProxy implements IEnableListenerTarget, MouseListener {
	protected UIMenu fPopupMenu;
	protected IEnableListener fEnabler = null;
	/**
	 * The partial URL for the help page for this widget
	 * This Url is appended to the HelpRoot URL of the UI and then displayed.
	 */
	protected String fHelpUrl = null;
/**
 * Add the  <code>UIComponent</code> to the receiver's list of children.
 *
 * @param component : A <code>UIComponent</code>
 */
public void add(UIComponent component) {
	Component comp = component.getComponent();
	if (comp != null)
		add(comp);
}
/**
 * Add the  <code>UIProxy</code> to the receiver's list of children.
 *
 * @param proxy : A <code>UIProxy</code>
 */
public void add(UIProxy proxy) {
	if (proxy instanceof UIComponent)
		add((UIComponent) proxy);
}
/**
 * Add the  <code>Component</code> to the receiver's list of children.
 * Subclasses need to hold and manage their children by themselves,
 * if required.
 *
 * @param component : A <code>Component</code>
 */
public void add(Component component) {
	trouble("add(Component)", "abstract method");
}
protected void addMouseListenerForPopupMenu() {
	Component c = this.getBasicComponent();
	if (c != null)
		c.addMouseListener(this);
}
/**
 * answer wether all of the color values are valid RGB values
 *
 * @param red  int the red value of an RGB color
 * @param green int the green value of an RGB color
 * @param blue  int the blue value of an RGB color
 */
protected boolean areValidRGBColorValues(int red, int green, int blue) {
	if (red < 0 || red > 255) {
		return false;
	}
	if (green < 0 || green > 255) {
		return false;
	}
	if (blue < 0 || blue > 255) {
		return false;
	}
	return true;
}
/**
 * Configure the UIColumn to use this widget as a renderer.
 * Subclasses must override this method to support being used as a renderer in a Table Column. 
 *
 * @see UIColumn#restoreState
 */
public void configureAsColumnRenderer(UIColumn column, boolean editable, IDataType dataType) {
	trouble("configureAsColumnRenderer", "doesn't support a " + getComponent().getClass().getName() + " renderer");
}
protected void forceFramePack() {
	Component p = getBasicComponent();
	Component parent = p;
	while (p != null) {
		parent = p;
		p = parent.getParent();
	}
	if (parent != null) {
		if (parent instanceof JFrame) {
			final JFrame f = (JFrame) parent;
			DeferredRequest r = new DeferredRequest(fConnection) {
				public void safeDispatch() {
					f.invalidate();
					f.pack();
				}
			};
			fConnection.postRequest(r);
		}
	}
}
protected void forceFrameRepaint() {
	Component p = getBasicComponent();
	Component parent = p;
	while (p != null) {
		parent = p;
		p = parent.getParent();
	}
	if (parent != null) {
		if (parent instanceof JFrame) {
			final JFrame f = (JFrame) parent;
			final Dimension size = f.getSize();
			f.setSize(f.getSize().width - 1, f.getSize().height - 1);
			DeferredRequest r = new DeferredRequest(fConnection) {
				public void safeDispatch() {
					f.setSize(size);
				}
			};
			fConnection.postRequest(r);
		}
	}
}
public void forceParentRepaint() {
	DeferredRequest r = new DeferredRequest(fConnection) {
		public void safeDispatch() {
			Component c = getComponent();
			if (c == null)
				c = getBasicComponent();
			Component parent = c;
			if (c != null)
				parent = c.getParent();
			if (parent != null) {
				parent.invalidate();
				parent.validate();
				parent.repaint();
			}
		}
	};
	fConnection.postRequest(r);
}
/**
 * Return the background of the receiver
 */
public Color getBackgroundColor() {
	Component c=getBasicComponent();
	
	if (c != null) {
		return c.getBackground();
	}
	else return null;
}
/**
 * Return the Component whose background can be set to show the mandatory property.
 *
 * @return java.awt.Component
 */
protected Component getBackgroundComponent() {
	return getComponent();
}
/**
 * Answer the primary swing component (widget) being
 * wrapped by the receiver.
 *
 * @see getComponent()
 */
public Component getBasicComponent() {
	return getComponent();
}
/**
 * Return the Component whose border can be set to show the mandatory property.
 *
 * @return java.awt.Component
 */
protected Component getBorderComponent() {
	return getComponent();
}
/**
 * Answer the component being handled by the receiver.
 *
 * This method returns the (swing) widget associated with the receiver, 
 * <i>except</i> in cases where there are containers within which the 
 * basic widget is located. In such a case, this method answers the 
 * container, while the basic widget itself can be accessed using the 
 * getBasicComponent() method.
 *
 * @see #getBasicComponent()
 */
abstract public Component getComponent();
/**
 * Return the foreground of the receiver
 */
public Color getForegroundColor() {
	Component c=getBasicComponent();
	
	if (c != null) {
		return c.getForeground();
	}
	else return null;
}
/**
 * Return the full Help Url for this component
 * 
 */
protected URL getHelpURL(UIHelpBrowserContext helpContext) {
	if (helpContext == null)
		return null;
	String helpUrlSuffix = getPartialHelpUrl();
	if (helpUrlSuffix == null)
		helpUrlSuffix = "";
	String helpRootUrl = helpContext.getHelpUrl();
	URL u = null;
	if (helpRootUrl != null) {
		try {
			u = new URL(helpRootUrl + helpUrlSuffix);
		} catch (Exception e) {
		}
	} else {
		try {
			u = new URL(helpUrlSuffix);
		} catch (Exception e) {
		}
	}
	return u;
}
/**
 * Return the partial help url for this component or null if not defined.
 */
public String getPartialHelpUrl() {
	return fHelpUrl;
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("setEnabled")) {
		setEnabled(args.asBoolean(true));
		return;
	}
	if (request.equals("setLabel")) {
		setLabel(args.asString("<no label>"));
		return;
	}
	if (request.equals("setToolTipText")) {
		if (args.isNull())
			setToolTipText(null);
		else
			setToolTipText(args.asString(""));
		return;
	}
	if (request.equals("setEnabler")) {
		setEnabler(conn, args);
		return;
	}
	if (request.equals("requestFocus")) {
		requestFocus();
		return;
	}
	if (request.equals("setCursor")) {
		setCursor(args.asInt(0));
		return;
	}
	if (request.equals("setPopupMenu")) {
		setPopupMenu(conn, args);
		return;
	}
	if (request.equals("setForeground")) {
		setForegroundColor(args.get("r", -1), args.get("g", -1), args.get("b", -1), true);
		return;
	}
	if (request.equals("setBackground")) {
		setBackgroundColor(args.get("r", -1), args.get("g", -1), args.get("b", -1), true);
		return;
	}
	if (request.equals("setDecoration")) {
		setDecoration(args, true);
		return;
	}
	if (request.equals("setFont")) {
		if (args.isNull())
			setFont(null);
		else
			setFont((UIFont) getManaged(UIFont.class, conn, args));
		return;
	}
	if (request.equals("setVisible")) {
		setVisible(args.asBoolean(true));
		return;
	}
	if (request.equals("insert")) {
		insert(conn, args);
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Inserts a child widget at the specified index.
 * Subclasses will typically override the <code>insert(UIProxy, int)</code> method
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> optional arguments 
 */
public void insert(ORBConnection conn, Anything args) {
	UIComponent m = (UIComponent)getManaged(UIComponent.class, conn, args.get("c"));
	if (m != null)
		insert(m, args.get("pos", -1));
}
/**
 * Inserts a child widget at the specified index.
 * Subclasses that support inserting child widgets must override this method
 *
 * @param managed 		the UIComponent widget that is being added as a child
 * @param index 		the Index at which the widget should be added
 */
public void insert(UIComponent managed, int index) {
	trouble("insert(UIProxy, int)", "abstract method");
}
protected void installHelpHandler() {
	Component c = getBasicComponent();
	if (c instanceof JComponent) {
		installKeyboardActions((JComponent)c);
	}
}
protected void installKeyboardActions(JComponent c) {

	// F1 press for Help
	c.registerKeyboardAction(new AbstractAction() {
		public void actionPerformed(ActionEvent e) {
			requestHelp();
		}
		public boolean isEnabled() {
			return true;
		}
	}
	, KeyStroke.getKeyStroke(KeyEvent.VK_F1, 0), JComponent.WHEN_IN_FOCUSED_WINDOW);

	// Request focus if it isn't set.
	if (!c.requestDefaultFocus()) {
		c.requestFocus();
	}
}
protected UiLabelAndMnemonic internalParseLabelForMnemonic(String labelString) {
	return new UiLabelAndMnemonic(labelString);
}
/**
 * Return true if my component is enabled.
 */
public boolean isEnabled() {
	Component bc = getBasicComponent();
	if (bc != null)
		return bc.isEnabled();
	return false;
}
/**
 * Answer boolean whether the receiver is currently
 * being displayed.
 */
public boolean isShowing() {
	return false;
}
/**
 * Invoked when the mouse has been clicked on a component.
 */
public void mouseClicked(MouseEvent event) {
	//System.out.println(" Mouse Clicked: " + e.toString());
}
/**
 * Invoked when the mouse enters a component.
 */
public void mouseEntered(MouseEvent event) {
	//System.out.println(" Mouse Entered: " + e.toString());
}
/**
 * Invoked when the mouse exits a component.
 */
public void mouseExited(MouseEvent event) {
	//System.out.println(" Mouse Exited: " + e.toString());
}
/**
 * Invoked when a mouse button has been pressed on a component.
 */
public void mousePressed(MouseEvent event) {
	// System.out.println(" Mouse Pressed: " + e.toString());
}
/**
 * Invoked when a mouse button has been released on a component.
 */
public void mouseReleased(MouseEvent event) {
	if (event.isPopupTrigger()) showPopup(event);
}
/**
 * refresh the receiver's UI to reflect the newly defined colors. 
 * By defaultthe receiver sends itself the <code>refreshComponentColor()</code> with its
 * colorComponent as parameter. Subclasses should not reimplement this method. If
 * for some reason they need to refresh a component other than that returned by
 * the <code>getColorComponent()</code> method, they should send the <code>
 * refreshComponentColor(JComponent)</code> with the appropriate component instead.
 * <br>This method is intended as a convenience method only
 *
 */
protected void refreshComponentColor() {
	refreshComponentColor((JComponent) getBasicComponent());
}
/**
 * refresh <code>component</code> to make sure the new color setting is reflected on the UI.
 * ignore this message if <code>component</code> is null
 *
 * @parm component javax.swing.JComponent the JComponent that needs to be repainted to show the colors
 */
protected void refreshComponentColor(JComponent component) {
	if (component != null) {
		component.revalidate();
		component.repaint();
	}
}
/**
 * Remove the  <code>UIComponent</code> from the receiver's list of children.
 *
 * @param component : A <code>UIComponent</code>
 */
public void remove(UIComponent component) {
	Component comp = component.getComponent();
	if (comp != null)
		remove(comp);
}
/**
 * Remove the  <code>UIProxy</code> from the receiver's list of children.
 *
 * @param proxy : A <code>UIProxy</code>
 */
public void remove(UIProxy proxy) {
	if (proxy instanceof UIComponent)
		remove((UIComponent) proxy);
}
/**
 * Remove the  <code>Component</code> from the receiver's list of children.
 * Subclasses need to hold and manage their children by themselves,
 * if required.
 *
 * @param component : A <code>Component</code>
 */
public void remove(Component component) {
	trouble("remove(Component)", "abstract method");
}
protected void removeMouseListenerForPopupMenu() {
	Component c = this.getBasicComponent();
	if (c != null)
		c.removeMouseListener(this);
}
/**
 * Request that the focus come onto the receiver.
 */
public void requestFocus() {
	Component c = getComponent();
	if (c != null) {
		c.requestFocus();
	}
}
protected void requestHelp() {
	UIApplication a = getApplication();
	if (a != null)
		a.displayHelp(this);
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	if (!args.get("enabled", true))
		setEnabled(false);
	if (!args.get("visible", true))
		setVisible(false);
	if (args.isDefined("tooltip"))
		setToolTipText(args.get("tooltip", "<no tool tip info>"));
	if (args.isDefined("cursor"))
		setCursor(args.get("cursor", 0));
	if (args.isDefined("hasFocus"))
		requestFocus();
	super.restoreState(conn, args);
	final Anything a = args;
	final ORBConnection c = conn;
	// we do this twice since some widgets do not have their peer built at restore state time.
	// hence we do it using a deferred request  once again.
	setDecoration(a, false);
	UIFont font = (UIFont) getManaged(UIFont.class, c, a.get("font"));
	if (font != null)
		setFont(font);
	DeferredRequest r = new DeferredRequest(conn) {
		public void safeDispatch() {
			setDecoration(a, false);
			UIFont f = (UIFont) getManaged(UIFont.class, c, a.get("font"));
			if (f != null)
				setFont(f);
		}
	};
	conn.postRequest(r);
	setPopupMenu(conn, args.get("popupMenu"));
	if (args.isDefined("enabler"))
		setEnabler(conn, args.get("enabler"));
	if (args.isDefined("helpUrl"))
		fHelpUrl = args.get("helpUrl").asString(null);
	installHelpHandler();
}
/**
 * Set the background of the receiver to the color defined
 * by the RGB int values <code>red, green, and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */
public void setBackgroundColor(int red, int green, int blue, boolean refresh) {
	Color color = null;
	JComponent c = (JComponent) getBasicComponent();
	if (c != null) {
		if (red == -1 || green == -1 || blue == -1) {
		}
		else {
			color = new Color(red, green, blue);
		}
		c.setBackground(color);
		if (refresh)
			refreshComponentColor(c);
	}
}
/**
 * Set the background of the receiver to the color defined
 * by the RGB int values <code>red, green, and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * <br>Since the receiver of this message normally has its
 * opaque flag set to false (meaning that it does not paint
 * its background itself), this method sets that flag depending
 * on whether the value is set ecplicitly or reverts to the LAF default
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */
protected void setBackgroundColorWithOpaque(int red, int green, int blue, boolean refresh) {
	Color color = null;
	JComponent c = (JComponent) getBasicComponent();
	if (c != null) {
		if (red == -1 || green == -1 || blue == -1) {
			c.setOpaque(false);
		}
		else {
			c.setOpaque(true);
			color = new Color(red, green, blue);
		}
		c.setBackground(color);
		if (refresh)
			refreshComponentColor(c);
	}
}
/**
 * Set the cursor prescribed by the given id.
 * For valid values of id,
 * @see java.awt.Cursor.
 */
public void setCursor(int id) {
	Component c = getBasicComponent();
	if (c != null)
		c.setCursor(Cursor.getPredefinedCursor(id));
}
/**
 * Set the color and font decorations of the receiver to the colors / fonts defined
 * as Anythings in <code>args</code>.
 * <br>make sure the receiver's UI is updated only once per request
 * 
 * 
 * @param conn ORGConnection this parameter is ignored
 * @param args Anything holds the anything specifications of all decorations
 */
public final void setDecoration(ORBConnection conn, Anything args) {
	setDecoration(args, true);
}
/**
 * Set the color and font decorations of the receiver to the colors / fonts defined
 * as Anythings in <code>args</code>.
 * <br>By default, UIComponents support fore- and background colors, and a font.
 * subclasses with additional decorations must reimplement this method and handle them
 * specifically.
 * <br>update the receiver's UI exactly once if both <code>refresh</code> and any decoration
 * has been reset
 * <br>Answer true if the component color should be refreshed. If refresh is true, the answer
 * is invariably false, because the refresh is done if needed in this method
 * 
 * @param args Anything holds the anything specifications of all decorations
 * @param refresh boolean indicates whether the receiver should refresh its UI
 */
protected boolean setDecoration(Anything args, boolean refreshLocal) {
	boolean refreshNeeded = false;
	Anything f = args.get("fc");
	Anything b = args.get("bc");
	if (f != null) {
		refreshNeeded = true;
		setForegroundColor(f.get("r", -1), f.get("g", -1), f.get("b", -1), (b == null && refreshLocal));
	}
	if (b != null) {
		refreshNeeded = true;
		setBackgroundColor(b.get("r", -1), b.get("g", -1), b.get("b", -1), refreshLocal);
	}
	return (refreshNeeded && !refreshLocal);
}
/**
 * Set the receiver as enabled/disabled.
 */
public void setEnabled(boolean state) {
	Component c = getComponent();
	Component bc = getBasicComponent();
	if (c != null)
		c.setEnabled(state);
	if (bc != c)
		bc.setEnabled(state);
}
/**
 * Set the enabler object for the receiver, based on the
 * given arguments.
 */
public void setEnabler(ORBConnection conn, Anything args) {
	if (fEnabler != null) {
		fEnabler.removeEnableListener(this);
		fEnabler = null;
	}
	IEnableListener el = (IEnableListener) getManaged(IEnableListener.class, conn, args, false);
	fEnabler = el;
	if (el != null) {
		el.addEnableListener(this);
	}
}
public void setFont(UIFont font) {
	Component c = this.getBasicComponent();
	if (c != null) {
		if (font == null)
			c.setFont(null);
		else
			c.setFont(font.getFont());
		c.invalidate();
		c.repaint();
	}
}
/**
 * Set the foreground of the receiver to the color defined
 * by the RGB int values <code>red, green, and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */
public void setForegroundColor(int red, int green, int blue, boolean refresh) {
	Color color = null;
	JComponent c = (JComponent) getBasicComponent();
	if (c != null) {
		if (areValidRGBColorValues(red, green, blue)) {
			color = new Color(red, green, blue);
		}
		c.setForeground(color);
		if (refresh)
			refreshComponentColor(c);
	}
}
/**
 * Set the foreground of the receiver to the color defined
 * by the RGB int values <code>red, green, and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * <br>Since the receiver of this message has its
 * opaque flag set to false (meaning that it does not paint
 * its background itself), this method sets that flag depending
 * on whether the value is set explicitly or reverts to the LAF default
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */
public void setForegroundColorWithOpaque(int red, int green, int blue, boolean refresh) {
	Color color = null;
	JComponent c = (JComponent) getBasicComponent();
	if (c != null) {
		if (red == -1 || green == -1 || blue == -1) {
			c.setOpaque(false);
		}
		else {
			c.setOpaque(true);
			color = new Color(red, green, blue);
		}
		c.setForeground(color);
		if (refresh)
			refreshComponentColor(c);
	}
}
public void setLabel(String label) {
	trouble("setLabel", "abstract method");
}
public void setParent(UIProxy parent) {
}
protected void setPopupMenu(ORBConnection conn, Anything args) {
	if (args != null) {
		UIMenu uiMenu = null;
		if (!args.isNull()) {
			uiMenu = (UIMenu) getManaged(UIMenu.class, conn, args);
		}
		setPopupMenu(uiMenu);
	}
}
protected void setPopupMenu(UIMenu menu) {
	if (menu == null) {
		if (fPopupMenu != null)
			removeMouseListenerForPopupMenu();
		fPopupMenu = null;
	}
	else {
		boolean listenerNeeded = (fPopupMenu == null);
		fPopupMenu = menu;
		if (listenerNeeded) {
			addMouseListenerForPopupMenu();
		}
	}
}
public void setToolTipText(String text) {
	Component c = (Component) getBasicComponent();
	if ((c != null) && (c instanceof JComponent))
		((JComponent)c).setToolTipText(text);
}
/**
 * Set the component to be visible or hidden.
 * 
 */
public void setVisible(boolean visible) {
	Component b = getBasicComponent();
	if (b != null)
		b.setVisible(visible);
	Component c = getComponent();
	if ((b != null) && (c != null) && (c != b))
		c.setVisible(visible);
	forceParentRepaint();
}
/**
 * Invoked when a popup has been requested
 */
public void showPopup(MouseEvent event) {
	if (fPopupMenu == null)
		return;
	Component popupOwner = event.getComponent();
	if (popupOwner != null) {
		fPopupMenu.prepareForPopup();
		JMenu jmenu = (JMenu) fPopupMenu.getComponent();
		JPopupMenu popup = jmenu.getPopupMenu();
		if (popup != null)
			popup.show(popupOwner, event.getX(), event.getY());
	}
}
}
