package com.ibm.ulc.ui;

/*
 * Copyright (c) 1997 Object Technology International Inc.
 */
import java.awt.*;
import javax.swing.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.ui.base.*;
import com.ibm.ulc.ui.dataTypes.*;
import com.ibm.ulc.ui.lists.*;
import javax.swing.plaf.*;
/**
 * A display area for a short text string or an image, or both.
 */

public class UILabel extends UIFormComponent {
	protected JLabel fLabel = new UiJLabel();
	protected UIDataType fDataType = null;
	protected UIIcon fUIIcon= null;
/**
 * Configure the UIColumn to use this widget as a renderer.
 *
 * @see UIColumn#restoreState
 */
public void configureAsColumnRenderer(UIColumn column, boolean editable, IDataType dataType) {
	column.getTableColumn().setCellRenderer(new UiTableCellRenderer((JComponent) getComponent()));
	if (editable) {
		JTextField field = new JTextField() {
			public boolean isManagingFocus() {
				return true;
			}
		};
		column.configureForTableEditing(field);
		JLabel c= (JLabel) getComponent();
		field.setHorizontalAlignment(getHorizontalAlignment());
		if (! (c.getFont() instanceof FontUIResource))		
			field.setFont(c.getFont());
		if (! (c.getForeground() instanceof ColorUIResource))
			field.setForeground(c.getForeground());
		if (! (c.getBackground() instanceof ColorUIResource))
			field.setBackground(c.getBackground());
		column.getTableColumn().setCellEditor(new UiCellEditor(field, dataType));
	} else
		column.getTableColumn().setCellEditor(null);
}
String convertToString(Object o) {
	if (fDataType != null) {
		String s = fDataType.convertToString(o, false);
		// System.out.println("convertToString(" + o.getClass().getName() + "): " + s);
		return s;
	}
	return o.toString();
}
public void free() {
	if (fLabel instanceof UiJLabel)
		 ((UiJLabel) fLabel).free();
	fLabel = null;
	fDataType = null;
	fUIIcon = null;
	super.free();
}
public Component getComponent() {
	return fLabel;
}
/**
 * Gets the alignment of the label's contents along the X axis.
 * <p>
 *
 * @return alignment  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>BOX_LEFT</code> (the default for text-only labels),
 *           <code>BOX_CENTER</code> (the default for image-only labels), or
 *           <code>BOX_RIGHT</code>.
 *
 * @see IDefaults 
 */
public int getHorizontalAlignment() {
	return fLabel.getHorizontalAlignment();
}
/**
 * Gets the horizontal position of the label's text,
 * relative to its image.
 *
 * @param x  One of the following constants
 *           defined in <code>SwingConstants</code>:
 *           <code>LEFT</code>,
 *           <code>CENTER</code>, or
 *           <code>RIGHT</code> (the default).
 *
 * @see SwingConstants
 * @beaninfo
 *       expert: true
 *  description: The horizontal position of the label's text, 
 *               relative to its image.
 */
public int getHorizontalTextPosition() {
	return fLabel.getHorizontalTextPosition();
}
public Icon getIcon() {
	if (fLabel != null)
		return fLabel.getIcon();
	else
		return null;
}
public String getText() {
	if (fLabel != null)
		return fLabel.getText();
	else
		return null;
}
/**
 * Gets the alignment of the label's contents along the Y axis.  
 * <p>
 * The default value of this property is BOX_CENTER.
 * <p>
 * 
 * @return alignment One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>BOX_TOP</code>,
 *           <code>BOX_CENTER</code> (the default), or
 *           <code>BOX_BOTTOM</code>.
 *
 */
public int getVerticalAlignment() {
	return fLabel.getVerticalAlignment();
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("setIcon")) {
		UIIcon icon = null;
		if (!args.isNull())
			icon = (UIIcon) getManaged(UIIcon.class, conn, args, false);
		setIconText(icon, fLabel.getText());
		return;
	}
	if (request.equals("setHorizontalAlignment")) {
		setHorizontalAlignment(args.asInt(BOX_LEFT));
		return;
	}
	if (request.equals("setVerticalAlignment")) {
		setVerticalAlignment(args.asInt(BOX_CENTER));
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	super.restoreState(conn, args);
	fDataType = (UIDataType) getManaged(UIDataType.class, conn, args.get("dataType"));
	UIFont font = (UIFont) getManaged(UIFont.class, conn, args.get("font"));
	if (font != null)
		fLabel.setFont(font.getFont());
	setHorizontalAlignment(args.get("ha", BOX_LEFT));
	setVerticalAlignment(args.get("va", BOX_CENTER));
	setHorizontalTextPosition(args.get("htp", BOX_RIGHT));
	if (fFormModel != null) {
		updateFromModel();
	} else {
		fUIIcon = (UIIcon) getManaged(UIIcon.class, conn, args.get("icon"));
		if (fUIIcon != null)
			setIconText(fUIIcon, args.get("label", ""));
		else
			setIconText(null, args.get("label", ""));
}
}
/**
 * Set the background of the receiver to the color defined
 * by the RGB int values <code>red, green, and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */
public void setBackgroundColor(int red, int green, int blue, boolean refresh) {
	setBackgroundColorWithOpaque(red, green, blue, refresh);
}
/**
 * We implement this method since the UILabel is
 * used in different contexts.
 *
 * Do not use this method even though it is
 * public.
 */
public void setComponent(Component component) {
	fLabel= (JLabel) component;
}
/**
 * Sets the alignment of the label's contents along the X axis.
 * <p>
 *
 * @param alignment  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>BOX_LEFT</code> (the default for text-only labels),
 *           <code>BOX_CENTER</code> (the default for image-only labels), or
 *           <code>BOX_RIGHT</code>.
 *
 * @see IDefaults 
 */
public void setHorizontalAlignment(int alignment) {
	fLabel.setHorizontalAlignment(alignment);
}
/**
 * Sets the horizontal position of the label's text,
 * relative to its image.
 *
 * @param x  One of the following constants
 *           defined in <code>SwingConstants</code>:
 *           <code>LEFT</code>,
 *           <code>CENTER</code>, or
 *           <code>RIGHT</code> (the default).
 *
 * @see SwingConstants
 * @beaninfo
 *       expert: true
 *  description: The horizontal position of the label's text, 
 *               relative to its image.
 */
public void setHorizontalTextPosition(int x) {
	fLabel.setHorizontalTextPosition(x);
}
public void setIcon(UIIcon uiIcon) {
	setIcon(uiIcon, true);
}
public void setIcon(UIIcon uiIcon, boolean validate) {
	fUIIcon = uiIcon;
	if (uiIcon != null) {
		Image image = uiIcon.getImage(this, "setIcon");
		if (image != null) {
			Icon icon = new ImageIcon(image);
			if (icon != null && fLabel.getIcon() != icon) {
				fLabel.setIcon(icon);
			}
		} else {
			fLabel.setIcon(null);
		}
	} else {
		fLabel.setIcon(null);
	}
	if (validate) {
		Component parent = fLabel.getParent();
		if (parent != null) {
			parent.validate();
			parent.repaint();
		}
	}
}
public void setIconText(UIIcon uiIcon, String text) {
	setIcon(uiIcon, false);
	if (text == null)
		fLabel.setText("");
	else
		if (!text.equals(fLabel.getText())) {
			fLabel.setText(text);
		}
	fLabel.revalidate();
	fLabel.repaint();
	Component parent = fLabel.getParent();
	if (parent != null) {
		parent.validate();
		parent.repaint();
	}
}
public void setLabel(String label) {
	if (label == null)
		setIconText(fUIIcon, "");
	else
		setIconText(fUIIcon, label);
}
/**
 * Sets the alignment of the label's contents along the Y axis.  
 * <p>
 * The default value of this property is BOX_CENTER.
 * <p>
 * 
 * @param alignment One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>BOX_TOP</code>,
 *           <code>BOX_CENTER</code> (the default), or
 *           <code>BOX_BOTTOM</code>.
 *
 */
public void setVerticalAlignment(int alignment) {
	fLabel.setVerticalAlignment(alignment);
}
public String toString() {
	return getText();
}
public void updateCachedResource(String methodName, ICachedResource resource, Object userData) {
	if (methodName.equals("setIcon")) {
		UIIcon icon = (UIIcon) resource;
		setIcon(icon);
		getBasicComponent().invalidate();
		forceFramePack();
		return;
	}
	super.updateCachedResource(methodName, resource, userData);
}
public void updateFromModel() {
	if (fFormModel != null) {
		Object o = fFormModel.getValueAt(fKey);
		if (o instanceof PendingCell)
			o = null;
		if (o == null) {
			setIconText(null, null);
			setEnabled(false);
		} else
			if (o instanceof UIIcon) {
				setIconText((UIIcon) o, null);
				setEnabled(true);
			} else {
				setIconText(null, convertToString(o));
				setEnabled(true);
			}
		forceParentRepaint(); //1EKNZV6

	}
}
}
