package com.ibm.ulc.ui.dataTypes;

/**
 * Instances of this class validate user input and format objects for output
 *
 * Creation date: (23-Nov-99 4:56:04 PM)
 * @author: Markus Marzetta (CS-ZRH)
 */

import java.util.Locale;
import java.util.ResourceBundle;
import java.text.DecimalFormatSymbols;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;

public class UINumberValidator extends UIDataType {
	/**
	 * The converter used to pass from String to Object and vice versa.
	 * @serial	 
	 */
	protected IStringConverter fConverter = null;
	/**
	 * Allowed input characters
	 * @serial	 
	 */
	protected static String validChars = null;
	static {
		DecimalFormatSymbols symbs = new DecimalFormatSymbols();
		StringBuffer buffer = new StringBuffer("0123456789.- ");
		buffer.append(symbs.getDecimalSeparator());
		buffer.append(symbs.getGroupingSeparator());
		buffer.append(symbs.getMinusSign());
		// DecimalFormatSymbols does not allow yet to access the exponentialSymbol.
		// Hence, let's hack it
		String exponentialSymbol = "";
		{
			Locale locale = Locale.getDefault();
			ResourceBundle rb = ResourceBundle.getBundle(
					"java.text.resources.LocaleElements", locale);
			exponentialSymbol = rb.getStringArray("NumberElements")[7];
		}
		buffer.append(exponentialSymbol);
		validChars = buffer.toString();
	}
	/**
	 * The formatting/converting style defined by the classes 
	 * Double, Integer, FloatingDecimal
	 * @serial	 
	 */
	public static final int JAVA_LANG_STYLE = 0;
	/**
	 * The formating/converting style defined by the class NumberFormat
	 * @serial	 
	 */
	public static final int JAVA_TEXT_STYLE = 1;
/**
 * UINumberValidator constructor comment.
 */
public UINumberValidator() {
	super();
}
/**
 * Check if the input string conforms to the range specified. Throw an exception on error.
 *
 * @param phase int	The current validation phase.
 * The phase can be one of the following:
 * <pre>
 *	FORM_NOTIFICATION_IMMEADIATE : Typically no validation is performed in this case the input string is returned as is.
 *	FORM_NOTIFICATION_ON_FOCUS_CHANGE 	 
 * 	FORM_NOTIFICATION_ON_REQUEST
 *	</pre>
 * @param newString The String to convert.
 * @return An object representing the converted String or null.
 *
 * @see IDataType#convertToObject(int, String, Object)
 */

public Object convertToObject(int phase, String newString, Object previousValue) throws DataTypeConversionException {
	if (phase == FORM_NOTIFICATION_IMMEDIATE)
		return newString; // no checking
	if (newString != null && newString.length() > 0) {
		return fConverter.convertToObject(newString);
	}
	return null;
}
/**
 * Return the String represenation of the object or the empty string if object is null.
 * There are 2 possible cases:
 * 1. we are in edit mode (forEditing==true),
 *    then object is the entered String, and 
 *    its string representation is the String itself.
 * 2. we are not in edit mode (forEditing==false),
 *    then the string representation is computed by fConverter
 *
 * @see IDataType#convertToString(Object, boolean)
 */
public String convertToString(Object object, boolean forEditing) {

	if (object == null) return "";
	if (object instanceof Number) {
		if (!forEditing) {
			return fConverter.convertToString(object);
		} else {
			return fConverter.convertToEditString(object);
		}
	} else {
		return object.toString();
	}
}
/**
 * The default implementation returns original string without modification.
 *
 * @see IDataType#filterInput(String)
 */
public String filterInput(String newString) {
	if (hasValidCharacters(newString))
		return newString;
	return null;
}
/**
 * Ensure that the given string has no invalid characters.
 *
 * @see IDataType#filterInput(String)
 */
public boolean hasValidCharacters(String s) {
	for (int i = 0; i < s.length(); i++)
		if (validChars.indexOf(s.charAt(i)) < 0)
			return false;
	return true;
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	super.restoreState(conn, args);

	boolean isInteger = args.get("iv", false);
	int style = args.get("style", JAVA_LANG_STYLE);
	
	IStringConverter conv = null;

	if (style == JAVA_LANG_STYLE) {
		conv = UIIntegerFloatingDecimalConverter.getInstance(isInteger);
	} else { //style == JAVA_TEXT_STYLE
		UINumberFormatConverter nfConv = UINumberFormatConverter.getInstance(isInteger);
		nfConv.setGroupingUsed(args.get("gru",true));
		nfConv.setMaximumFractionDigits(args.get("maxFD",2));
		nfConv.setMinimumFractionDigits(args.get("minFD",2));
		conv = nfConv;
	}

	Object min = null;
	if (args.isDefined("min")) {
		if (isInteger) {
			min = new Integer(args.get("min",0));
		} else {
			min = new Double(args.get("min",0.0));
		}
	}
	
	Object max = null;
	if (args.isDefined("max")) {
		if (isInteger) {
			max = new Integer(args.get("max",0));
		} else {
			max = new Double(args.get("max",0.0));
		}
	}
	
	if (min != null || max != null) {
		UIRangeConverter rConv = new UIRangeConverter(conv);
		rConv.setMin(min);
		rConv.setMax(max);
		fConverter = rConv;
	} else {
		fConverter = conv;
	}
}
}
