package com.ibm.ulc.ui.lists;

import javax.swing.*;
import java.awt.event.*;

/**
 * This class enhances Jlabel in order to be used
 * as an icon in a container. This means it offers
 * visual feedback for the icon and the text if it
 * is selected. The selected icon can either be set
 * or will be computed.
 *
 * @see JLabel 
 */
public class UiJContainerIcon extends JLabel {

	transient private boolean fIsSelected= false;
	transient private Icon fSelectedIcon;
/**
 * UiContainerIcon constructor comment.
 */
public UiJContainerIcon() {
	this("", null, CENTER);
}
/**
 * UiContainerIcon constructor comment.
 * @param text java.lang.String
 */
public UiJContainerIcon(String text) {
	this(text, null, LEFT);
}
/**
 * UiContainerIcon constructor comment.
 * @param text java.lang.String
 * @param horizontalAlignment int
 */
public UiJContainerIcon(String text, int horizontalAlignment) {
	this(text, null, horizontalAlignment);
}
/**
 * UiContainerIcon constructor comment.
 * @param text java.lang.String
 * @param icon javax.swing.Icon
 * @param horizontalAlignment int
 */
public UiJContainerIcon(String text, javax.swing.Icon icon) {
	this(text, icon, CENTER);
}
/**
 * UiContainerIcon constructor comment.
 * @param text java.lang.String
 * @param icon javax.swing.Icon
 * @param horizontalAlignment int
 */
public UiJContainerIcon(String text, javax.swing.Icon icon, int horizontalAlignment) {
	super(text, icon, horizontalAlignment);
	setHorizontalTextPosition(CENTER);
	setVerticalTextPosition(BOTTOM);
	addMouseListener(
		new MouseAdapter() {
			public void mousePressed(MouseEvent e) {forwardToParent(e);}
			public void mouseReleased(MouseEvent e) {forwardToParent(e);}
		}
	);
	addMouseMotionListener(
		new MouseMotionListener() {
			public void mouseDragged(MouseEvent e) {forwardToParent(e);}
			public void mouseMoved(MouseEvent e) {forwardToParent(e);}
		}
	);
	// Listen for Focus events
	addFocusListener(
		new FocusListener() {
			public void focusGained(FocusEvent event) {
				repaint();
			}
			public void focusLost(FocusEvent event) {
				// repaint focus is lost
				repaint();
			}
		}
	);
	// Fix Swing bug that prevents keyboard actions
	// if not registered with tool tip manager
	ToolTipManager.sharedInstance().registerComponent(this);
}
/**
 * UiContainerIcon constructor comment.
 * @param image javax.swing.Icon
 */
public UiJContainerIcon(javax.swing.Icon image) {
	this(null, image, CENTER);
}
/**
 * UiContainerIcon constructor comment.
 * @param image javax.swing.Icon
 * @param horizontalAlignment int
 */
public UiJContainerIcon(javax.swing.Icon image, int horizontalAlignment) {
	this(null, image, horizontalAlignment);
}
/**
 * Forward the event to its container
 */
protected void forwardToParent(KeyEvent e) {
	java.awt.Container parent= getParent();
	if (parent != null)
		parent.dispatchEvent(e);
}
/**
 * Forward the event to its container
 */
protected void forwardToParent(MouseEvent e) {
	java.awt.Container parent= getParent();
	if (parent != null)
		parent.dispatchEvent(e);
}
/**
 * Return the (computed) icon to be used when the
 * component is in the selected state and enabled.
 * If the selected icon is null we compute it.
 *
 * @return the disabled icon of this component.
 * @see ImageIcon
 * @see #isSelected
 * @see #setSelected
 */
public Icon getSelectedIcon() {
	if ((fSelectedIcon == null) &&
	   (getIcon() != null) &&
	   (getIcon() instanceof ImageIcon)) {
		java.awt.Image selectedImage= SelectedImageFilter.createSelectedImage(((ImageIcon) getIcon()).getImage());
		fSelectedIcon= new ImageIcon(selectedImage);
	}
	return fSelectedIcon;
}
/**
 * @return true if the icon is selected
 * @see #isSelected
 */
public boolean isSelected() {
	return fIsSelected;
}
/**
 * We have to reset the computed selectedIcon
 * if the "normal" icon changes
 *
 * @see JLabel#setIcon for details.
 * @param icon the "normal" Icon for this component
 */
public void setIcon(Icon icon) {
	if (icon != getIcon())
		fSelectedIcon= null;
	super.setIcon(icon);
}
/**
 * Set the state of the container icon to be
 * either selected or not selected
 * Default is false
 */
public void setSelected(boolean newState) {
	firePropertyChange("selected", fIsSelected, newState);
	if (fIsSelected != newState) {
		fIsSelected= newState;
		repaint();
	}
}
/**
 * We install a single UI class for the LAF here.
 */
public void updateUI() {
	setUI(UiJContainerIconUI.createUI(this));
}
}
