package com.ibm.ulc.util;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.io.*;
import java.util.*;

/**
 * Anything is a polymorphic, self describing, and dynamic data structure.
 * An Anything can represent simple data types like longs, boolean, doubles or strings,
 * arrays of Anythings, and dictionaries mapping between strings and Anythings
 * (i.e. associative arrays). Because Anythings are recursive by definition,
 * they can model arbitrarily complex structures.
 * <P>
 * Anything are streamed with AnythingWriters and AnythingReaders. Two formats
 * exist: a pretty printing pure ASCII format (classes AnythingReader and AnythingWriter)
 * and a more compact easier to parse format (classes AnythingReader2 and AnythingWriter2).
 * <P>
 * In the context of ULC Anythings are used as a simple but robust marshalling mechanism.
 *
 * @see AnythingReader
 * @see AnythingWriter
 * @see AnythingReader2
 * @see AnythingWriter2
 */

public final class Anything extends Object implements Serializable {
	private Object fContents;
	private Hashtable fDict;
	private int fTag;
	public static final TimeZone fgGmtTimeZone= TimeZone.getTimeZone("GTM");
	public static final int NULL = 0;
	public static final int BOOLEAN = 1;
	public static final int LONG = 2;
	public static final int DOUBLE = 3;
	public static final int STRING = 4;
	public static final int BYTES = 5;
	public static final int VECTOR = 6;
	public static final int SERIALIZABLE = 7;
	/**
	 * Constructs an empty (null) Anything.
	 */
	public Anything() {
		fTag= NULL;
		fContents= null;
	}
	/**
	 * Constructs an Anything representing a byte array.
	 */
	public Anything(byte[] s) {
		fTag= BYTES;
		fContents= s;
	}
/**
 * Constructs an Anything representing a String.
 */
public Anything(int[] ints) {
	for (int i = 0; i < ints.length; i++)
		append(new Anything(ints[i]));
}
/**
 * Constructs an Anything representing a String.
 */
public Anything(String[] strings) {
	for (int i = 0; i < strings.length; i++)
		append(new Anything(strings[i]));
}
	/**
	 * Constructs an Anything representing a double.
	 */
	public Anything(double d) {
		fTag= DOUBLE;
		fContents= new Double(d);
	}
	/**
	 * Constructs an Anything representing an int.
	 */
	public Anything(int l) {
		fTag= LONG;
		fContents= new Long(l);
	}
	/**
	 * Make a shallow copy
	 */
	private Anything(int tag, Object contents, Hashtable dict) {
		fTag= tag;
		fContents= contents;
		fDict= dict;
	}
	/**
	 * Constructs an Anything representing a long.
	 */
	public Anything(long l) {
		fTag= LONG;
		fContents= new Long(l);
	}
	/**
	 * Constructs an Anything representing a Serializable.
	 * @param ob java.io.Serializable
	 */
	public Anything(Serializable s) {
		fTag= SERIALIZABLE;
		fContents= s;
	}
	/**
	 * Constructs an Anything representing a double.
	 */
	public Anything(Double d) {
		fTag= DOUBLE;
		fContents= d;
	}
	/**
	 * Constructs an Anything representing a String.
	 */
	public Anything(String s) {
		fTag= STRING;
		fContents= s;
	}
/**
 * Constructs an Anything representing a String.
 */
public Anything(Vector v) {
	Enumeration e = v.elements();
	while (e.hasMoreElements()) {
		Object o = e.nextElement();
		append(Anything.fromObject(o));
	}
}
	/**
	 * Constructs an Anything representing a boolean.
	 */
	public Anything(boolean l) {
		fTag= BOOLEAN;
		fContents= new Boolean(l);
	}
	/**
	 * Converts this Anything to a vector and adds the given Anything to the end of the vector.
	 */
	public void append(Anything value) {
		if (fTag != VECTOR) {
			Vector v= new Vector();
			if (fContents != null) {
				v.addElement(new Anything(fTag, fContents, fDict));
			}	
			fTag= VECTOR;
			fContents= v;
		}
		put(size(), value);
	}
	/**
	 * Generates the boolean representation of this Anything.
	 * If the Anything doesn't represent a simple type or if it cannot converted int a boolean
	 * the given default value is returned.
	 */
	public boolean asBoolean(boolean dflt) {
		switch (fTag) {
		case BOOLEAN:
		    Boolean b= (Boolean)fContents;
			return b.booleanValue();
		case LONG:
		case DOUBLE:
		    Number n= (Number)fContents;
			return n.longValue() != 0;
		case STRING:
		case BYTES:
		case SERIALIZABLE:
		default:
			break;
		}
		return dflt;
	}
/**
 * Generates the byte array representation of this Anything.
 * If the Anything doesn't represent a simple type or if it cannot converted into a byte array
 * (boolean, long, double, string, byte[]) null is returned.
 */
public byte[] asBytes() {
	switch (fTag) {
		case BOOLEAN :
		case LONG :
		case DOUBLE :
		case STRING :
			String s = fContents.toString();
			if (Anything.isWide(s))
				return null;
			int l = s.length();
			byte[] buf = new byte[ (int) l];
			for (int i = 0; i < l; i++) {
				char c = s.charAt(i);
				buf[i] = (byte) c;
			}
			return buf;
		case BYTES :
			return (byte[]) fContents;
		case SERIALIZABLE :
			try {
				ByteArrayOutputStream bytes = new ByteArrayOutputStream();
				ObjectOutputStream oos = new ObjectOutputStream(bytes);
				oos.writeObject(fContents);
				oos.flush();
				oos.close();
				return bytes.toByteArray();
			} catch (Exception e) {
				System.out.println(e);
			}
		default :
			break;
	}
	return null;
}
/**
 * Generates the double representation of this Anything.
 * If the Anything doesn't represent a simple type or if it cannot converted int a double
 * the given default value is returned.
 */
public double asDouble(double dflt) {
	switch (fTag) {
		case LONG :
		case DOUBLE :
			Number n = (Number) fContents;
			return n.doubleValue();
		case STRING :
			try {
				Double d = new Double((String) fContents);
				return d.doubleValue();
			} catch (NumberFormatException e) {
				try {
					String str = (String) fContents;
					if (str.equals("NaN"))
						return Double.NaN;
				} catch (NumberFormatException e2) {
				}
			}
			break;
		default :
			break;
	}
	return dflt;
}
	/**
	 * Generates the int representation of this Anything.
	 * If the Anything doesn't represent a simple type or if it cannot converted int a int
	 * the given default value is returned.
	 */
	public int asInt(int dflt) {
		switch (fTag) {
		case LONG:
		case DOUBLE:
		    Number n= (Number)fContents;
			return n.intValue();
		case STRING:
		case BYTES:
	        try {
	            return (int) Long.parseLong((String)fContents);
	        } catch (NumberFormatException e) { }
	        break;
		default:
			break;
		}
		return dflt;
	}
	/**
	 * Generates the long representation of this Anything.
	 * If the Anything doesn't represent a simple type or if it cannot converted int a long
	 * the given default value is returned.
	 */
	public long asLong(long dflt) {
		switch (fTag) {
		case LONG:
		case DOUBLE:
		    Number n= (Number)fContents;
			return n.longValue();
		case STRING:
		case BYTES:
	        try {
	            return Long.parseLong((String)fContents);
	        } catch (NumberFormatException e) { }
	        break;
		default:
			break;
		}
		return dflt;
	}
	/**
	 * Generates the Serializable representation of this Anything.
	 * If the Anything doesn't represent a Serializable type or if it cannot converted int a Serializable
	 * the given default value is returned.
	 */
	public Serializable asSerializable(Serializable dflt) {
		switch (fTag) {
		case SERIALIZABLE:
			return (Serializable) fContents;
		case BYTES:
			try {
				ObjectInputStream ois= new ObjectInputStream(new ByteArrayInputStream(asBytes()));
				Serializable ser= (Serializable) ois.readObject();
				ois.close();				
				return ser;
			} catch (Exception e) {
				break;
			}			
		default:
			break;
		}
		return dflt;
	}
	/**
	 * Generates the string representation of this Anything.
	 * If the Anything doesn't represent a simple type
	 * (boolean, long, double, string, byte[]) the given default value is returned.
	 */
	public String asString(String dflt) {
		switch (fTag) {
		case BOOLEAN:
		case LONG:
		case DOUBLE:
		case STRING:
		case BYTES:
		case SERIALIZABLE:
			return fContents.toString();
		default:
			break;
		}
		return dflt;
	}
	/**
	 * Clones an Anything. Creates a clone by using the storing
	 * mechanism to flatten the Anything to a stream followed by
	 * resurrecting it from the same stream.
	 * @return Object
	 */
	public Object clone() {
		Object o= null;
		try {
			ByteArrayOutputStream output= new ByteArrayOutputStream(200);
			write(output, new AnythingWriter2());
			output.close();
			ByteArrayInputStream input= new ByteArrayInputStream(output.toByteArray());
			o= Anything.read(input);
			input.close();
		} catch (IOException e) {
		}
		return o;
	}
	/**
	 * Pretty prints the Anything to the given stream.
	 */
	public void dump(OutputStream out) {
		dump(out, false);
	}
	/**
	 * Pretty prints the Anything to the given stream.
	 */
	public void dump(OutputStream out, boolean compact) {
		write(out, new AnythingWriter(compact, true));	// don't print raw bytes
	}
/**
 * Constructs an Anything representing an Object
 */
public static Anything fromObject(Object o) {
	if (o == null)
		return new Anything();
	if (o instanceof String) {
		return new Anything((String) o);
	}
	if (o instanceof Boolean) {
		Boolean b = (Boolean) o;
		return new Anything(b.booleanValue());
	}
	if (o instanceof Integer) {
		Integer i = (Integer) o;
		return new Anything(i.intValue());
	}
	if (o instanceof Long) {
		Long l = (Long) o;
		return new Anything(l.longValue());
	}
	if (o instanceof Double || o instanceof Float) {
		Number n = (Number) o;
		return new Anything(n.doubleValue());
	}
	if (o instanceof Date) {
		Calendar calendar= new GregorianCalendar();
		Calendar outCalendar= new GregorianCalendar(fgGmtTimeZone);
		calendar.setTime((Date) o);
		outCalendar.clear();	// important, otherwise milliseconds will make result dirty
		outCalendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), calendar.get(Calendar.DATE), calendar.get(Calendar.HOUR), calendar.get(Calendar.MINUTE), calendar.get(Calendar.SECOND)); //GMT0 based
		Anything a= new Anything();
		a.put("type", "date");
		a.put("milliseconds", outCalendar.getTime().getTime());
		return a;
	}
	if (o instanceof Vector) {
		Anything a = new Anything();
		a.put("type", "collection");
		a.put("value", (Vector) o);
		return a;
	}
	if (o instanceof Anything)
		return (Anything) o;
	return null;
}
	/**
	 * Retrieves the element at the specified index in this Anything.
	 * If the Anything doesn't represent a vector or if the index is out of bounds
	 * an empty Anything is returned.
	 */
	public Anything get(int index) {
		switch (fTag) {
		case NULL:
			break;

		case BOOLEAN:
		case LONG:
		case DOUBLE:
		case STRING:
		case BYTES:
		case SERIALIZABLE:
			return this;

		case VECTOR:
			Vector v= (Vector) fContents;
			Object o= null;
			try {
				o= v.elementAt(index);
			}
			catch (ArrayIndexOutOfBoundsException e) {
			}
			if (o != null)
				return (Anything) o;
			break;

		default:
			break;
		}

		return new Anything();
	}
	/**
	 * Retrieves the element associated with the given key from this Anything.
	 * If the Anything doesn't represent a dictionary or if there is no element associated
	 * with the key null is returned.
	 */
	public Anything get(String key) {
		if (fDict != null) {
			Integer i= (Integer) fDict.get(key);
			if (i != null)
				return get(i.intValue());
		}
		return null;
	}
	/**
	 * Retrieves the element associated with the given key from this Anything and converts it to
	 * a double.
	 * If the Anything doesn't represent a dictionary or if there is no element associated
	 * with the key the given default double is returned.
	 */
	public double get(String key, double dflt) {
		Anything a= get(key);
		if (a == null)
			return dflt;
		return a.asDouble(dflt);
	}
	/**
	 * Retrieves the element associated with the given key from this Anything and converts it to
	 * an int.
	 * If the Anything doesn't represent a dictionary or if there is no element associated
	 * with the key the given default int is returned.
	 */
	public int get(String key, int dflt) {
		Anything a= get(key);
		if (a == null)
			return dflt;
		return a.asInt(dflt);
	}
	/**
	 * Retrieves the element associated with the given key from this Anything and converts it to
	 * a long.
	 * If the Anything doesn't represent a dictionary or if there is no element associated
	 * with the key the given default long is returned.
	 */
	public long get(String key, long dflt) {
		Anything a= get(key);
		if (a == null)
			return dflt;
		return a.asLong(dflt);
	}
	/**
	 * Retrieves the element associated with the given key from this Anything and converts it to
	 * a Serializable.
	 * If the Anything doesn't represent a dictionary or if there is no element associated
	 * with the key the given default Serializable is returned.
	 */
	public Serializable get(String key, Serializable dflt) {
		Anything a= get(key);
		if (a == null)
			return dflt;
		return a.asSerializable(dflt);
	}
	/**
	 * Retrieves the element associated with the given key from this Anything and converts it to
	 * a String.
	 * If the Anything doesn't represent a dictionary or if there is no element associated
	 * with the key the given default string is returned.
	 */
	public String get(String key, String dflt) {
		Anything a= get(key);
		if (a == null)
			return dflt;
		return a.asString(dflt);
	}
	/**
	 * Retrieves the element associated with the given key from this Anything and converts it to
	 * a boolean.
	 * If the Anything doesn't represent a dictionary or if there is no element associated
	 * with the key the given default boolean is returned.
	 */
	public boolean get(String key, boolean dflt) {
		Anything a= get(key);
		if (a == null)
			return dflt;
		return a.asBoolean(dflt);
	}
	/**
	 * Retrieves the element associated with the given key from this Anything and converts it to
	 * a byte array.
	 * If the Anything doesn't represent a dictionary or if there is no element associated
	 * with the key null is returned.
	 */
	public byte[] getBytes(String key) {
		Anything a= get(key);
		if (a == null)
			return null;
		return a.asBytes();
	}
	/**
	 * Retrieve the tag from the Anything.
	 */
	public int getType() {
	    return fTag;
	}
	/**
	 * Determines whether the given key is in this Anything.
	 */
	public boolean isDefined(String key) {
		if (fDict != null)
			return fDict.containsKey(key);
		return false;
	}
	/**
	 * Is the Anything empty?
	 */
	public boolean isNull() {
	    return fTag == NULL;
	}
static boolean isWide(String s) {
	for (int i = 0; i < s.length(); i++)
		if (s.charAt(i) > 255)
			return true;
	return false;
}
	/**
	 * Gets all slot names (keys) of the Anything
	 *
	 * If the Anything doesn't represent a dictionary or if the index is out of bounds
	 * null is returned.
	 * @return java.util.Enumeration
	 */
	public Enumeration keys() {
		if (fDict != null)
			return fDict.keys();
		return new ObjectEnumerator(null);
	}
	/**
	 * A unit test :-)
	 */
	public static void main(String args[]) {

		Anything employees= new Anything();

		Anything anEmployee= new Anything();
		anEmployee.put("LastName", "Smith");
		
		Anything firstNames=new Anything("Jane");
//	converts to a Vector before appending the new value 
		firstNames.append(new Anything("Tarzan"));
		anEmployee.put("FirstNames", firstNames);
	
		Anything address= new Anything();
		
		address.put("Country", "USA");
		address.put("City", "Palo Alto");
		address.put("Street", "123 Main Street");
		address.put("State", "CA");
		address.put("ZIP", 94301);

		address.put("Color", new Anything(new java.awt.Color(50,50,50)));
		java.awt.Color color= null;
		address.put("NullColor", new Anything(color));
		
		anEmployee.put("Address", address);		
		
		employees.append(anEmployee);
		
		//clone is the ultimate test for the AnythingReader2 and AnythingWriter2
		Anything anotherEmployee= (Anything)anEmployee.clone();
		anotherEmployee.put("MiddleName", "K.");
		anotherEmployee.put("FirstNames", "James");
		employees.append(anotherEmployee);

		employees.write("C:\\example.any");
		
		for (Enumeration allEmployees= Anything.read("c:\\example.any").values(); allEmployees.hasMoreElements();){
			Anything emp= (Anything)allEmployees.nextElement();
			System.out.println("-----------Employee------------");
			for (Enumeration allKeys= emp.keys(); allKeys.hasMoreElements();) {
				String key= (String) allKeys.nextElement();
				System.out.println(key + " : " + emp.get(key));
				};	
			// test the enumerations
			for (Enumeration allValues= emp.get("FirstNames").values(); allValues.hasMoreElements();) {
				Anything value= (Anything) allValues.nextElement();
				System.out.println("value of FirstName : " + value);
				};	
			for (Enumeration allValues= emp.get("LastName").values(); allValues.hasMoreElements();) {
				Anything value= (Anything) allValues.nextElement();
				System.out.println("value of LastName : " + value);
				};				
					
			// test the serializable	
			Anything addr= emp.get("Address");
			java.awt.Color obj= (java.awt.Color)addr.get("Color", new java.awt.Color(0,0,0));
			System.out.println("Color : " + obj);
			obj= (java.awt.Color)addr.get("NullColor", new java.awt.Color(0,0,0));
			System.out.println("NullColor : " + obj);
			obj= (java.awt.Color)addr.get("NotDefinedColor", new java.awt.Color(0,0,0));
			System.out.println("NotDefinedColor : " + obj);
		}	
	
	}
	/**
	 * Replaces the element at the specified index in this vector.
	 * If the Anything doesn't represent a vector it is converted to a vector first.
	 */
	public void put(int index, Anything value) {
		if (fTag != VECTOR) {
			fTag= VECTOR;
			fContents= new Vector();
		}
		Vector v= (Vector) fContents;
		if (index >= v.size())
		    v.setSize(index+1);
		v.setElementAt(value, index);
	}
	/**
	 * Stores the given byte array value under the given key in this Anything.
	 */
	public void put(String key, byte[] value) {
	    put(key, new Anything(value));
	}
/**
 * Stores the given String value under the given key in this Anything.
 */
public void put(String key, int[] ints) {
	put(key, new Anything(ints));
}
	/**
	 * Stores the given double value under the given key in this Anything.
	 */
	public void put(String key, double value) {
	    put(key, new Anything(value));
	}
	/**
	 * Stores the given int value under the given key in this Anything.
	 */
	public void put(String key, int value) {
	    put(key, new Anything(value));
	}
	/**
	 * Stores the given long value under the given key in this Anything.
	 */
	public void put(String key, long value) {
	    put(key, new Anything(value));
	}
/**
 * Stores the given value under the given key in this Anything.
 * If the Anything doesn't represent a dictionary it is converted to a dictionary first.
 */
public void put(String key, Anything value) {
	if (fDict == null)
		fDict = new UlcHashtable();
	int s;
	Integer i = (Integer) fDict.get(key);
	if (i != null)
		s = i.intValue();
	else {
		s = size();
		fDict.put(key, new Integer(s));
	}
	put(s, value);
}
	public void put(String key, Serializable o) {
		put(key, new Anything(o));
	}
	/**
	 * Stores the given String value under the given key in this Anything.
	 */
	public void put(String key, String value) {
	    put(key, new Anything(value));
	}
/**
 * Stores the given String value under the given key in this Anything.
 */
public void put(String key, Vector v) {
	put(key, new Anything(v));
}
	/**
	 * Stores the given boolean value under the given key in this Anything.
	 */
	public void put(String key, boolean value) {
	    put(key, new Anything(value));
	}
	/**
	 * Read an Anything from the given stream.
	 * In case of errors null is returned.
	 * This method is able to detect both formats automatically and uses
	 * either an AnythingReader or an AnythingReader2.
	 */
	public static Anything read(InputStream in) {
		int c;
		try {
			c= in.read();
		} catch (IOException e) {
			return null;
		}
		IAnythingReader ar= null;
		if (c == 'V') {
		    ar= new AnythingReader2(c);
		} else {
		    ar= new AnythingReader(c);
		}
		return ar.read(in);
	}
	/**
	 * Read an Anything from the given file.
	 * In case of errors null is returned.
	 * This method is able to detect both formats automatically and uses
	 * either an AnythingReader or an AnythingReader2.
	 */
	public static
	Anything read(String filename) {
		try {
			InputStream in= new FileInputStream(filename);
			Anything a= read(new BufferedInputStream(in));
			in.close();
			return a;
		} catch (Exception e) {
		}
		return null;
	}
	/**
	 * Removes the element at the specified index from this vector.
	 */
	public void remove(int ix) {
		Assert.isTrue(fTag == VECTOR);
		Vector v= (Vector) fContents;
		v.removeElementAt(ix);
	}
/**
 * Stores the given value under the given key in this Anything.
 * If the Anything doesn't represent a dictionary it is converted to a dictionary first.
 */
public void remove(String key) {
	if (fDict == null)
		return;
	int s;
	Integer i = (Integer) fDict.get(key);
	if (i != null) {
		s = i.intValue();
		((Vector) fContents).setElementAt(new Anything(), s);
		//fDict.remove(key);
	}
}
	/**
	 * The size of the Anything.
	 * The size of an empty (null) Anything is 0.
	 * The Size of Anythings representing simple types (boolean, long, double, String, byte[]) is 1.
	 * For Anythings representing the structured types vector and dictionary the number of slots is returned.
	 */
	public int size() {
		switch (fTag) {
		case NULL:
			return 0;

		case BOOLEAN:
		case LONG:
		case DOUBLE:
		case STRING:
		case BYTES:
		case SERIALIZABLE:
			return 1;

		case VECTOR:
		    Vector v= (Vector) fContents;
			return v.size();

		default:
			break;
		}
		return 0;
	}
	/**
	 * Gets the slot name (key) at the given index.
	 * If the Anything doesn't represent a dictionary or if the index is out of bounds
	 * null is returned.
	 */
	public String slotName(int ix) {
		if (fDict != null) {
			for (Enumeration e= fDict.keys(); e.hasMoreElements();) {
				String s= (String) e.nextElement();
				Integer i= (Integer) fDict.get(s);
				if (i.intValue() == ix)
					return s;
			}
		}
		return null;
	}
/**
 * Stores the given value under the given key in this Anything.
 * If the Anything doesn't represent a dictionary it is converted to a dictionary first.
 */
public Vector toCollection() {
	if (isNull())
		return new Vector();
	Anything values = get("value");
	if (values != null) {
		return values.toCollection();
	}
	else {
		if (fTag == VECTOR) {
			Vector v = new Vector();
			for (int i = 0; i < size(); i++) {
				v.addElement((get(i).toObject()));
			}
			return v;
		}
	}
	return null;
}
/**
 * Stores the given value under the given key in this Anything.
 * If the Anything doesn't represent a dictionary it is converted to a dictionary first.
 */
public Object toObject() {
	switch (getType()) {
		case VECTOR :
			String type = get("type", null);
			if (type != null) {
				// System.out.println("convert: type: " + type);
				if (type.equals("date"))
					return new Date(get("milliseconds", 0L));
				if (type.equals("collection"))
					return toCollection();
			}
			break;
		case LONG :
			return new Integer(asInt(0));
		case BOOLEAN :
			return new Boolean(asBoolean(false));
		case DOUBLE :
			return new Double(asDouble(0.0));
		case STRING :
			return toString();
	}
	return null;
}
	/**
	 * Generates the string representation of this Anything.
	 */
	public String toString() {
		if (fContents != null)
			return fContents.toString();
		return null;
	}
	/**
	 * Retrieves the values of this Anything
	 */
	public Enumeration values() {
		switch (fTag) {
			case NULL:
				break;

			case BOOLEAN:
			case LONG:
			case DOUBLE:
			case STRING:
			case BYTES:
			case SERIALIZABLE:
				return new ObjectEnumerator(this);

			case VECTOR:
				Vector v= (Vector)fContents;
				return v.elements();
			
			default:
				break;
		}

		return new ObjectEnumerator(null);
	}
	/**
	 * Writes the Anything to the specified stream using the given format.
	 */
	public void write(OutputStream out, IAnythingWriter format) {
		format.print(out, this);
		try { out.flush(); } catch (IOException e) {}
	}
	/**
	 * Write the Anything to the given file by using the pretty printing format.
	 * @see AnythingWriter
	 */
	public void write(String filename) {
		try {
			OutputStream out= new FileOutputStream(filename);
			write(new BufferedOutputStream(out), new AnythingWriter());
			out.close();
		} catch (FileNotFoundException e) {
			System.out.println("File " + filename + " not found");
		} catch (IOException e) {
			System.out.println("IO Exception");
		}
	}
}
