package com.ibm.ulc.util;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.io.*;

/**
 * Writes an Anything in a compact and easy to parse format.
 * The format is still "mostly" ASCII but it is not as readable as the format produced
 * by AnythingWriter.
 * <p>
 * The (unverified) format is:
 * <pre>
 * format:=           &lt;version&gt; &lt;anything&gt; &lt;end_marker&gt;
 * end_marker:=       'E'
 * version:=          'V' &lt;number&gt;
 * anything:=         &lt;null&gt; | &lt;long&gt; | double | &lt;boolean&gt; | &lt;string&gt; | &lt;bytes&gt; | &lt;serializable&gt; | &lt;vector&gt;
 * null:=             '*'
 * long:=             'l' &lt;number&gt;
 * double:=           'd' &lt;double_as_string&gt;
 * double_as_string:= &lt;buffer&gt;
 * boolean:=          't' | 'f'
 * string:=           's' &lt;buffer&gt;
 * length:=           &lt;number&gt;
 * buffer:=           &lt;length&gt; { &lt;any_char&gt; }
 * bytes:=            'b' &lt;buffer&gt;
 * serializable:=		'o' lt;serializedJavaObject&gt;
 * vector:=           'a' &lt;length&gt; { [ ' ' | &lt;key&gt; ] &lt;anything&gt; }
 * key:=              '/' &lt;buffer&gt;
 * number:=           [ '+' | '-' ] &lt;digits&gt; { &lt;digit&gt; } '-'
 * digit:=            '0' | '1' | '2' | '3' | '4' | '5' | '6' | '7' | '8' | '9'
 */

public
class AnythingWriter2 implements IAnythingWriter {
	private OutputStream fOut;
	public AnythingWriter2() {
	}
	/**
	 * Creates the corresponding reader for this format
	 */
	public IAnythingReader createReader() {
		return new AnythingReader2();
	}
private void intPrint(byte[] b) throws InterruptedIOException, IOException {
	if (b.length > 0) {
		if (fOut != null)
			fOut.write(b);
	}
}
private void intPrint(int c)
throws InterruptedIOException, IOException {
	if (fOut != null)
		fOut.write(c);
}
private void intPrint(String s) throws InterruptedIOException, IOException {
	if (s.length() > 0)
		if (fOut != null) {
			long l = s.length();
			byte[] buf = new byte[ (int) l];
			for (int i = 0; i < l; i++) {
				char c = s.charAt(i);
				buf[i] = (byte) c;
			}
			intPrint(buf);
		}
}
/**
 * Prints an Anything in the compact format.
 */
public void print(OutputStream os, Anything a) {
	try {
		printEx(os, a);
	}
	catch (InterruptedIOException e1) {
		// We've been interrupted.  Make sure we're still interrupted.
		Thread.currentThread().interrupt();
	}
	catch (IOException e2) {
	}
}
private void print2(Anything a) throws InterruptedIOException, IOException {
	switch (a.getType()) {
		case Anything.NULL :
			intPrint('*');
			break;
		case Anything.LONG :
			intPrint('l');
			printLong(a.asLong(0L));
			break;
		case Anything.DOUBLE :
			intPrint('d');
			printString(a.toString());
			break;
		case Anything.BOOLEAN :
			if (a.asBoolean(false))
				intPrint('t');
			else
				intPrint('f');
			break;
		case Anything.STRING :
			String ss = a.toString();
			if (Anything.isWide(ss)) {
				intPrint('S');
				printWideString(ss);
			} else {
				intPrint('s');
				printString(ss);
			}
			break;
		case Anything.BYTES :
			intPrint('b');
			printBytes(a.asBytes());
			break;
		case Anything.SERIALIZABLE :
			intPrint('o');
			printSerializable(a.asSerializable(null));
			break;
		case Anything.VECTOR :
			intPrint('a');
			int s = a.size();
			printLong(s);
			for (int i = 0; i < s; i++) {
				String key = a.slotName(i);
				if (key != null) {
					intPrint('/');
					printString(key);
				} else {
					intPrint(' ');
				}
				print2(a.get(i));
			}
			break;
		default :
			break;
	}
}
private void printBytes(byte[] b)
throws InterruptedIOException, IOException {
	printLong(b.length);
	intPrint(b);
}
/**
 * Prints an Anything in the compact format.
 */
public synchronized void printEx(OutputStream os, Anything a) throws InterruptedIOException, IOException {
	fOut = os;
	intPrint('V'); // version
	printLong(1L);
	print2(a);
	intPrint('E');
}
private void printLong(long l) throws InterruptedIOException, IOException {
	intPrint(Long.toString(l));
	intPrint('-');
}
private void printSerializable(Serializable ob) throws InterruptedIOException, IOException {
	ByteArrayOutputStream bos = new ByteArrayOutputStream();
	new ObjectOutputStream(bos).writeObject(ob);
	printBytes(bos.toByteArray());
}
private void printString(String k) throws InterruptedIOException, IOException {
	printLong(k.length());
	intPrint(k);
}
private void printWideString(String s) throws InterruptedIOException, IOException {
	long l = s.length();
	printLong(l);
	byte[] buf = new byte[ (int) (l * 2)];
	int j = 0;
	for (int i = 0; i < l; i++) {
		char c = s.charAt(i);
		buf[j] = (byte) (c >>> 8);
		buf[j + 1] = (byte) (c & 0xff);
		j += 2;
	}
	intPrint(buf);
}
}
