package com.ibm.vap.Transactions;


/**
 *
 * Licensed Material - Property of IBM 
 * IBM(R) VisualAge(TM) for Java(TM) Version 2.0 
 * (C) Copyright IBM Corp. 1997, 1998 - All Rights Reserved. 
 * US Government Users Restricted Rights - Use, duplication or disclosure 
 * restricted by GSA ADP Schedule Contract with IBM Corp. 
 *
 * Business Object Manager
 * Manages the transactional state of a Business Object, ties it into the Transaction
 * and Persistence services.
 */

import java.util.Vector;
import javax.ejb.EntityBean;
import java.rmi.RemoteException;
import javax.ejb.RemoveException;
import com.ibm.vap.Isolation.ObjectIsolationImplementor;
import com.ibm.vap.Isolation.InternalIsolationImplementor;
import com.ibm.vap.Isolation.ExternalIsolationImplementor;
import com.ibm.vap.Isolation.InternalObjectLock;
import com.ibm.vap.Isolation.ExternalObjectLock;
import com.ibm.vap.common.VapMissingObjectVersionException;
import com.ibm.vap.common.VapAccessException;
import com.ibm.vap.common.VapReadFailureException;
import java.beans.*;

public class BOManager extends Object
{
	private VapEJBObject ejbObject = null;
	private VapHome home = null;
	private OID objectIdentifier;
	private OID temporaryIdentifier;
	private Version cachedVersion = null;
	private Transaction cachedTransaction = null;
	private InternalObjectLock internalLock;
	private ExternalObjectLock externalLock;
	private Vector predicateValues;
	private static final java.lang.String copyright  = "(c) Copyright International Business Machines Corporation, 1998";
/**
 * BOManager constructor
 */
public BOManager (VapEJBObject anObject, VapHome aHome)
{
	this.setEjbObject(anObject);
	this.setHome(aHome);
}
/* The dependents are held in the update version's data */
public void addPropertyChangeListener(PropertyChangeListener aListener) throws RemoteException,VapAccessException
{
	this.getUnrealizedVersionForRead().addPropertyChangeListener(aListener);
}
/**
* Answer the version's bean for read
*/
public EntityBean getBeanForRead() throws RemoteException,VapAccessException
{
	Version version;

	version = this.getVersionForRead();
	version.markRead();
	return version.getBean();
}
/**
* Answer the version's bean for update
*/
public EntityBean getBeanForUpdate() throws RemoteException,VapAccessException
{
	Version version;
	version = this.getVersionForUpdate();
	return version.getBean();
}
/**
 * Answer the cached transaction
 */
public Transaction getCachedTransaction()
{
	return cachedTransaction;
}
/**
 * Answer the cached version
 */
public Version getCachedVersion()
{
	return cachedVersion;
}
/**
 * Answer the ejb object
 */
public VapEJBObject getEjbObject()
{
	return ejbObject;
}
/**
 * Answer the external lock
 */
public ExternalObjectLock getExternalLock()
{
	return externalLock;
}
/***
 * Answer my home
 **/

public VapHome getHome()
{
	return home;
}
/**
 * Answer the internal lock
 */
public InternalObjectLock getInternalLock()
{
	return internalLock;
}
/**
 * Answer the isolation implementor
 */
public ObjectIsolationImplementor getIsolationImplementor()
{
	return this.getHome().getIsolationImplementorFor(this.getEjbObject());
}
/**
* Extract the object identifier for the passed versionData/BO, store it, and return it.
* Object identity is considered to be immutable, so it is only calculated once.
*/
public OID getObjectIdentifierFor(EntityBean aBean)
{
	if (objectIdentifier == null)
		objectIdentifier = new OID(this.getHome(),this.getEjbObject().getKeyFor(aBean));

	return objectIdentifier;	
}
/**
 * Answer the saved values of the predicate fields
 **/
public Vector getPredicateValues() {
	if (predicateValues == null) predicateValues = new Vector(0);
	return predicateValues;
}
/**
 * Answer the current state
 **/
public VersionState getState()
{
	VersionState state=null;

	try {
		state = this.getVersionForRead().getState();}
	catch (RemoteException e) {}

	return state;
}
/**
 * Answer the temporary identifier
 **/
public OID getTemporaryIdentifier()
{
	TemporaryKey key;
	
	if (temporaryIdentifier == null)
	{
		key = new TemporaryKey(this.getEjbObject());
		temporaryIdentifier = new OID(this.getHome(), key);
	}	

	return temporaryIdentifier;
}
/**
* Answer a version for update, do not mark it modified
*/
public Version getUnmarkedVersionForUpdate() throws RemoteException,VapAccessException
{
	Transaction currentTransaction;
	Version version;

	synchronized(this)
	{
		currentTransaction = Transaction.getCurrent();
		if ((cachedTransaction != currentTransaction) ||
		(cachedVersion == null) || (cachedVersion.isReadOnlyVersion()))
		{
			cachedTransaction = currentTransaction;
			cachedVersion = cachedTransaction.getView().getVersionForUpdate(this.getVersionIdentifier());
		}
		version = cachedVersion;
	}	

	if (version == null)
		version = this.reloadVersion();

	return version;
}
/**
* Answer a version for read
*/
public Version getUnrealizedVersionForRead() throws RemoteException,VapAccessException
{
	Transaction currentTransaction;
	Version version;

	synchronized(this)
	{
		currentTransaction = Transaction.getCurrent();
		if (cachedTransaction != currentTransaction)
		{
			cachedTransaction = currentTransaction;
			cachedVersion = cachedTransaction.getView().getUnrealizedVersionForRead(this.getVersionIdentifier());
		}	
		version = cachedVersion;
	}	

	if (version == null)
		throw new VapMissingObjectVersionException(this.getEjbObject());

	return version;
}
/**
* Answer a version for update
*/
public Version getUnrealizedVersionForUpdate() throws RemoteException,VapAccessException
{
	Transaction currentTransaction;
	Version version;

	synchronized(this)
	{
		currentTransaction = Transaction.getCurrent();
		if ((cachedTransaction != currentTransaction) ||
		(cachedVersion == null) || (cachedVersion.isReadOnlyVersion()))
		{
			cachedTransaction = currentTransaction;
			cachedVersion = cachedTransaction.getView().getUnrealizedVersionForUpdate(this.getVersionIdentifier());
		}
		version = cachedVersion;
	}	

	if (version == null)
		throw new VapMissingObjectVersionException(this.getEjbObject());

	return version;
}
/**
* Answer a version for read
*/
public Version getVersionForRead() throws RemoteException,VapAccessException
{
	Transaction currentTransaction;
	Version version;

	synchronized(this)
	{
		currentTransaction = Transaction.getCurrent();
		if (cachedTransaction != currentTransaction)
		{
			cachedTransaction = currentTransaction;
			cachedVersion = cachedTransaction.getView().getVersionForRead(this.getVersionIdentifier());
		}	
		version = cachedVersion;
	}	

	if (version == null)
		version = this.reloadVersion();

	return version;
}
/**
* Answer a version for update
*/
public Version getVersionForUpdate() throws RemoteException,VapAccessException
{
	Version version;

	version = this.getUnmarkedVersionForUpdate();
	version.markModified();

	return version;
}
/*
 * Return the current version identifier for this BOM.
 * used to lookup the current version.  Use the cached temporary identifier
 * or the OID.
 */
public OID getVersionIdentifier()
{
	return (temporaryIdentifier != null ? temporaryIdentifier : objectIdentifier);
}
/**
 * Answer true if the bom has an external lock
 */
public boolean hasExternalLock()
{
	return this.getExternalLock() != null;
}
/**
 * Answer true if the bom has an internal lock
 */
public boolean hasInternalLock()
{
	return this.getInternalLock() != null;
}
/**
 * Answer true if the bom has a version within this transaction
 */
public boolean hasVersion()
{
	if (this.getCachedTransaction() == Transaction.getCurrent())
		return cachedVersion != null;
	else
		return Transaction.getCurrentView().hasVersionAt(this.getVersionIdentifier());
}
/**
 * Answer true if the version is old
 **/
public boolean isOld() throws RemoteException,VapReadFailureException
{
	return this.getVersionForRead().isOld();
}
/**
 * Mark the version to be locally removed
 **/
public void markLocallyRemoved() throws RemoteException,RemoveException
{
	this.getVersionForRead().markLocallyRemoved();
}
/**
 * Mark version modified
 */
public void markModified() throws RemoteException,VapAccessException
{
	this.getUnmarkedVersionForUpdate().markModified();

	/* tell our business object to inform any counter links	that we have changed.
	When this happens if the links has already recorded our business object as 
	having added us we do not want to filter this, i.e. we want the signalling
	to broadcast the change event if we have been added. */
	
	getEjbObject().signalChangeAcrossLinks();
}
/**
 * Mark the version to be removed
 **/
public void markRemoved() throws RemoteException,RemoveException
{
	this.getUnmarkedVersionForUpdate().markRemoved();
}
/**
 * Answer the version's bean.
 * WARNING! To be used only during a new object creation.
 */
public EntityBean primGetBean()
{
	return this.getCachedVersion().getBean();
}
/**
* Answer the object identifier
*/
public OID primGetObjectIdentifier()
{
	return objectIdentifier;	
}
/**
* Answer a version for read
*/
public Version primGetVersionForRead() throws RemoteException,VapAccessException
{
	Transaction currentTransaction;
	Version version;

	synchronized(this)
	{
		currentTransaction = Transaction.getCurrent();
		if (cachedTransaction != currentTransaction)
		{
			cachedTransaction = currentTransaction;
			cachedVersion = cachedTransaction.getView().primGetVersionForRead(this.getVersionIdentifier());
		}
		version = cachedVersion;
	}	

	if (version == null)
		throw new VapMissingObjectVersionException(this.getEjbObject());

	return version;
}
/**
* Answer a version for update
*/
public Version primGetVersionForUpdate() throws RemoteException,VapAccessException
{
	Transaction currentTransaction;
	Version version;

	synchronized(this)
	{
		currentTransaction = Transaction.getCurrent();
		if ((cachedTransaction != currentTransaction) ||
		(cachedVersion == null) || (cachedVersion.isReadOnlyVersion()))
		{
			cachedTransaction = currentTransaction;
			cachedVersion = cachedTransaction.getView().primGetVersionForUpdate(this.getVersionIdentifier());
		}
		version = cachedVersion;
	}	

	if (version == null)
		throw new VapMissingObjectVersionException(this.getEjbObject());

	return version;
}
/**
* Reload a version
*/
public Version reloadVersion() throws VapMissingObjectVersionException
{
	try {
		this.getHome().reload(this.getEjbObject());
		return this.getCachedVersion();}
	catch (VapReadFailureException e) {
		throw new VapMissingObjectVersionException(this.getEjbObject());}
	catch (RemoteException e) {
		throw new VapMissingObjectVersionException(this.getEjbObject());}
}
/* The dependents are held in the update version's data */
public void removePropertyChangeListener(PropertyChangeListener aListener) throws RemoteException,VapAccessException
{
	this.getUnrealizedVersionForRead().removePropertyChangeListener(aListener);
}
/**
 * Reset the cached version and transaction
 */
public void resetCachedItemsFor(Version aVersion)
{
	if (aVersion == this.getCachedVersion())
	{
		this.setCachedVersion(null);
		this.setCachedTransaction(null);
	}
}
/*
 * Reset the cached temporary identifier, so that the real
 * object identifier will be used as the version identifier.
 * Sent when a new version is promoted to the shared view.
 */
public void resetTemporaryIdentifier()
{
	temporaryIdentifier = null;
}
/**
 * Set the cached transaction
 */
public void setCachedTransaction(Transaction aTransaction)
{
	cachedTransaction = aTransaction;
}
/**
 * Set the cached version
 */
public void setCachedVersion(Version aVersion)
{
	cachedVersion = aVersion;
}
/**
 * Set the ejb object
 */
public void setEjbObject(VapEJBObject anObject)
{
	ejbObject = anObject;
}
/**
 * Set the external lock
 */
public void setExternalLock(ExternalObjectLock aLock)
{
	externalLock = aLock;
}
protected void setHome(VapHome aHome)
{
	home = aHome;
}
/**
 * Set the internal lock
 */
public void setInternalLock(InternalObjectLock aLock)
{
	internalLock = aLock;
}
/**
 * Store the saved values of the predicate fields
 **/
public void setPredicateValues(Vector aVector) {
	predicateValues = aVector;
}
}