package com.ibm.vap.common;


/*
 * 
 */

import java.io.*;
import java.util.*;

/**
 * This is an implmentation of a Smalltalk style
 * set based on the Java Hashtable implementation.
 */
public class VapSet extends Object implements Cloneable, java.io.Serializable {
	private transient VapSetEntry table[];
	private transient int count;
	private int threshold;
	private float loadFactor;
	private static final long serialVersionUID = 1421746759512286392L;


	/**
	 * Constructs a new, empty set with a default capacity and load
	 * factor. 
	 *
	 */
	public VapSet() {
	this(101, 0.75f);
	}
	/**
	 * Constructs a new, empty hashtable with the specified initial capacity
	 * and default load factor.
	 *
	 * @param   initialCapacity   the initial capacity of the hashtable.
	 * @since   JDK1.0
	 */
	public VapSet(int initialCapacity) {
	this(initialCapacity, 0.75f);
	}
	/**
	 * Constructs a new, empty hashtable with the specified initial 
	 * capacity and the specified load factor. 
	 *
	 * @param      initialCapacity   the initial capacity of the hashtable.
	 * @param      loadFactor        a number between 0.0 and 1.0.
	 * @exception  IllegalArgumentException  if the initial capacity is less
	 *               than or equal to zero, or if the load factor is less than
	 *               or equal to zero.
	 * @since      JDK1.0
	 */
	public VapSet(int initialCapacity, float loadFactor) {
	if ((initialCapacity <= 0) || (loadFactor <= 0.0)) {
	    throw new IllegalArgumentException();
	}
	this.loadFactor = loadFactor;
	table = new VapSetEntry[initialCapacity];
	threshold = (int)(initialCapacity * loadFactor);
	}
/*
 * Add a single element to the set and return the set.
 * (This allows a type of cascading).  Nulls maybe added.
 *
 */
	public synchronized VapSet add(Object anObject) {
	// Make sure the value is not null
	if (anObject == null) 
	{
	    addNull();
	    return this;
	}

	// Makes sure the object is not already in the hashtable.
	VapSetEntry tab[] = table;
	int hash = anObject.hashCode();
	int index = (hash & 0x7FFFFFFF) % tab.length;
	for (VapSetEntry e = tab[index] ; e != null ; e = e.next) {
	    if ((e.hash == hash) && e.elementData.equals(anObject)) {
		return this;
	    }
	}

	if (count >= threshold) {
	    // Rehash the table if the threshold is exceeded
	    rehash();
	    return add(anObject);
	} 

	// Creates the new entry.
	VapSetEntry e = new VapSetEntry();
	e.hash = hash;
	e.elementData = anObject;
	e.next = tab[index];
	tab[index] = e;
	count++;
	return this;
	}
/*
 * Add a collection of elements to the set.
 */

public synchronized VapSet addAll(VapSet aVapSet)
{
	Enumeration e = aVapSet.elements();
	
	while(e.hasMoreElements())
		add(e.nextElement());
		
	return this;
}
/*
 * Add a collection of elements to the set.
 */

public synchronized VapSet addAll(Vector aVector)
{
	Enumeration e = aVector.elements();
	
	while(e.hasMoreElements())
		add(e.nextElement());
		
	return this;
}
/*
 * Call the add() again with the singleton null value
 * object.
 */
	private void addNull() 
	{
		add(NullValue.singleton());
	}
	/**
	 * Clears this hashtable so that it contains no keys. 
	 *
	 * @since   JDK1.0
	 */
	public synchronized void clear() {
	VapSetEntry tab[] = table;
	for (int index = tab.length; --index >= 0; )
	    tab[index] = null;
	count = 0;
	}
	/**
	 * Creates a shallow copy of this hashtable. The keys and values 
	 * themselves are not cloned. 
	 * This is a relatively expensive operation.
	 *
	 * @return  a clone of the hashtable.
	 * @since   JDK1.0
	 */
	public synchronized Object clone() {
	try { 
	    VapSet t = (VapSet)super.clone();
	    t.table = new VapSetEntry[table.length];
	    for (int i = table.length ; i-- > 0 ; ) {
		t.table[i] = (table[i] != null) 
		    ? (VapSetEntry)table[i].clone() : null;
	    }
	    return t;
	} catch (CloneNotSupportedException e) { 
	    // this shouldn't happen, since we are Cloneable
	    throw new InternalError();
	}
	}
	/**
	 * Tests if the specified object is a key in this set
	 * (including a null);
	 * 
	 */
	public synchronized boolean contains(Object anObject) {

	if (anObject == null) 
		return contains(nullValue());

	VapSetEntry tab[] = table;
	int hash = anObject.hashCode();
	int index = (hash & 0x7FFFFFFF) % tab.length;
	for (VapSetEntry e = tab[index] ; e != null ; e = e.next) {
	    if ((e.hash == hash) && e.elementData.equals(anObject)) {
		return true;
	    }
	}
	return false;
	}
	/**
	 * Returns an enumeration of the values in this hashtable.
	 * Use the Enumeration methods on the returned object to fetch the elements
	 * sequentially.
	 *
	 * @return  an enumeration of the values in this hashtable.
	 * @see     java.util.Enumeration
	 * @see     java.util.Hashtable#keys()
	 * @since   JDK1.0
	 */
	public synchronized Enumeration elements() {
	return new VapSetEnumerator(table);
	}
	/**
	 * Tests if this hashtable maps no keys to values.
	 *
	 * @return  <code>true</code> if this hashtable maps no keys to values;
	 *          <code>false</code> otherwise.
	 * @since   JDK1.0
	 */
	public boolean isEmpty() {
	return count == 0;
	}
/*
 * Answer the singleton null value object.
 */
	private NullValue nullValue() 
	{
		return NullValue.singleton();
	}
	/**
	 * readObject is called to restore the state of the hashtable from
	 * a stream.  Only the keys and values are serialized since the
	 * hash values may be different when the contents are restored.
	 * Read count elements and insert into the hashtable. 
	 */
	private synchronized void readObject(java.io.ObjectInputStream s)
		 throws IOException, ClassNotFoundException
	{
	// Read in the length, threshold, and loadfactor
	s.defaultReadObject();

	// Read the original length of the array and number of elements
	int origlength = s.readInt();
	int elements = s.readInt();

	// Compute new size with a bit of room 5% to grow but
	// No larger than the original size.  Make the length
	// odd if it's large enough, this helps distribute the entries.
	// Guard against the length ending up zero, that's not valid.
	int length = (int)(elements * loadFactor) + (elements / 20) + 3;
	if (length > elements && (length & 1) == 0)
	    length--;
	if (origlength > 0 && length > origlength)
	    length = origlength;

	table = new VapSetEntry[length];
	count = 0;

	// Read the number of elements and then all the key/value objects
	for (; elements > 0; elements--) {
	    Object anObject = s.readObject();
	    add(anObject);
	}
	}
	/**
	 * Rehashes the contents of the set into a set with a 
	 * larger capacity. This method is called automatically when the 
	 * number of objects in the set exceeds this set's capacity 
	 * and load factor. 
	 *
	 * 
	 */
	protected void rehash() {
	int oldCapacity = table.length;
	VapSetEntry oldTable[] = table;

	int newCapacity = oldCapacity * 2 + 1;
	VapSetEntry newTable[] = new VapSetEntry[newCapacity];

	threshold = (int)(newCapacity * loadFactor);
	table = newTable;

	//System.out.println("rehash old=" + oldCapacity + ", new=" + newCapacity + ", thresh=" + threshold + ", count=" + count);

	for (int i = oldCapacity ; i-- > 0 ;) {
	    for (VapSetEntry old = oldTable[i] ; old != null ; ) {
		VapSetEntry e = old;
		old = old.next;

		int index = (e.hash & 0x7FFFFFFF) % newCapacity;
		e.next = newTable[index];
		newTable[index] = e;
	    }
	}
	}
	/**
	 * Removes the object  from this set. This method does nothing if 
	 * the object is not in the set.
	 *
	 */
	public synchronized Object remove(Object anObject) {
	VapSetEntry tab[] = table;
	int hash = anObject.hashCode();
	int index = (hash & 0x7FFFFFFF) % tab.length;
	for (VapSetEntry e = tab[index], prev = null ; e != null ; prev = e, e = e.next) {
	    if ((e.hash == hash) && e.elementData.equals(anObject)) {
		if (prev != null) {
		    prev.next = e.next;
		} else {
		    tab[index] = e.next;
		}
		count--;
		return e.elementData;
	    }
	}
	return null;
	}
	/**
	 * Returns the number of keys in this hashtable.
	 *
	 * @return  the number of keys in this hashtable.
	 * @since   JDK1.0
	 */
	public int size() {
	return count;
	}
	/**
	 * Convert to an array.
	 */
	public synchronized Object[] toArray() {
		
		Enumeration e = elements();
		Object[] anArray = new Object[size()];
		Vector aVector = new Vector(size());
		int i = 0;
		
		while(e.hasMoreElements())
		{
			anArray[i] = (e.nextElement());
			++i;
		}	
		return anArray;		
	}
	/**
	 * 
	 * A better "toString".  Inserts new lines chars after each element
	 * and takes size of the into account, i.e. avoid printing large 
	 * sets (more than a hundred or so).
	 */
	public synchronized String toString() {
		
		if (size() - 1 < 100)
			return this.toStringForSmallSet();
			else return this.toStringForLargeSet();
	}
	/**
	 * 
	 * A better "toString".  Inserts new lines chars after each element
	 * and takes size of the into account, i.e. avoid printing huge 
	 * sets (more than a thousand or so).
	 */
	private  String toStringForLargeSet() {
		
	int max = size() - 1;
	StringBuffer buf = new StringBuffer();
	Enumeration k = elements();

	
	buf.append("{");

	for (int i = 0; i <= 30; i++) 
	{
	   String s1 = k.nextElement().toString();
	   buf.append(s1);
	   if (i < 30) 
			buf.append(" ");
	}
	buf.append("...etc...");
	for (int i = max - 30; i <= max; i++) 
	{
	  String s1 = k.nextElement().toString();
	  buf.append(s1);
	  if (i < max) 
		  buf.append(" ");
	}
	buf.append("}");
	return buf.toString();
}
	/**
	 * 
	 * A better "toString".  Inserts new lines chars after each element
	 * and takes size of the into account, i.e. avoid printing huge 
	 * sets (more than a thousand or so).
	 */
	private  String toStringForSmallSet() {
		
	int max = size() - 1;
	StringBuffer buf = new StringBuffer();
	Enumeration e = elements();
	
	buf.append("{");

	for (int i = 0; i <= max; i++) {
	    String s1 = e.nextElement().toString();
	    buf.append(s1);
	    if (i < max) 
		  buf.append(" ");
	}
	buf.append("}");
	return buf.toString();
	}
	/**
	 * Convert to a Vector.
	 */
	public synchronized Vector toVector() {
		
		Enumeration e = elements();
		Vector aVector = new Vector(size());
		while(e.hasMoreElements())
			aVector.addElement(e.nextElement());
		return aVector;		
	}
	/**
	 * WriteObject is called to save the state of the hashtable to a stream.
	 * Only the keys and values are serialized since the hash values may be
	 * different when the contents are restored.
	 * iterate over the contents and write out the keys and values.
	 */
	private synchronized void writeObject(java.io.ObjectOutputStream s)
		throws IOException
	{
	// Write out the length, threshold, loadfactor
	s.defaultWriteObject();

	// Write out length, count of elements and then the key/value objects
	s.writeInt(table.length);
	s.writeInt(count);
	for (int index = table.length-1; index >= 0; index--) {
	    VapSetEntry entry = table[index];

	    while (entry != null) {
		s.writeObject(entry.elementData);
		entry = entry.next;
	    }
	}
	}
}