/*****************************************************************************/
/*                                                                           */
/*                           OCO SOURCE MATERIALS                            */
/*                             IBM CONFIDENTIAL                              */
/*                                    OR                                     */
/*                        IBM CONFIDENTIAL RESTRICTED                        */
/*            WHEN COMBINED WITH THE AGGREGATED OCO SOURCE MODULES           */
/*                           FOR THIS PROGRAM PRODUCT                        */
/*                                                                           */
/*      VisualAge Generator Server for OS/2, AIX, HP-UX, and Windows NT      */
/*                          Version 3.0  5697-C28                            */
/*                    (C) COPYRIGHT IBM CORP. 1994,1997                      */
/*                                                                           */
/*****************************************************************************/
#ifndef FCWEXMAP_H
#define FCWEXMAP_H

#include <stddef.h>

//-----------------------------------------------------------------------------
// Coordinate class                                                           -
//-----------------------------------------------------------------------------
// This is the base class for all of the coordinate type classes such as      -
// position and size.                                                         -
//-----------------------------------------------------------------------------
class Coord
{
   public:
      enum enum_Coord
      {
         None,
         RowCol,
         Field
      };
      Coord();
      Coord(Coord const& coord);
      Coord(enum_Coord, int y, int x);
      Coord(enum_Coord, char*, int index=1);
      virtual ~Coord();

      int                row()    const;
      int                y()      const;
      int                col()    const;
      int                x()      const;
      FCWString          field()  const;
      int                index()  const;
      enum_Coord         type()   const;

      Bool       operator < (Coord const&)          const;
      Bool       operator > (Coord const&)          const;
      Bool       operator ==(Coord const&)          const;
      void       operator=( Coord const & );

   private:
      Coord::enum_Coord coord_type;
      int               coord_y;
      int               coord_x;
      char*             coord_Name;
      int               coord_index;
      void*             pData;
};

//-----------------------------------------------------------------------------
// Position class                                                             -
//-----------------------------------------------------------------------------
// This class defines the position of a map on the screen as well as fields   -
// within a map.                                                              -
//-----------------------------------------------------------------------------
class Position : public Coord
{
   public:
      Position(Position const& pos);
      Position(enum_Coord type=Coord::RowCol, int y=0, int x=0);
      void       operator=( Position const & );

   private:
      void* pData;
};

//-----------------------------------------------------------------------------
// Size class                                                                 -
//-----------------------------------------------------------------------------
// This class defines the size of a map as well as the size of fields on a    -
// map.                                                                       -
//-----------------------------------------------------------------------------
class Size : public Coord
{
   public:
      Size(Size const& siz);
      Size(enum_Coord type=Coord::RowCol, int cy=0, int cx=0);
      void       operator=( Size const & );

   private:
      void* pData;
};

//-----------------------------------------------------------------------------
// Fold class                                                                 -
//-----------------------------------------------------------------------------
// This class defines the Fold enumeration values that will be used during map-
// field editing.                                                             -
//-----------------------------------------------------------------------------
class Fold
{
   public:
      enum enum_Fold
      {
         Map,
         Yes,
         No
      };
};

//-----------------------------------------------------------------------------
// CursorPosition class                                                       -
//-----------------------------------------------------------------------------
// The CursorPosition class is used to determine the default cursor position  -
// when a map is conversed.                                                   -
//-----------------------------------------------------------------------------
class CursorPosition : public Coord
{
   public:
      CursorPosition();
      CursorPosition(CursorPosition const& cp);
      CursorPosition(enum_Coord type, char* p, int index=1);
      CursorPosition(enum_Coord type, int y, int x);
      void       operator=( CursorPosition const & );

   private:
      void* pData;
};

//-----------------------------------------------------------------------------
// Justify class                                                              -
//-----------------------------------------------------------------------------
// The Attribute class is used to define the enumerations that are used to    -
// set and query the field's justification.                                   -
//-----------------------------------------------------------------------------
class Justify
{
   public:
      enum enum_Justify
      {
         None,
         Left,
         Right
      };
};

//-----------------------------------------------------------------------------
// InputRequired class                                                        -
//-----------------------------------------------------------------------------
// The InputRequired class is used to define if input is required in an edit  -
// field. If input is required, a user message number can be supplied as well.-
//-----------------------------------------------------------------------------
class InputRequired
{
   public:
      InputRequired (Bool bRequired=False, int msgnum=0);
      InputRequired (InputRequired const&);
      ~InputRequired();

      Bool      isRequired() const;
      int       msgnum()     const;
      void       operator=( InputRequired const & );

   private:
      Bool      ir_bRequired;
      int       ir_msgnum;
      void*     pData;
};

//-----------------------------------------------------------------------------
// MinInput class                                                             -
//-----------------------------------------------------------------------------
// The MinInput class is used to define if a minimum amount of input is       -
// required in an edit field. This means that is input is supplied in a field -
// a minimum number of characters must be supplied. A user message number may -
// also be supplied.                                                          -
//-----------------------------------------------------------------------------
class MinInput
{
   public:
      MinInput(int min=0, int msgnum=0);
      MinInput(MinInput const &);
      ~MinInput();

      int MinimumInput() const;
      int msgnum()       const;
      void       operator=( MinInput const & );

   private:
      int     mi_minimum;
      int     mi_msgnum;
      void*   pData;
};

//-----------------------------------------------------------------------------
// EditRoutine class                                                          -
//-----------------------------------------------------------------------------
// The EditRoutine class defines the edit routine for a map field. The edit   -
// routine can be a modulo edit, table edit, or statement group edit.         -
// A user message number may also be supplied.                                -
//-----------------------------------------------------------------------------
class EditRoutine
{
   public:
      enum enum_EditRoutine
      {
         None,
         StatementGroup,
         Table,
         Other
      };

      EditRoutine(enum_EditRoutine t=EditRoutine::None, char* pName=NULL, int msgno=0);
      EditRoutine(EditRoutine const &);
      ~EditRoutine();

      FCWString        Name()   const;
      int              msgnum() const;
      enum_EditRoutine Type()   const;
      void       operator=( EditRoutine const & );

   private:
      EditRoutine::enum_EditRoutine   er_type;
      char*                           er_Name;
      int                             er_msgnum;
      void*                           pData;
};

//-----------------------------------------------------------------------------
// Sign class                                                                 -
//-----------------------------------------------------------------------------
// The Sign class is used to define the enumerations that are used to         -
// set and query a numeric field's sign.                                      -
//-----------------------------------------------------------------------------
class Sign
{
   public:
      enum enum_Sign
      {
         None,
         Leading,
         Trailing
      };
};

//-----------------------------------------------------------------------------
// NumEdits class                                                             -
//-----------------------------------------------------------------------------
// The NumEdits class is used to define the numeric editing that will be done -
// for output and input edits for a numeric field.                            -
//-----------------------------------------------------------------------------
class NumEdits
{
   public:
      NumEdits();
      NumEdits(NumEdits const&);
      NumEdits(int              decimals,
               Bool             bCurrency,
               Sign::enum_Sign  sign,
               Bool             bSeparators,
               int              msgnum=0);
      ~NumEdits();

      int              msgnum() const;

      FCWString        format  (FCWString const&, int, char, Bool bZeroEdit=False);
      int              validate(FCWString const&, FCWString&, int);
      void             operator=( NumEdits const & );

   private:
      int              ne_decimals;
      Bool             ne_currency;
      Sign::enum_Sign  ne_sign;
      Bool             ne_separators;
      int              ne_msgnum;
      void*            pData;

      int              decimals()   const;
      Bool             currency()   const;
      Sign::enum_Sign  sign()       const;
      Bool             separators() const;
};

//-----------------------------------------------------------------------------
// MixEdits class                                                             -
//-----------------------------------------------------------------------------
// The MixEdits class is used to define if the SOSI editing is used for       -
// the field and if there is a data type user message associated with this    -
// edit check.                                                                -
//-----------------------------------------------------------------------------
class MixEdits
{
   public:
      MixEdits(Bool bSOSI=False, int msgnum=0);
      MixEdits(MixEdits const &);
      ~MixEdits();

      Bool SOSIEditing() const;
      int  msgnum()      const;
      void       operator=( MixEdits const & );

   private:
      int     me_SOSI;
      int     me_msgnum;
      void*   pData;
};

//-----------------------------------------------------------------------------
// MinMaxValue class                                                          -
//-----------------------------------------------------------------------------
// The MinMaxValue class defines the minimum and maximum value range edit that-
// can be done for numeric fields.                                            -
//-----------------------------------------------------------------------------
class MinMaxValue
{
    public:
       MinMaxValue (char* pMin = NULL,
                    char* pMax = NULL,
                    int   msgnum = 0);
       MinMaxValue (MinMaxValue const &);
       ~MinMaxValue();

       char* pMin()   const;
       char* pMax()   const;
       int   msgnum() const;
      void       operator=( MinMaxValue const & );

    private:
       char*   mmv_pMin;
       char*   mmv_pMax;
       int     mmv_msgnum;
       void*   pData;
};

//-----------------------------------------------------------------------------
// Field class                                                                -
//-----------------------------------------------------------------------------
// The Field class is the base class for all fields including static and      -
// edit fields.                                                               -
//-----------------------------------------------------------------------------
class eMap;                            /* forward references               */
class iMap;

class Field
{
   public:
      enum Field_Type
      {
         Sbcs  = 1,
         Mixed = 2,
         Dbcs  = 4
      };


      virtual FCWString formatOut(iMap*);
      virtual Attribute getAttribute(iMap*);

      virtual Bool       isEditField() { return False; }
      virtual Field_Type fieldType()   { return Sbcs; }

      Size  const&      size();
      Position const&   position();

   protected:
      Field();
      Field(eMap&             rmap,
            char*             pText,
            int               leadingSpace,
            Position const&   pos,
            Size     const&   size,
            Attribute const&  attr);
      virtual ~Field();

      void           Set(eMap&             rmap,
                         char*             pText,
                         int               leadingSpace,
                         Position const&   pos,
                         Size     const&   size,
                         Attribute const&  attr);

      eMap&          map();
      FCWString      sosiAdjust(FCWString const&);

   private:
      eMap*          f_peMap;
      char*          f_pText;
      int            f_leading;
      Position       f_position;
      Size           f_size;
      Attribute      f_attribute;
      void*          pData;

      char*          pText();
      int            leading();
      Attribute&     attribute();
};

//-----------------------------------------------------------------------------
// STATICCHA class                                                            -
//-----------------------------------------------------------------------------
// The STATICCHA class defines a static sbcs field on the map.                -
//-----------------------------------------------------------------------------
class STATICCHA : public Field
{
   public:
      STATICCHA();
      STATICCHA(eMap &              rmap,
                char*               pText,
                int                 leadingSpace,
                Position const&     pos,
                Size     const&     size,
                Attribute const&    attr);

      void      Set(eMap &              rmap,
                    char*               pText,
                    int                 leadingSpace,
                    int                 posRow,
                    int                 posCol,
                    int                 size,
                    unsigned char       baseAttr,
                    Color::enum_Color   colorAttr,
                    Hilite::enum_Hilite hiliteAttr,
                    unsigned char       outlineAttr);

   private:
      void*     pData;
};

//-----------------------------------------------------------------------------
// STATICMIX class                                                            -
//-----------------------------------------------------------------------------
// The STATICMIX class defines a mixed sbcs and dbcs field on the map.        -
//-----------------------------------------------------------------------------
class STATICMIX : public Field
{
   public:
      STATICMIX();
      STATICMIX(eMap &              rmap,
                char*               pText,
                int                 leadingSpace,
                Position const&     pos,
                Size     const&     size,
                Attribute const&    attr);

      void               Set(eMap &              rmap,
                             char*               pText,
                             int                 leadingSpace,
                             int                 posRow,
                             int                 posCol,
                             int                 size,
                             unsigned char       baseAttr,
                             Color::enum_Color   colorAttr,
                             Hilite::enum_Hilite hiliteAttr,
                             unsigned char       outlineAttr);

      virtual Field_Type fieldType()   { return Mixed; }
   private:
      void*     pData;
};

//-----------------------------------------------------------------------------
// STATICDBCS class                                                           -
//-----------------------------------------------------------------------------
// The STATICDBCS class defines a dbcs constant field on the map.             -
//-----------------------------------------------------------------------------
class STATICDBCS : public Field
{
   public:
      STATICDBCS();
      STATICDBCS(eMap &              rmap,
                 char*               pText,
                 int                 leadingSpace,
                 Position const&     pos,
                 Size     const&     size,
                 Attribute const&    attr);

      void               Set(eMap &              rmap,
                             char*               pText,
                             int                 leadingSpace,
                             int                 posRow,
                             int                 posCol,
                             int                 size,
                             unsigned char       baseAttr,
                             Color::enum_Color   colorAttr,
                             Hilite::enum_Hilite hiliteAttr,
                             unsigned char       outlineAttr);

      virtual Field_Type fieldType()   { return Dbcs; }
   private:
      void*     pData;
};

//-----------------------------------------------------------------------------
// EditField class                                                            -
//-----------------------------------------------------------------------------
// The EditField class is the base class for all the edit fields on a map.    -
//-----------------------------------------------------------------------------
class Driver;                          /* forward reference                */
class FieldData;                       /* forward reference                */

class EditField_Data;

class EditField : public Field
{
   protected:

      EditField();
      EditField(eMap&                  rmap,
                char*                  pName,
                char*                  pText,
                int                    leadingSpace,
                Position const&        pos,
                Size     const&        size,
                Attribute const&       attr,
                int                    editOrder,
                int                    index,
                char                   fillchar,
                Justify::enum_Justify  justify,
                InputRequired const &  inputReq,
                MinInput const &       minInput,
                EditRoutine const &    editRoutine);
      virtual ~EditField();

      void              Set(eMap&                  rmap,
                            char*                  pName,
                            char*                  pText,
                            int                    leadingSpace,
                            Position const&        pos,
                            Size     const&        size,
                            Attribute const&       attr,
                            int                    editOrder,
                            int                    index,
                            char                   fillchar,
                            Justify::enum_Justify  justify,
                            InputRequired const &  inputReq,
                            MinInput const &       minInput,
                            EditRoutine const &    editRoutine);

      char              fillChar();
      Bool              isDBCSField();
      FCWString         justify(FCWString const&, char fill=' ');
      Justify::enum_Justify  justify();

      Bool              isRequired();
      int               RequiredMsg();
      int               MinimumInput();
      int               MinimumInputMsg();

      virtual int       inputEdits(iMap*, FCWString const&) = 0;

      virtual int       minInput(FCWString const&, iMap*);
      virtual int       modulus (FCWString const&);
      virtual int       editRoutine(FCWString const&);
      virtual Bool      fieldIsBlank(iMap*);
      virtual int       foldEdits(FCWString const&, FCWString&);

      virtual Bool      foldInput();

   public:

      virtual Bool      isEditField() { return True; }
      int               editOrder();
      int               index();
      FCWString         Name();
      Bool              isCursorField(iMap*);
      void              setCursor(iMap*, Bool);
      void              resetStatus(iMap*);
      Bool              hasBeenModified(iMap*);
      void              setModified(iMap*, Bool);

      virtual int       formatInput(iMap*, FCWString const&, Bool);
      virtual Attribute getAttribute(iMap*);

   private:
      char*                  ef_Name;
      int                    ef_editOrder;
      int                    ef_index;
      char                   ef_fillChar;
      Justify::enum_Justify  ef_justify;
      InputRequired          ef_inputRequired;
      MinInput               ef_minInput;
      EditRoutine            ef_editRoutine;
      void*                  pData;

      InputRequired&         inputRequired();
      MinInput&              minInput();
      EditRoutine&           editRoutine();
};

//-----------------------------------------------------------------------------
// DateMask class                                                             -
//-----------------------------------------------------------------------------
// The DateMask class is the base class for date masks                        -
//-----------------------------------------------------------------------------
class DateMask
{
   public:
      DateMask(int fldsize=0, char* pMask=NULL);
      ~DateMask();
      void       operator=( DateMask const & );

      FCWString const&  Mask()        const;
      int               MonthPos()    const;
      int               MonthRelPos() const;
      int               MonthLen()    const;
      int               MonthSeg()    const;
      int               YearPos()     const;
      int               YearRelPos()  const;
      int               YearLen()     const;
      int               YearSeg()     const;
      int               DayPos()      const;
      int               DayRelPos()   const;
      int               DayLen()      const;
      int               DaySeg()      const;
      int               Segments()    const;
      Bool              isGregorian() const;

      enum MaskType
      {
         ShortJulian = 0,
         LongJulian  = 1,
         ShortGreg   = 2,
         LongGreg    = 3
      };

   private:
      DateMask::MaskType  dm_Type;     /* Date Type                        */
      FCWString           dm_Mask;                     /* The original mask*/
      int                 dm_iMonth;/* Index of the start of the month segment*/
      int                 dm_irMonth;/* index of month segment without separators*/
      int                 dm_lMonth;   /* length of the month segment      */
      int                 dm_sMonth;   /* sequence of the month segment    */
      int                 dm_iYear;/* Index of the start of the year segment*/
      int                 dm_irYear;/* index of year segment without separators*/
      int                 dm_lYear;    /* length of the year segment       */
      int                 dm_sYear;    /* sequence of the year segment     */
      int                 dm_iDay; /* Index of the start of the day segment*/
      int                 dm_irDay;/* index of day segment without separators*/
      int                 dm_lDay;     /* length of the day segment        */
      int                 dm_sDay;     /* sequence of the day segment      */
      int                 dm_iSeg;     /* number of segments in the mask   */
      void*               pData;

      MaskType          Type()        const;
};

//-----------------------------------------------------------------------------
// EDITNUM class                                                              -
//-----------------------------------------------------------------------------
// The EDITNUM class defines a numeric edit field on a map.                   -
//-----------------------------------------------------------------------------
class EDITNUM : public EditField
{
   public:

      EDITNUM();
      EDITNUM (eMap&                  rmap,
               char*                  pName,
               char*                  pText,
               int                    leadingSpace,
               Position const&        pos,
               Size     const&        size,
               Attribute const&       attr,
               int                    editOrder,
               int                    index,
               char                   fillchar,
               Justify::enum_Justify  justify,
               InputRequired const &  inputReq,
               MinInput const &       minInput,
               EditRoutine const &    editRoutine,
               Bool                   zeroEdit,
               NumEdits const &       numEdits,
               MinMaxValue const &    minmax);

      EDITNUM (eMap&                  rmap,
               char*                  pName,
               char*                  pText,
               int                    leadingSpace,
               Position const&        pos,
               Size     const&        size,
               Attribute const&       attr,
               int                    editOrder,
               int                    index,
               char                   fillchar,
               Justify::enum_Justify  justify,
               InputRequired const &  inputReq,
               MinInput const &       minInput,
               EditRoutine const &    editRoutine,
               Bool                   zeroEdit,
               char*                  datemask);
      ~EDITNUM();

      void Set (eMap&                  rmap,
                char*                  pName,
                char*                  pText,
                int                    leadingSpace,
                int                    row,
                int                    col,
                int                    size,
                unsigned char          baseAttr,
                Color::enum_Color      colorAttr,
                Hilite::enum_Hilite    hiliteAttr,
                unsigned char          outlineAttr,
                int                    editOrder,
                int                    index,
                char                   fillchar,
                Justify::enum_Justify  justify,
                Bool                   bInputReq,
                int                    irMsgNum,
                int                    miMin,
                int                    miMsgno,
                EditRoutine::enum_EditRoutine       erType,
                char*                  erName,
                int                    erMsgno,
                Bool                   zeroEdit,
                int                    Decimals,
                Bool                   bCurrency,
                Sign::enum_Sign        sign,
                Bool                   bSeparators,
                int                    neMsgno,
                char*                  pMin,
                char*                  pMax,
                int                    mmMsgno);

      void Set (eMap&                  rmap,
                char*                  pName,
                char*                  pText,
                int                    leadingSpace,
                int                    row,
                int                    col,
                int                    size,
                unsigned char          baseAttr,
                Color::enum_Color      colorAttr,
                Hilite::enum_Hilite    hiliteAttr,
                unsigned char          outlineAttr,
                int                    editOrder,
                int                    index,
                char                   fillchar,
                Justify::enum_Justify  justify,
                Bool                   bInputReq,
                int                    irMsgNum,
                int                    miMin,
                int                    miMsgno,
                EditRoutine::enum_EditRoutine       erType,
                char*                  erName,
                int                    erMsgno,
                Bool                   zeroEdit,
                char*                  datemask);

      virtual FCWString  formatOut(iMap*);

   protected:

      virtual int        inputEdits(iMap*, FCWString const&);
      virtual int        minInput(FCWString const&, iMap*);
      virtual int        modulus (FCWString const&);
      virtual Bool       fieldIsBlank(iMap*);
              FCWString  outputEdits(FCWString const&);
              int        numInputEdits(FCWString const&, FCWString&);
              FCWString  dateoEdits(FCWString const&);
              Bool       dateiEdits(FCWString const&, FCWString&);
              int        rangeCheck(FCWString&);

   private:
      DateMask           en_datemask;
      Bool               en_zeroEdit;
      NumEdits           en_numEdits;
      MinMaxValue        en_minmax;
      void*              pData;

      DateMask&          datemask();
      Bool               zeroEdit();
      NumEdits&          numEdits();
      MinMaxValue&       minMaxValue();
};

//-----------------------------------------------------------------------------
// EDITCHA class                                                              -
//-----------------------------------------------------------------------------
// The EDITCHA class defines a CHA edit field on a map.                       -
//-----------------------------------------------------------------------------
class EDITCHA : public EditField
{
   public:
      EDITCHA();
      EDITCHA (eMap&                  rmap,
               char*                  pName,
               char*                  pText,
               int                    leadingSpace,
               Position const&        pos,
               Size     const&        size,
               Attribute const&       attr,
               int                    editOrder,
               int                    index,
               char                   fillchar,
               Justify::enum_Justify  justify,
               InputRequired const &  inputReq,
               MinInput const &       minInput,
               EditRoutine const &    editRoutine,
               Fold::enum_Fold        fold = Fold::Map,
               Bool                   bHexEdits = False);

      EDITCHA (eMap&                  rmap,
               char*                  pName,
               char*                  pText,
               int                    leadingSpace,
               Position const&        pos,
               Size     const&        size,
               Attribute const&       attr,
               int                    editOrder,
               int                    index,
               char                   fillchar,
               Justify::enum_Justify  justify,
               InputRequired const &  inputReq,
               MinInput const &       minInput,
               EditRoutine const &    editRoutine,
               char*                  datemask);
      ~EDITCHA();

      void Set(eMap&                  rmap,
               char*                  pName,
               char*                  pText,
               int                    leadingSpace,
               int                    posrow,
               int                    poscol,
               int                    size,
               unsigned char          baseAttr,
               Color::enum_Color      colorAttr,
               Hilite::enum_Hilite    hiliteAttr,
               unsigned char          outlineAttr,
               int                    editOrder,
               int                    index,
               char                   fillchar,
               Justify::enum_Justify  justify,
               Bool                   bInputReq,
               int                    irMsgNum,
               int                    miMin,
               int                    miMsgno,
               EditRoutine::enum_EditRoutine       erType,
               char*                  erName,
               int                    erMsgno,
               Fold::enum_Fold        fold = Fold::Map,
               Bool                   bHexEdits = False);

      void Set(eMap&                  rmap,
               char*                  pName,
               char*                  pText,
               int                    leadingSpace,
               int                    posrow,
               int                    poscol,
               int                    size,
               unsigned char          baseAttr,
               Color::enum_Color      colorAttr,
               Hilite::enum_Hilite    hiliteAttr,
               unsigned char          outlineAttr,
               int                    editOrder,
               int                    index,
               char                   fillchar,
               Justify::enum_Justify  justify,
               Bool                   bInputReq,
               int                    irMsgNum,
               int                    miMin,
               int                    miMsgno,
               EditRoutine::enum_EditRoutine       erType,
               char*                  erName,
               int                    erMsgno,
               char*                  datemask);

      virtual FCWString  formatOut(iMap*);

   protected:

      virtual int        inputEdits(iMap*, FCWString const&);
      virtual int        modulus   (FCWString const&);
              int        hexEdits  (FCWString&);
              Bool       dateiEdits(FCWString const&, FCWString&);

      virtual Bool       foldInput();

              FCWString  outputEdits(FCWString const&);
              FCWString  dateoEdits (FCWString const&);

   private:
      DateMask         ec_datemask;
      Fold::enum_Fold  ec_fold;
      Bool             ec_hexEdits;
      void*            pData;

      DateMask&        datemask();
      Fold::enum_Fold  fold();
      Bool             hexEdits();
};

//-----------------------------------------------------------------------------
// EDITMIX class                                                              -
//-----------------------------------------------------------------------------
// The EDITMIX class defines a CHA edit field on a map.                       -
//-----------------------------------------------------------------------------
class EDITMIX : public EditField
{
   public:
      EDITMIX();
      EDITMIX (eMap&                  rmap,
               char*                  pName,
               char*                  pText,
               int                    leadingSpace,
               Position const&        pos,
               Size     const&        size,
               Attribute const&       attr,
               int                    editOrder,
               int                    index,
               char                   fillchar,
               Justify::enum_Justify  justify,
               InputRequired const &  inputReq,
               MinInput const &       minInput,
               EditRoutine const &    editRoutine,
               Fold::enum_Fold        fold = Fold::Map,
               MixEdits const &       mixedits = MixEdits(False));
      ~EDITMIX();

      void Set(eMap&                  rmap,
               char*                  pName,
               char*                  pText,
               int                    leadingSpace,
               int                    posrow,
               int                    poscol,
               int                    size,
               unsigned char          baseAttr,
               Color::enum_Color      colorAttr,
               Hilite::enum_Hilite    hiliteAttr,
               unsigned char          outlineAttr,
               int                    editOrder,
               int                    index,
               char                   fillchar,
               Justify::enum_Justify  justify,
               Bool                   bInputReq,
               int                    irMsgNum,
               int                    miMin,
               int                    miMsgno,
               EditRoutine::enum_EditRoutine       erType,
               char*                  erName,
               int                    erMsgno,
               Fold::enum_Fold        fold = Fold::Map,
               Bool                   bMixEdits = False,
               int                    meMsgno = 0);

      virtual FCWString  formatOut(iMap*);
      virtual Field_Type fieldType()   { return Mixed; }

   protected:

      virtual int        inputEdits(iMap*, FCWString const&);
              int        mixEdits  (FCWString const&);

      virtual Bool       foldInput();

              FCWString  outputEdits(FCWString const&);

   private:
      Fold::enum_Fold  em_fold;
      MixEdits         em_mixEdits;
      void*            pData;

      Fold::enum_Fold  fold();
      MixEdits&        mixEdits();
};

//-----------------------------------------------------------------------------
// EDITDBCS class                                                             -
//-----------------------------------------------------------------------------
// The EDITDBCS class defines a DBCS edit field on a map.                     -
//-----------------------------------------------------------------------------
class EDITDBCS : public EditField
{
   public:
      EDITDBCS();
      EDITDBCS(eMap&                  rmap,
               char*                  pName,
               char*                  pText,
               int                    leadingSpace,
               Position const&        pos,
               Size     const&        size,
               Attribute const&       attr,
               int                    editOrder,
               int                    index,
               char                   fillchar,
               Justify::enum_Justify  justify,
               InputRequired const &  inputReq,
               MinInput const &       minInput,
               EditRoutine const &    editRoutine);

      void Set(eMap&                  rmap,
               char*                  pName,
               char*                  pText,
               int                    leadingSpace,
               int                    posrow,
               int                    poscol,
               int                    size,
               unsigned char          baseAttr,
               Color::enum_Color      colorAttr,
               Hilite::enum_Hilite    hiliteAttr,
               unsigned char          outlineAttr,
               int                    editOrder,
               int                    index,
               char                   fillchar,
               Justify::enum_Justify  justify,
               Bool                   bInputReq,
               int                    irMsgNum,
               int                    miMin,
               int                    miMsgno,
               EditRoutine::enum_EditRoutine       erType,
               char*                  erName,
               int                    erMsgno);

      virtual FCWString  formatOut(iMap*);
      virtual Field_Type fieldType()   { return Dbcs; }

   protected:

      virtual Bool       fieldIsBlank(iMap*);
      virtual int        inputEdits(iMap*, FCWString const&);

   private:
      void*    pData;

};

//-----------------------------------------------------------------------------
// eMap class                                                                 -
//-----------------------------------------------------------------------------
// The eMap class defines the external map definition. It contains all of the -
// external information about a map.                                          -
//-----------------------------------------------------------------------------
class MapGroup;                        /* forward reference                */

class eMap
{
   public:

      eMap(MapGroup&               mg,
           char*                   pName,
           Position       const&   pos,
           Size           const&   size,
           CursorPosition const&   cursor,
           char*                   pHelpName,
           Key::enum_Key           helpkey,
           ByPassKeys const&       bypasskeys,
           Fold::enum_Fold         fold,
           Bool                    bSOSI=False,
           Bool                    bDBCS=False);

      virtual ~eMap();

      virtual Bool  isPrintmap()    { return False; }
      virtual Bool  isTerminalmap() { return False; }

      Bool       isFloatMap();
      Bool       isDBCSMap();

      void       addField(Field *);
      void       addField(EditField *);

      EditField* EZEMSG();

      Position&        position();
      Size&            size();
      CursorPosition&  cursor();

      ByPassKeys const& BypassKeys();
      Bool              hasHelpMap();
      Key::enum_Key     HelpKey();
      FCWString         helpMap();
      FCWString         Name();
      Bool              foldInput();

      MapGroup&  mapGroup();

      SLIST(Field*)&     list_pField();
      SLIST(EditField*)& list_pEditFieldeo();
      SLIST(EditField*)& list_pEditFieldrc();

   private:
      MapGroup*          map_MapGroup;
      char*              map_Name;
      Position           map_position;
      Size               map_size;
      CursorPosition     map_Cursor;
      char*              map_HelpName;
      Key::enum_Key      map_HelpKey;
      ByPassKeys         map_BypassKeys;
      Fold::enum_Fold    map_Fold;
      Bool               map_bSOSI;
      Bool               map_bDBCS;
      EditField*         map_pEZEMSG;
      Bool               map_isFloat;

      SLIST(Field*)      map_list_pFieldrc; /* order based on row/col order*/
      SLIST(EditField*)  map_list_pEditFieldrc;/* order based on row/col order*/
      SLIST(EditField*)  map_list_pEditFieldeo;/* order based on edit order*/
      void*              pData;

      Bool               bSOSI();
};

//-----------------------------------------------------------------------------
// TerminalMap class                                                          -
//-----------------------------------------------------------------------------
// The TerminalMap class defines the external information about terminal maps -
//-----------------------------------------------------------------------------
class TerminalMap : public eMap
{
   public:
      TerminalMap(MapGroup&                mg,
                  char*                    pName,
                  Position        const&   pos,
                  Size            const&   size,
                  CursorPosition  const&   cursor,
                  char*                    pHelpName,
                  Key::enum_Key            helpkey,
                  ByPassKeys      const&   bypasskeys,
                  Fold::enum_Fold          fold,
                  Bool                     bSOSI=False,
                  Bool                     bDBCS=False);

      TerminalMap(MapGroup&               mg,
                  char*                   pName,
                  int                     posrow,
                  int                     poscol,
                  int                     sizerow,
                  int                     sizecol,
                  char*                   pCursorFldName,
                  int                     CursorFldIndx,
                  char*                   pHelpName,
                  Key::enum_Key           helpkey,
                  ByPassKeys const&       bypasskeys,
                  Fold::enum_Fold         fold,
                  Bool                    bSOSI=False,
                  Bool                    bDBCS=False);
      virtual ~TerminalMap();

      virtual Bool  isTerminalmap() { return True; }

   private:
      void*   pData;
};

//-----------------------------------------------------------------------------
// PrintMap class                                                             -
//-----------------------------------------------------------------------------
// The PrintMap class defines the external information about printer maps.    -
//-----------------------------------------------------------------------------
class PrintMap : public eMap
{
   public:
      PrintMap(MapGroup&                mg,
               char*                    pName,
               Position        const&   pos,
               Size            const&   size,
               CursorPosition  const&   cursor,
               char*                    pHelpName,
               Key::enum_Key            helpkey,
               ByPassKeys      const&   bypasskeys,
               Fold::enum_Fold          fold,
               Bool                     bSOSI=False,
               Bool                     bDBCS=False);

      PrintMap(MapGroup&               mg,
               char*                   pName,
               int                     posrow,
               int                     poscol,
               int                     sizerow,
               int                     sizecol,
               char*                   pCursorFldName,
               int                     CursorFldIndx,
               char*                   pHelpName,
               Key::enum_Key           helpkey,
               ByPassKeys const&       bypasskeys,
               Fold::enum_Fold         fold,
               Bool                    bSOSI=False,
               Bool                    bDBCS=False);

      virtual ~PrintMap();

      virtual Bool  isPrintmap() { return True; }

   private:
      void*   pData;
};

//-----------------------------------------------------------------------------
// FloatArea class                                                            -
//-----------------------------------------------------------------------------
// This class defines the floating area. There is one floating area for print -
// maps and one floating area for terminal maps.                              -
//-----------------------------------------------------------------------------
class FloatArea
{
   public:
      FloatArea(Position const& pos, Size const& size);
      FloatArea(int posrow, int poscol, int sizerow, int sizecol);
      ~FloatArea();

      Position const & position();
      Size     const & size();

   private:
      Position   fa_position;
      Size       fa_size;
      void*      pData;
};

//-----------------------------------------------------------------------------
// MapGroup class                                                             -
//-----------------------------------------------------------------------------
// This class defines the map group class. This class basically serves as an  -
// entry point into the map group and access can be obtained to maps through  -
// the map group.                                                             -
//-----------------------------------------------------------------------------
class MapGroup
{
   public:

      MapGroup(char* pName,
               FloatArea * pTermArea  = NULL,
               FloatArea * pPrintArea = NULL);
      virtual ~MapGroup();

      eMap*             find(FCWString const& mapname);
      FloatArea*        TerminalFloatArea();
      FloatArea*        PrinterFloatArea();

      void       addMap(eMap*);
      eMap*      find(FCWString const&)   const;

      FCWString  Name();

   private:
      char*             mg_Name;
      FloatArea *       mg_pTermArea;
      FloatArea *       mg_pPrintArea;

      LIST(eMap *)      mg_list_peMap;
      void*             pData;

      LIST(eMap *)&     list_peMap();
};

#endif
