/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * (C) Copyright The MITRE Corporation 1999  All rights reserved.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * The program provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * Please see release.txt distributed with this file for more information.
 *
 * Contributor(s): Tom Kneeland
 *                 Peter Van der Beken <peter.vanderbeken@pandora.be>
 *
 */

/**
 * Implementation of the wrapper class to convert the Mozilla nsIDOMElement
 * interface into a TransforMiiX Element interface.
 */

#include "mozilladom.h"
#include "nsIDOMElement.h"
#include "nsINodeInfo.h"
#include "nsString2.h"
#include "nsIDOMAttr.h"
#include "nsIDOMNamedNodeMap.h"

extern nsINameSpaceManager* gTxNameSpaceManager;

/**
 * Construct a wrapper with the specified Mozilla object and document owner.
 *
 * @param aElement the nsIDOMElement you want to wrap
 * @param aOwner the document that owns this object
 */
Element::Element(nsIDOMElement* aElement, Document* aOwner) :
        Node(aElement, aOwner)
{
    nsCOMPtr<nsIContent> cont(do_QueryInterface(aElement));
    NS_ASSERTION(cont, "Element doesn't implement nsIContent");
    nsCOMPtr<nsINodeInfo> nodeInfo;
    cont->GetNodeInfo(*getter_AddRefs(nodeInfo));
    NS_ASSERTION(nodeInfo, "a element without nodeinfo");
    nodeInfo->GetNamespaceID(mNamespaceID);
}

/**
 * Destructor
 */
Element::~Element()
{
}

/**
 * Call nsIDOMElement::SetAttributeNS to set an attribute to the specified
 * value.
 *
 * @param aNamespaceURI the URI of the namespace of the attribute
 * @param aName the name of the attribute you want to set
 * @param aValue the value you want to set the attribute to
 */
void Element::setAttributeNS(const nsAString& aNamespaceURI,
                             const nsAString& aName,
                             const nsAString& aValue)
{
    NSI_FROM_TX(Element);
    nsElement->SetAttributeNS(aNamespaceURI, aName, aValue);
}

/**
 * Call nsIContent::GetAttr for the localname and nsID.
 */
MBool Element::getAttr(nsIAtom* aLocalName, PRInt32 aNSID,
                       nsAString& aValue)
{
    nsCOMPtr<nsIContent> cont(do_QueryInterface(mMozObject));
    NS_ASSERTION(cont, "Element doesn't implement nsIContent");
    if (!cont || !cont->HasAttr(aNSID, aLocalName)) {
        aValue.Truncate();
        return MB_FALSE;
    }
    nsresult rv;
    rv = cont->GetAttr(aNSID, aLocalName, aValue);
    NS_ENSURE_SUCCESS(rv, MB_FALSE);
    return MB_TRUE;
}

Node*
Element::getAttrNode(nsIAtom* aLocalName, PRInt32 aNSID)
{
#if 1
    nsCOMPtr<nsIContent> cont(do_QueryInterface(mMozObject));
    PRInt32 i, numAttr = 0;
    cont->GetAttrCount(numAttr);
    
    for (i = 0; i < numAttr; ++i) {
        nsCOMPtr<nsIAtom> localName, prefix;
        PRInt32 nsid;
        cont->GetAttrNameAt(i, nsid, *getter_AddRefs(localName),
                            *getter_AddRefs(prefix));
        if (nsid == aNSID && localName == aLocalName) {
            nsCOMPtr<nsIDOMNamedNodeMap> map;
            NSI_FROM_TX(Node);
            nsresult rv = nsNode->GetAttributes(getter_AddRefs(map));
            NS_ENSURE_SUCCESS(rv, nsnull);
            
            nsCOMPtr<nsIDOMNode> attrNode;
            rv = map->Item(i, getter_AddRefs(attrNode));
            NS_ENSURE_SUCCESS(rv, nsnull);
            
            nsCOMPtr<nsIDOMAttr> attr = do_QueryInterface(attrNode);
            return mOwnerDocument->createAttribute(attr);
        }
    }

    return nsnull;
#endif

#if 0
    nsAutoString name, ns;
    aLocalName->ToString(name);
    gTxNameSpaceManager->GetNameSpaceURI(aNSID, ns);

    NSI_FROM_TX(Element);
    nsCOMPtr<nsIDOMAttr> attr;
    nsElement->GetAttributeNodeNS(ns, name, getter_AddRefs(attr));

    if (!attr) {
        return nsnull;
    }

    return mOwnerDocument->createAttribute(attr);*/
#endif
}

/**
 * Call nsIContent::GetAttr for the localname and nsID.
 */
MBool Element::hasAttr(nsIAtom* aLocalName, PRInt32 aNSID)
{
    nsCOMPtr<nsIContent> cont(do_QueryInterface(mMozObject));
    NS_ASSERTION(cont, "Element doesn't implement nsIContent");
    if (!cont) {
        return MB_FALSE;
    }
    return cont->HasAttr(aNSID, aLocalName);
}

/**
 * Returns the local name atomized
 *
 * @return the node's localname atom
 */
MBool Element::getLocalName(nsIAtom** aLocalName)
{
    NS_ENSURE_TRUE(aLocalName, MB_FALSE);
    nsCOMPtr<nsIContent> cont(do_QueryInterface(mMozObject));
    NS_ASSERTION(cont, "Element doesn't implement nsIContent");
    if (!cont) {
        return MB_FALSE;
    }
    cont->GetTag(*aLocalName);
    return MB_TRUE;
}
