/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is TransforMiiX XSLT processor.
 *
 * The Initial Developer of the Original Code is
 * IBM Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2002
 * IBM Corporation. All Rights Reserved.
 *
 * Contributor(s):
 *   IBM Corporation
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "txXPathOptimizer.h"
#include "txAtoms.h"

txXPathOptimizer::txXPathOptimizer(PRBool aCaseInsensitiveNameTests)
    : mCaseInsensitiveNameTests(aCaseInsensitiveNameTests)
{
}

nsresult
txXPathOptimizer::optimize(nsAutoPtr<Expr>& aExpr, Expr::ResultType aResultType)
{
    nsresult rv = aExpr->iterateSubItems(this);
    NS_ENSURE_SUCCESS(rv, rv);

    switch (aExpr->getType()) {
        case Expr::LOCATIONSTEP_EXPR:
        {
            return optimizeStep(aExpr, aResultType);
        }
        case Expr::RELATIONAL_EXPR:
        {
            return optimizeRelationalExpr(aExpr, aResultType);
        }
        case Expr::PATH_EXPR:
        {
            return optimizePathExpr(aExpr, aResultType);
        }
        case Expr::FILTER_EXPR:
        {
            return optimizeFilterExpr(aExpr, aResultType);
        }
    }

    return NS_OK;
}

nsresult
txXPathOptimizer::walkedExpr(nsAutoPtr<Expr>& aExpr,
                             Expr::ResultType aResultType)
{
    return optimize(aExpr, aResultType);
}

nsresult
txXPathOptimizer::walkedNodeTest(nsAutoPtr<txNodeTest>& aNodeTest)
{
    return aNodeTest->iterateSubItems(this);
}

nsresult
txXPathOptimizer::optimizeStep(nsAutoPtr<Expr>& aExpr,
                               Expr::ResultType aResultType)
{
    LocationStep* step = NS_STATIC_CAST(LocationStep*, aExpr.get());

    // Test for @foo type steps.
    txNameTest* nameTest = nsnull;
    if (step->isEmpty() &&
        step->mAxisIdentifier == LocationStep::ATTRIBUTE_AXIS &&
        step->mNodeTest->getType() == txNameTest::NAME_TEST &&
        (nameTest = NS_STATIC_CAST(txNameTest*, step->mNodeTest.get()))->
            mLocalName != txXPathAtoms::_asterix &&
        step->isEmpty()) {
        NS_ASSERTION(nameTest->mNodeType == Node::ATTRIBUTE_NODE,
                     "how could this be");
        txNamedAttributeStep::Type type;
        if (aResultType == Expr::BOOLEAN_RESULT) {
            type = txNamedAttributeStep::HAS_ATTR;
        }
        else if (aResultType == Expr::STRING_RESULT ||
                 aResultType == Expr::NUMBER_RESULT) {
            type = txNamedAttributeStep::ATTR_VALUE;
        }
        else {
            type = txNamedAttributeStep::NODESET;
        }
        
        Expr* newExpr = new txNamedAttributeStep(type, nameTest->mNamespace,
                                                 nameTest->mPrefix,
                                                 nameTest->mLocalName);
        NS_ENSURE_TRUE(newExpr, NS_ERROR_OUT_OF_MEMORY);
        
        aExpr = newExpr;

        return NS_OK; // return since we no longer have a step-object.
    }

    // Test for predicates like [local-name() = current()/some/expr]
    Expr* pred;
    if (!step->isEmpty() && !mCaseInsensitiveNameTests &&
        (pred = NS_STATIC_CAST(Expr*, step->predicates.get(step->predicates
             .getLength() - 1)))->getType() == Expr::RELATIONAL_EXPR) {
        RelationalExpr* rel = NS_STATIC_CAST(RelationalExpr*, pred);
        rel->swapIfRightIsA(Expr::FUNCTIONCALL_EXPR);
        if (rel->mOp == RelationalExpr::EQUAL &&
            rel->mLeftExpr->getType() == Expr::FUNCTIONCALL_EXPR &&
            !(rel->mRightExpr->getContextSensitivity() &
              (Expr::NODE_ONLY_CONTEXT | Expr::NODESET_CONTEXT))/* &&
            !(rel->mRightExpr->getReturnType() &
              ~(Expr::STRING_RESULT | Expr::NODESET_RESULT))*/) {
            FunctionCall* fun = NS_STATIC_CAST(FunctionCall*, rel->mLeftExpr.get());
            nsCOMPtr<nsIAtom> funName;
            fun->getNameAtom(getter_AddRefs(funName));
            if (funName == txXPathAtoms::localName && fun->params.isEmpty()) {
                txLocalNameFilter* lnFilter = new txLocalNameFilter(aExpr, rel->mRightExpr);
                NS_ENSURE_TRUE(lnFilter, NS_ERROR_OUT_OF_MEMORY);

                aExpr = lnFilter;
                delete step->predicates.remove(pred);

                // return since we no longer have a step-object.
                return optimizeStep(lnFilter->mExpr, aResultType);
            }
        }
    }

    // Test for predicates that can be combined into the nodetest
    while (!step->isEmpty() &&
           !((pred = NS_STATIC_CAST(Expr*, step->predicates.get(0)))
                 ->getReturnType() & Expr::NUMBER_RESULT) &&
           !(pred->getContextSensitivity() & Expr::NODESET_CONTEXT)) {
        txNodeTest* predTest = new txPredicatedNodeTest(step->mNodeTest,
                                                        pred);
        NS_ENSURE_TRUE(predTest, NS_ERROR_OUT_OF_MEMORY);

        step->predicates.remove(pred);
        step->mNodeTest = predTest;
    }

    return NS_OK;
}

nsresult
txXPathOptimizer::optimizeRelationalExpr(nsAutoPtr<Expr>& aExpr,
                                         Expr::ResultType aResultType)
{
    RelationalExpr* rel = NS_STATIC_CAST(RelationalExpr*, aExpr.get());

    // Evaluate nodesets first to try to shortcut if it is empty
    if ((rel->mRightExpr->getReturnType() & Expr::NODESET_RESULT) &&
        !(rel->mLeftExpr->getReturnType() & Expr::NODESET_RESULT)) {
        rel->mEvalRightFirst = PR_TRUE;
    }

    // Test for @foo = 'bar'
    rel->swapIfLeftIsA(Expr::LITERAL_EXPR);
    if (rel->mOp == RelationalExpr::EQUAL &&
        rel->mLeftExpr->getType() == Expr::NAMEDATTRIBUTESTEP_EXPR &&
        rel->mRightExpr->getType() == Expr::LITERAL_EXPR &&
        rel->mRightExpr->getReturnType() == Expr::STRING_RESULT) {
        txNamedAttributeStep* attrStep =
            NS_STATIC_CAST(txNamedAttributeStep*, rel->mLeftExpr.get());
        txLiteralExpr* lit =
            NS_STATIC_CAST(txLiteralExpr*, rel->mRightExpr.get());

        StringResult* strRes =
            NS_STATIC_CAST(StringResult*,
                           NS_STATIC_CAST(txAExprResult*,
                                          lit->mValue));
        attrStep->mString = new nsString(strRes->mValue);
        NS_ENSURE_TRUE(attrStep->mString, NS_ERROR_OUT_OF_MEMORY);

        attrStep->mType = txNamedAttributeStep::VALUE_COMPARISON;
        aExpr = rel->mLeftExpr.forget();

        return NS_OK;
    }

    return NS_OK;
}

nsresult
txXPathOptimizer::optimizePathExpr(nsAutoPtr<Expr>& aExpr,
                                   Expr::ResultType aResultType)
{
    nsresult rv = NS_OK;
    PathExpr* path = NS_STATIC_CAST(PathExpr*, aExpr.get());

    PathExpr::PathExprItem* pxi;
    txListIterator iter(&path->expressions);
    // look for steps like "//foo" that can be turned into "/descendant::foo"
    while ((pxi = NS_STATIC_CAST(PathExpr::PathExprItem*, iter.next()))) {
        if (pxi->pathOp == PathExpr::DESCENDANT_OP) {
            Expr* expr = pxi->expr;
            if (expr->getType() == Expr::LOCALNAMEFILTER_EXPR) {
                expr = NS_STATIC_CAST(txLocalNameFilter*, expr)->mExpr;
            }
            LocationStep* step;
            if (expr->getType() == Expr::LOCATIONSTEP_EXPR &&
                (step = NS_STATIC_CAST(LocationStep*, expr))->isEmpty() &&
                step->mAxisIdentifier == LocationStep::CHILD_AXIS) {
                step->mAxisIdentifier = LocationStep::DESCENDANT_AXIS;
                pxi->pathOp = PathExpr::RELATIVE_OP;
            }
        }
    }

    // look for paths starting with a '.' step
    iter.reset();
    pxi = NS_STATIC_CAST(PathExpr::PathExprItem*, iter.next());
    if (pxi->expr->getType() == Expr::LOCATIONSTEP_EXPR) {
        LocationStep* step = NS_STATIC_CAST(LocationStep*, pxi->expr.get());
        if (step->isEmpty() &&
            step->mAxisIdentifier == LocationStep::SELF_AXIS &&
            step->mNodeTest->getType() == txNameTest::NODETYPE_TEST &&
            NS_STATIC_CAST(txNodeTypeTest*, step->mNodeTest.get())->mNodeType ==
                txNodeTypeTest::NODE_TYPE &&
            NS_STATIC_CAST(PathExpr::PathExprItem*, iter.next())->pathOp ==
                PathExpr::RELATIVE_OP) {
            iter.previous();
            iter.remove();
            delete step;

            // Check if there is only one step left.
            pxi = NS_STATIC_CAST(PathExpr::PathExprItem*, iter.next());
            if (!iter.hasNext()) {
                aExpr = pxi->expr;

                // return since we no longer have a path-object.
                return NS_OK;
            }
        }
    }

    // look for txLocalNameFilter that can be moved up.
    iter.reset();
    iter.next();  // There is no point in moving the first step
    while ((pxi = NS_STATIC_CAST(PathExpr::PathExprItem*, iter.next()))) {
        if (pxi->expr->getType() == Expr::LOCALNAMEFILTER_EXPR) {
            txLocalNameFilter* lnFilter = NS_STATIC_CAST(txLocalNameFilter*, pxi->expr.get());
            if (!(lnFilter->mLocalName->getContextSensitivity() & Expr::DOCUMENT_CONTEXT)) {

                pxi->expr.forget();
                pxi->expr = lnFilter->mExpr;

                if (!iter.hasNext()) {
                    lnFilter->mExpr = aExpr;
                    aExpr = lnFilter;

                    return NS_OK;
                }

                nsAutoPtr<Expr> lnOwn(lnFilter); // in case something fails

                nsAutoPtr<PathExpr> newPath(new PathExpr());
                NS_ENSURE_TRUE(newPath, NS_ERROR_OUT_OF_MEMORY);

                while (iter.current()) {
                    rv = newPath->expressions.insert(0, iter.current());
                    NS_ENSURE_SUCCESS(rv, rv);

                    iter.remove();
                }

                lnFilter->mExpr = newPath.forget();
                rv = path->insertExpr(0, lnFilter, PathExpr::RELATIVE_OP);
                NS_ENSURE_SUCCESS(rv, rv);

                lnOwn.forget();
                iter.next();
            }
        }
    }

    return NS_OK;
}

nsresult
txXPathOptimizer::optimizeFilterExpr(nsAutoPtr<Expr>& aExpr,
                                     Expr::ResultType aResultType)
{
    FilterExpr* filter = NS_STATIC_CAST(FilterExpr*, aExpr.get());

    // Test for predicates like [local-name() = current()/some/expr]
    Expr* pred;
    if (!mCaseInsensitiveNameTests &&
        (pred = NS_STATIC_CAST(Expr*, filter->predicates.get(filter->predicates
             .getLength() - 1)))->getType() == Expr::RELATIONAL_EXPR) {
        RelationalExpr* rel = NS_STATIC_CAST(RelationalExpr*, pred);
        rel->swapIfRightIsA(Expr::FUNCTIONCALL_EXPR);
        if (rel->mOp == RelationalExpr::EQUAL &&
            rel->mLeftExpr->getType() == Expr::FUNCTIONCALL_EXPR &&
            !(rel->mRightExpr->getContextSensitivity() &
              (Expr::NODE_CONTEXT | Expr::NODESET_CONTEXT))/* &&
            !(rel->mRightExpr->getReturnType() &
              ~(Expr::STRING_RESULT | Expr::NODESET_RESULT))*/) {
            FunctionCall* fun = NS_STATIC_CAST(FunctionCall*, rel->mLeftExpr.get());
            nsCOMPtr<nsIAtom> funName;
            fun->getNameAtom(getter_AddRefs(funName));
            if (funName == txXPathAtoms::localName && fun->params.isEmpty()) {
                txLocalNameFilter* lnFilter = new txLocalNameFilter(aExpr, rel->mRightExpr);
                NS_ENSURE_TRUE(lnFilter, NS_ERROR_OUT_OF_MEMORY);

                aExpr = lnFilter;
                delete filter->predicates.remove(pred);

                if (filter->isEmpty()) {
                    lnFilter->mExpr = filter->expr;
                }

                return NS_OK; // return since we no longer have a filter-object.
            }
        }
    }

    return NS_OK;
}

