/*
    This file is part of libkabc.
    Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

/*
Enhanced Version of the file for platform independent KDE tools.
Copyright (c) 2004 Ulf Schenk

$Id: resourceqtopia.cpp,v 1.10 2004/08/20 19:54:12 ulf69 Exp $
*/
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>
//US #include <qtimer.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
//US #include <ksavefile.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>

#include <qpe/pim/addressbookaccess.h>


#include "resourceqtopiaconfig.h"
#include "stdaddressbook.h"

#include "qtopiaconverter.h"
#include "syncprefwidget.h"

#include "resourceqtopia.h"

using namespace KABC;
extern "C"
{
  void *init_microkabc_qtopia()
  {
    return new KRES::PluginFactory<ResourceQtopia,ResourceQtopiaConfig, SyncPrefWidgetContainer>();
  }
}

ResourceQtopia::ResourceQtopia( const KConfig *config, bool syncable  )
  : Resource( config, syncable ), mConverter (0)
{
  // we can not choose the filename. Therefore use the default to display
  QString fileName = QDir::homeDirPath() + "/Applications/addressbook/addressbook.xml";
  init( fileName );
}

ResourceQtopia::ResourceQtopia( const QString &fileName, bool syncable  )
  : Resource( 0, syncable )
{
  init( fileName );
}

void ResourceQtopia::init( const QString &fileName )
{

  connect( &mDirWatch, SIGNAL( dirty(const QString&) ), SLOT( fileChanged() ) );
  connect( &mDirWatch, SIGNAL( created(const QString&) ), SLOT( fileChanged() ) );
  connect( &mDirWatch, SIGNAL( deleted(const QString&) ), SLOT( fileChanged() ) );

  setFileName( fileName );
}

ResourceQtopia::~ResourceQtopia()
{
  if (mConverter != 0)
    delete mConverter;

  if(mAccess != 0)
    delete mAccess;
}

void ResourceQtopia::writeConfig( KConfig *config )
{
  Resource::writeConfig( config );
}

Ticket *ResourceQtopia::requestSaveTicket()
{
  kdDebug(5700) << "ResourceQtopia::requestSaveTicket()" << endl;

  qDebug("ResourceQtopia::requestSaveTicket: %s", fileName().latin1());

  if ( !addressBook() ) return 0;

  if ( !lock( fileName() ) ) {
    kdDebug(5700) << "ResourceQtopia::requestSaveTicket(): Unable to lock file '"
                  << fileName() << "'" << endl;
    return 0;
  }
  return createTicket( this );
}


bool ResourceQtopia::doOpen()
{
  qDebug("ResourceQtopia::doOpen(): %s", fileName().latin1());

  mAccess = new AddressBookAccess();

  if ( !mAccess ) {
    qDebug("Unable to load file() %s", fileName().latin1());
    return false;
  }


  if (mConverter == 0)
  {
    mConverter = new QtopiaConverter();
    bool res = mConverter->init();
    if ( !res )
    {
	  QString msg("Unable to initialize qtopia converter. Most likely a problem with the category file");
      qDebug(msg);
      delete mAccess;
      mAccess = 0;
      return false;
    }
  }

  return true;
}

void ResourceQtopia::doClose()
{
  qDebug("ResourceQtopia::doClose: %s", fileName().latin1());

  if(mAccess)
  {
    delete mAccess;
    mAccess = 0;
  }
  // it seems so, that deletion of access deletes backend as well
  //delete backend;

  return;
}

bool ResourceQtopia::load()
{
  qDebug("ResourceQtopia::load: %s", fileName().latin1());

  AddressBookIterator it(*mAccess);
  const PimContact* contact;
  bool res;

  for (contact=it.toFirst(); it.current(); ++it)
  {
    contact = it.current();

    KABC::Addressee addressee;

    res = mConverter->qtopiaToAddressee( (*contact), addressee );

    if ( !addressee.isEmpty() && res )
    {
      addressee.setResource( this );
      addressBook()->insertAddressee( addressee );
    }
  }

  return true;
}

bool ResourceQtopia::save( Ticket *ticket )
{
  qDebug("ResourceQtopia::save: %s", fileName().latin1());

  mDirWatch.stopScan();

  KABC::AddressBook::Iterator it;
  bool res;

  for ( it = addressBook()->begin(); it != addressBook()->end(); ++it ) {
    PimContact c;
    KABC::Addressee addressee = (*it);

    res = mConverter->addresseeToQtopia( *it, c );
    if (res == true)
    {
      mAccess->addContact(c);
//      if (res == false)
//        qDebug("Unable to append Contact %s", c.fullName().latin1());
    }
    else
    {
      qDebug("Unable to convert Addressee %s", addressee.formattedName().latin1());
    }
  }

//  mAccess->addressBookUpdated();

  mDirWatch.startScan();

  delete ticket;
  unlock( fileName() );

  return true;
}

bool ResourceQtopia::lock( const QString &lockfileName )
{
  qDebug("ResourceQtopia::lock: %s", fileName().latin1());

  kdDebug(5700) << "ResourceQtopia::lock()" << endl;

  QString fn = lockfileName;

  KURL url(fn);
  QString lockName = locateLocal( "data", "kabc/lock/" + url.fileName() + ".lock" );

  kdDebug(5700) << "-- lock name: " << lockName << endl;

  if (QFile::exists( lockName ))
  {
    qDebug("ResourceOpie::lock: lockfile %s already exists. (Delete it before continuing if nobody else is accessing the resourcefile %s)", lockName.latin1(), fileName().latin1());
    return false;
  }

  QString lockUniqueName;
  lockUniqueName = fn + KApplication::randomString( 8 );

  url = lockUniqueName;
//US  mLockUniqueName = locateLocal( "data", "kabc/lock/" + lockUniqueName );
  mLockUniqueName = locateLocal( "data", "kabc/lock/" + url.fileName() );
  kdDebug(5700) << "-- lock unique name: " << mLockUniqueName << endl;

  // Create unique file
  QFile file( mLockUniqueName );
  file.open( IO_WriteOnly );
  file.close();

  // Create lock file
  int result = 0;
#ifndef _WIN32_
  result = ::link( QFile::encodeName( mLockUniqueName ),
                       QFile::encodeName( lockName ) );
#endif
  if ( result == 0 ) {
    addressBook()->emitAddressBookLocked();
    return true;
  }

  // TODO: check stat

  return false;
}

void ResourceQtopia::unlock( const QString &fileName )
{
  qDebug("ResourceQtopia::unlock() %s", fileName.latin1());

  QString fn = fileName;
  KURL url(fn);
  QString lockName = locateLocal( "data", "kabc/lock/" + url.fileName() + ".lock" );

  QFile::remove( lockName );
  QFile::remove( mLockUniqueName );
  addressBook()->emitAddressBookUnlocked();
}

void ResourceQtopia::setFileName( const QString &newFileName )
{
  mDirWatch.stopScan();
  mDirWatch.removeFile( fileName() );

  Resource::setFileName( newFileName );

  mDirWatch.addFile( fileName() );
  mDirWatch.startScan();
}


void ResourceQtopia::fileChanged()
{
  // There is a small theoretical chance that KDirWatch calls us before
  // we are fully constructed
  if (!addressBook())
    return;

  QString text( i18n( "Qtopia resource '%1'<br> has been changed by third party.<br>Do you want to reload?").arg( fileName() ) );
  if ( readOnly() || KMessageBox::questionYesNo( 0, text ) == KMessageBox::Yes ) {
    load();
    addressBook()->emitAddressBookChanged();
  }
}

void ResourceQtopia::removeAddressee( const Addressee &addr )
{
}

void ResourceQtopia::cleanUp()
{
  unlock( fileName() );
}

//US #include "resourceqtopia.moc"
