/*
    This file is part of libkabc.
    Copyright (c) 2003 Tobias Koenig <tokoe@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include <qregexp.h>

#include <kmdcodec.h>

#include "vcardparser.h"

#define FOLD_WIDTH 75

using namespace KABC;

VCardParser::VCardParser()
{
}

VCardParser::~VCardParser()
{
}

VCard::List VCardParser::parseVCards( const QString& text )
{
  VCard currentVCard;
  VCard::List vCardList;
  QString currentLine;

  QStringList lines = QStringList::split( QRegExp( "[\x0d\x0a]" ), text );
  QStringList::Iterator it;

  bool inVCard = false;
  for ( it = lines.begin(); it != lines.end(); ++it ) {

    if ( (*it).isEmpty() ) // empty line
      continue;

    if ( (*it)[ 0 ] == ' ' || (*it)[ 0 ] == '\t' ) { // folded line => append to previous
      currentLine += (*it).remove( 0, 1 );
      continue;
    } else {
      if ( inVCard && !currentLine.isEmpty() ) { // now parse the line
        int colon = currentLine.find( ':' );
        if ( colon == -1 ) { // invalid line
          currentLine = (*it);
          continue;
        }

        VCardLine vCardLine;
        QString key = currentLine.left( colon ).stripWhiteSpace();
        QString value = currentLine.mid( colon + 1 );

        QStringList params = QStringList::split( ';', key );
        vCardLine.setIdentifier( params[0] );
        if ( params.count() > 1 ) { // find all parameters
          for ( uint i = 1; i < params.count(); ++i ) {
            QStringList pair = QStringList::split( '=', params[i] );
//US            if ( pair.size() == 1 ) {
            if ( pair.count() == 1 ) {
                pair.prepend( "type" );
            }
            if ( pair[1].contains( ',' ) ) { // parameter in type=x,y,z format
              QStringList args = QStringList::split( ',', pair[ 1 ] );
              for ( uint j = 0; j < args.count(); ++j )
                vCardLine.addParameter( pair[0].lower(), args[j] );
            } else
              vCardLine.addParameter( pair[0].lower(), pair[1] );
          }
        }

        params = vCardLine.parameterList();
        if ( params.contains( "encoding" ) ) { // have to decode the data
#if 0
          QByteArray input, output;
          input = value.local8Bit();
          if ( vCardLine.parameter( "encoding" ).lower() == "b" )
            KCodecs::base64Decode( input, output );
          else if ( vCardLine.parameter( "encoding" ).lower() == "quoted-printable" )
            KCodecs::quotedPrintableDecode( input, output );

          //qDebug("VCardParser::parseVCards has to be verified");
          //US I am not sure if this is correct
          //US          vCardLine.setValue( output );
          QCString cs(output);
          qDebug("len1 %d len2 %d ",input.size(), output.size( ));
#endif
          QCString cs = value.local8Bit();
          qDebug("****************************************** ");
          qDebug("************* WARNING ******************** ");
          qDebug("****************************************** ");
          qDebug("Make sure, the decoding is done after");
          qDebug("QVariant conversion!");
          qDebug("Insert Line DECODING OKAY, where this is implemented");
          // use for decoding the above code!
          vCardLine.setValue( cs );
        } else {
        
            //qDebug("VCardParser::parseVCards has to be verified");
//US          vCardLine.setValue( value.replace( "\\n", "\n" ) );
          vCardLine.setValue( value.replace( QRegExp("\\n"), "\n" ) );
        }

        currentVCard.addLine( vCardLine );
      }
      // we do not save the start and end tag as vcardline
      if ( (*it).lower().startsWith( "begin:vcard" ) ) {
        inVCard = true;
        //qDebug("VCardParser::parseVCards has to be verified");
//US        currentLine.setLength( 0 );
        currentLine = "";
        currentVCard.clear(); // flush vcard
        continue;
      }

      if ( (*it).lower().startsWith( "end:vcard" ) ) {
        inVCard = false;
        vCardList.append( currentVCard );
        //qDebug("VCardParser::parseVCards has to be verified");
//US        currentLine.setLength( 0 );
        currentLine = "";
        currentVCard.clear(); // flush vcard
        continue;
      }

      currentLine = (*it);
    }
  }

  return vCardList;
}

QString VCardParser::createVCards( const VCard::List& list )
{
  QString text;
  QString textLine;
  QString encodingType;
  QStringList idents;
  QStringList params;
  QStringList values;
  QStringList::ConstIterator identIt;
  QStringList::Iterator paramIt;
  QStringList::Iterator valueIt;

  VCardLine::List lines;
  VCardLine::List::Iterator lineIt;
  VCard::List::ConstIterator cardIt;

  bool hasEncoding;

  
  // iterate over the cards
  for ( cardIt = list.begin(); cardIt != list.end(); ++cardIt ) {
    text.append( "BEGIN:VCARD\r\n" );

    idents = (*cardIt).identifiers();
    for ( identIt = idents.begin(); identIt != idents.end(); ++identIt ) {
      VCard card = (*cardIt);
      lines = card.lines( (*identIt) );

      // iterate over the lines
      for ( lineIt = lines.begin(); lineIt != lines.end(); ++lineIt ) {
        if ( !(*lineIt).value().asString().isEmpty() ) {
          textLine = (*lineIt).identifier();

          params = (*lineIt).parameterList();
          hasEncoding = false;
          if ( params.count() > 0 ) { // we have parameters
            for ( paramIt = params.begin(); paramIt != params.end(); ++paramIt ) {
              if ( (*paramIt) == "encoding" ) {
                hasEncoding = true;
                encodingType = (*lineIt).parameter( "encoding" ).lower();
              }

              values = (*lineIt).parameters( *paramIt );
              for ( valueIt = values.begin(); valueIt != values.end(); ++valueIt ) {
                textLine.append( ";" + (*paramIt).upper() );
                if ( !(*valueIt).isEmpty() )
                  textLine.append( "=" + (*valueIt) );
              }
            }
          }

          if ( hasEncoding ) { // have to encode the data
            QByteArray input, output;
            
    qDebug("VCardParser::createVCards has to be verified");
//US            input = (*lineIt).value().toByteArray();

//US I am not sure if this is correct    
            QCString cs ((*lineIt).value().toCString());
            input = cs;
            
            if ( encodingType == "b" )
              KCodecs::base64Encode( input, output );
            else if ( encodingType == "quoted-printable" )
              KCodecs::quotedPrintableEncode( input, output );
            textLine.append( ":" + QString( output ) );
          } else {
    qDebug("VCardParser::createVCards has to be verified");
//US            textLine.append( ":" + (*lineIt).value().asString().replace( "\n", "\\n" ) );
            textLine.append( ":" + (*lineIt).value().asString().replace( QRegExp("\n"), "\\n" ) );
          }
          if ( textLine.length() > FOLD_WIDTH ) { // we have to fold the line
            for ( uint i = 0; i <= ( textLine.length() / FOLD_WIDTH ); ++i )
              text.append( ( i == 0 ? "" : " " ) + textLine.mid( i * FOLD_WIDTH, FOLD_WIDTH ) + "\r\n" );
          } else
            text.append( textLine + "\r\n" );
        }
      }
    }

    text.append( "END:VCARD\r\n" );
    text.append( "\r\n" );
  }

  return text;
}
