/*
    This file is part of KAddressbook.
    Copyright (c) 2003 Tobias Koenig <tokoe@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

/*
Enhanced Version of the file for platform independent KDE tools.
Copyright (c) 2004 Ulf Schenk

$Id: kabcore.cpp,v 1.42 2004/09/11 22:40:15 zautrix Exp $
*/

#include "kabcore.h"

#include <stdaddressbook.h>
#include <klocale.h>

#ifndef KAB_EMBEDDED
#include <qclipboard.h>
#include <qdir.h>
#include <qfile.h>
#include <qapplicaton.h>
#include <qlayout.h>
#include <qregexp.h>
#include <qvbox.h>
#include <kabc/addresseelist.h>
#include <kabc/errorhandler.h>
#include <kabc/resource.h>
#include <kabc/vcardconverter.h>
#include <kapplication.h>
#include <kactionclasses.h>
#include <kcmultidialog.h>
#include <kdebug.h>
#include <kdeversion.h>
#include <kkeydialog.h>
#include <kmessagebox.h>
#include <kprinter.h>
#include <kprotocolinfo.h>
#include <kresources/selectdialog.h>
#include <kstandarddirs.h>
#include <ktempfile.h>
#include <kxmlguiclient.h>
#include <kaboutdata.h>
#include <libkdepim/categoryselectdialog.h>

#include "addresseeutil.h"
#include "addresseeeditordialog.h"
#include "extensionmanager.h"
#include "kstdaction.h"
#include "kaddressbookservice.h"
#include "ldapsearchdialog.h"
#include "printing/printingwizard.h"
#else // KAB_EMBEDDED

#include <kapplication.h>
#include "KDGanttMinimizeSplitter.h"
#include "kaddressbookmain.h"
#include "kactioncollection.h"
#include "addresseedialog.h"
//US
#include <libkdepim/addresseeview.h>

#include <qapp.h>
#include <qmenubar.h>
//#include <qtoolbar.h>
#include <qmessagebox.h>
#include <kdebug.h>
#include <kiconloader.h> // needed for SmallIcon
#include <kresources/kcmkresources.h>
#include <ktoolbar.h>


//#include <qlabel.h>


#ifndef DESKTOP_VERSION
#include <qpe/ir.h>
#include <qpe/qpemenubar.h>
#include <qtopia/qcopenvelope_qws.h>
#else

#include <qmenubar.h>
#endif

#endif // KAB_EMBEDDED
#include "kcmconfigs/kcmkabconfig.h"
#include "kcmconfigs/kcmkdepimconfig.h"
#include "kpimglobalprefs.h"
#include "externalapphandler.h"


#include <kresources/selectdialog.h>
#include <kmessagebox.h>

#include <picture.h>
#include <resource.h>

//US#include <qsplitter.h>
#include <qmap.h>
#include <qdir.h>
#include <qfile.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qclipboard.h>
#include <qtextstream.h>

#include <libkdepim/categoryselectdialog.h>
#include <kabc/vcardconverter.h>


#include "addresseeutil.h"
#include "undocmds.h"
#include "addresseeeditordialog.h"
#include "viewmanager.h"
#include "details/detailsviewcontainer.h"
#include "kabprefs.h"
#include "xxportmanager.h"
#include "incsearchwidget.h"
#include "jumpbuttonbar.h"
#include "extensionmanager.h"
#include "addresseeconfig.h"
#include <kcmultidialog.h>

#ifdef _WIN32_

#include "kaimportoldialog.h"
#endif

bool pasteWithNewUid = true;

#ifdef KAB_EMBEDDED
KABCore::KABCore( KAddressBookMain *client, bool readWrite, QWidget *parent, const char *name )
  : QWidget( parent, name ), mGUIClient( client ), mViewManager( 0 ),
    mExtensionManager( 0 ),mConfigureDialog( 0 ),/*US  mLdapSearchDialog( 0 ),*/
    mReadWrite( readWrite ), mModified( false ), mMainWindow(client)
#else //KAB_EMBEDDED
KABCore::KABCore( KXMLGUIClient *client, bool readWrite, QWidget *parent, const char *name )
  : QWidget( parent, name ), mGUIClient( client ), mViewManager( 0 ),
    mExtensionManager( 0 ), mConfigureDialog( 0 ), mLdapSearchDialog( 0 ),
    mReadWrite( readWrite ), mModified( false )
#endif //KAB_EMBEDDED
{

  mExtensionBarSplitter = 0;
  mIsPart = !parent->inherits( "KAddressBookMain" );

  mAddressBook = KABC::StdAddressBook::self();
  KABC::StdAddressBook::setAutomaticSave( false );

#ifndef KAB_EMBEDDED
  mAddressBook->setErrorHandler( new KABC::GUIErrorHandler );
#endif //KAB_EMBEDDED

  connect( mAddressBook, SIGNAL( addressBookChanged( AddressBook * ) ),
           SLOT( addressBookChanged() ) );

  mAddressBook->addCustomField( i18n( "Department" ), KABC::Field::Organization,
                                "X-Department", "KADDRESSBOOK" );
  mAddressBook->addCustomField( i18n( "Profession" ), KABC::Field::Organization,
                                "X-Profession", "KADDRESSBOOK" );
  mAddressBook->addCustomField( i18n( "Assistant's Name" ), KABC::Field::Organization,
                                "X-AssistantsName", "KADDRESSBOOK" );
  mAddressBook->addCustomField( i18n( "Manager's Name" ), KABC::Field::Organization,
                                "X-ManagersName", "KADDRESSBOOK" );
  mAddressBook->addCustomField( i18n( "Spouse's Name" ), KABC::Field::Personal,
                                "X-SpousesName", "KADDRESSBOOK" );
  mAddressBook->addCustomField( i18n( "Office" ), KABC::Field::Personal,
                                "X-Office", "KADDRESSBOOK" );
  mAddressBook->addCustomField( i18n( "IM Address" ), KABC::Field::Personal,
                                "X-IMAddress", "KADDRESSBOOK" );
  mAddressBook->addCustomField( i18n( "Anniversary" ), KABC::Field::Personal,
                                "X-Anniversary", "KADDRESSBOOK" );

  //US added this field to become compatible with Opie/qtopia addressbook
  // values can be "female" or "male" or "". An empty field represents undefined.
  mAddressBook->addCustomField( i18n( "Gender" ), KABC::Field::Personal,
                                "X-Gender", "KADDRESSBOOK" );
  mAddressBook->addCustomField( i18n( "Children" ), KABC::Field::Personal,
                                "X-Children", "KADDRESSBOOK" );
  mAddressBook->addCustomField( i18n( "FreeBusyUrl" ), KABC::Field::Personal,
                                "X-FreeBusyUrl", "KADDRESSBOOK" );

  initGUI();

  mIncSearchWidget->setFocus();


  connect( mViewManager, SIGNAL( selected( const QString& ) ),
           SLOT( setContactSelected( const QString& ) ) );
  connect( mViewManager, SIGNAL( executed( const QString& ) ),
           SLOT( executeContact( const QString& ) ) );

  connect( mViewManager, SIGNAL( deleteRequest( ) ),
           SLOT( deleteContacts( ) ) );
  connect( mViewManager, SIGNAL( modified() ),
           SLOT( setModified() ) );

  connect( mExtensionManager, SIGNAL( modified( const KABC::Addressee::List& ) ), this, SLOT( extensionModified( const KABC::Addressee::List& ) ) );
  connect( mExtensionManager, SIGNAL( changedActiveExtension( int ) ), this, SLOT( extensionChanged( int ) ) );

  connect( mXXPortManager, SIGNAL( modified() ),
           SLOT( setModified() ) );

  connect( mJumpButtonBar, SIGNAL( jumpToLetter( const QString& ) ),
           SLOT( incrementalSearch( const QString& ) ) );
  connect( mIncSearchWidget, SIGNAL( fieldChanged() ),
           mJumpButtonBar, SLOT( recreateButtons() ) );

  connect( mDetails, SIGNAL( sendEmail( const QString& ) ),
           SLOT( sendMail( const QString& ) ) );


  connect( ExternalAppHandler::instance(), SIGNAL (requestForNameEmailUidList(const QString&, const QString&)),this, SLOT(requestForNameEmailUidList(const QString&, const QString&)));
  connect( ExternalAppHandler::instance(), SIGNAL (requestForDetails(const QString&, const QString&, const QString&, const QString&, const QString&)),this, SLOT(requestForDetails(const QString&, const QString&, const QString&, const QString&, const QString&)));


#ifndef KAB_EMBEDDED
  connect( mViewManager, SIGNAL( urlDropped( const KURL& ) ),
           mXXPortManager, SLOT( importVCard( const KURL& ) ) );

  connect( mDetails, SIGNAL( browse( const QString& ) ),
           SLOT( browse( const QString& ) ) );


  mAddressBookService = new KAddressBookService( this );

#endif //KAB_EMBEDDED
  mEditorDialog = 0;
  createAddresseeEditorDialog( this );
  setModified( false );
}

KABCore::~KABCore()
{
    //  save();
    //saveSettings();
    //KABPrefs::instance()->writeConfig();
    delete AddresseeConfig::instance();
  mAddressBook = 0;
  KABC::StdAddressBook::close();
}

void KABCore::restoreSettings()
{
  mMultipleViewsAtOnce = KABPrefs::instance()->mMultipleViewsAtOnce;

  bool state;

  if (mMultipleViewsAtOnce)
    state = KABPrefs::instance()->mDetailsPageVisible;
  else
    state = false;

  mActionDetails->setChecked( state );
  setDetailsVisible( state );

  state = KABPrefs::instance()->mJumpButtonBarVisible;

  mActionJumpBar->setChecked( state );
  setJumpButtonBarVisible( state );
/*US
 QValueList<int> splitterSize = KABPrefs::instance()->mDetailsSplitter;
  if ( splitterSize.count() == 0 ) {
    splitterSize.append( width() / 2 );
    splitterSize.append( width() / 2 );
  }
  mMiniSplitter->setSizes( splitterSize );
  if (  mExtensionBarSplitter ) {
  splitterSize = KABPrefs::instance()->mExtensionsSplitter;
  if ( splitterSize.count() == 0 ) {
    splitterSize.append( width() / 2 );
    splitterSize.append( width() / 2 );
  }
  mExtensionBarSplitter->setSizes( splitterSize );

  }
*/
  mViewManager->restoreSettings();
  mIncSearchWidget->setCurrentItem( KABPrefs::instance()->mCurrentIncSearchField );
  mExtensionManager->restoreSettings();
#ifdef DESKTOP_VERSION
  int wid = width();
  if ( wid < 10 )
      wid = 400;
#else
  int wid = QApplication::desktop()->width();
  if ( wid < 640 )
      wid = QApplication::desktop()->height();
#endif
  QValueList<int> splitterSize;// = KABPrefs::instance()->mDetailsSplitter;
  if ( true /*splitterSize.count() == 0*/ ) {
      splitterSize.append( wid / 2 );
      splitterSize.append( wid / 2 );
  }
  mMiniSplitter->setSizes( splitterSize );
  if (  mExtensionBarSplitter ) {
      //splitterSize = KABPrefs::instance()->mExtensionsSplitter;
      if ( true /*splitterSize.count() == 0*/ ) {
          splitterSize.append( wid / 2 );
          splitterSize.append( wid / 2 );
      }
      mExtensionBarSplitter->setSizes( splitterSize );

  }


}

void KABCore::saveSettings()
{
  KABPrefs::instance()->mJumpButtonBarVisible = mActionJumpBar->isChecked();
  if ( mExtensionBarSplitter )
    KABPrefs::instance()->mExtensionsSplitter = mExtensionBarSplitter->sizes();
  KABPrefs::instance()->mDetailsPageVisible = mActionDetails->isChecked();
  KABPrefs::instance()->mDetailsSplitter = mMiniSplitter->sizes();
#ifndef KAB_EMBEDDED

  KABPrefs::instance()->mExtensionsSplitter = mExtensionBarSplitter->sizes();
  KABPrefs::instance()->mDetailsSplitter = mDetailsSplitter->sizes();
#endif //KAB_EMBEDDED
  mExtensionManager->saveSettings();
  mViewManager->saveSettings();

  KABPrefs::instance()->mCurrentIncSearchField = mIncSearchWidget->currentItem();
}

KABC::AddressBook *KABCore::addressBook() const
{
  return mAddressBook;
}

KConfig *KABCore::config()
{
#ifndef KAB_EMBEDDED
  return KABPrefs::instance()->config();
#else //KAB_EMBEDDED
  return KABPrefs::instance()->getConfig();
#endif //KAB_EMBEDDED
}

KActionCollection *KABCore::actionCollection() const
{
  return mGUIClient->actionCollection();
}

KABC::Field *KABCore::currentSearchField() const
{
   if (mIncSearchWidget)
    return mIncSearchWidget->currentField();
  else
    return 0;
}

QStringList KABCore::selectedUIDs() const
{
  return mViewManager->selectedUids();
}

KABC::Resource *KABCore::requestResource( QWidget *parent )
{
  QPtrList<KABC::Resource> kabcResources = addressBook()->resources();

  QPtrList<KRES::Resource> kresResources;
  QPtrListIterator<KABC::Resource> resIt( kabcResources );
  KABC::Resource *resource;
  while ( ( resource = resIt.current() ) != 0 ) {
    ++resIt;
    if ( !resource->readOnly() ) {
      KRES::Resource *res = static_cast<KRES::Resource*>( resource );
      if ( res )
        kresResources.append( res );
    }
  }

  KRES::Resource *res = KRES::SelectDialog::getResource( kresResources, parent );
  return static_cast<KABC::Resource*>( res );
}

#ifndef KAB_EMBEDDED
KAboutData *KABCore::createAboutData()
#else //KAB_EMBEDDED
void KABCore::createAboutData()
#endif //KAB_EMBEDDED
{
#ifndef KAB_EMBEDDED
  KAboutData *about = new KAboutData( "kaddressbook", I18N_NOOP( "KAddressBook" ),
                                      "3.1", I18N_NOOP( "The KDE Address Book" ),
                                      KAboutData::License_GPL_V2,
                                      I18N_NOOP( "(c) 1997-2003, The KDE PIM Team" ) );
  about->addAuthor( "Tobias Koenig", I18N_NOOP( "Current maintainer " ), "tokoe@kde.org" );
  about->addAuthor( "Don Sanders", I18N_NOOP( "Original author " ) );
  about->addAuthor( "Cornelius Schumacher",
                    I18N_NOOP( "Co-maintainer, libkabc port, CSV import/export " ),
                    "schumacher@kde.org" );
  about->addAuthor( "Mike Pilone", I18N_NOOP( "GUI and framework redesign " ),
                    "mpilone@slac.com" );
  about->addAuthor( "Greg Stern", I18N_NOOP( "DCOP interface" ) );
  about->addAuthor( "Mark Westcott", I18N_NOOP( "Contact pinning" ) );
  about->addAuthor( "Michel Boyer de la Giroday", I18N_NOOP( "LDAP Lookup\n" ),
                    "michel@klaralvdalens-datakonsult.se" );
  about->addAuthor( "Steffen Hansen", I18N_NOOP( "LDAP Lookup " ),
                    "hansen@kde.org" );

  return about;
#endif //KAB_EMBEDDED

  QString version;
#include <../version>
    QMessageBox::about( this, "About KAddressbook/Pi",
                        "KAddressbook/Platform-independent\n"
                        "(KA/Pi) " +version + " - " +
#ifdef DESKTOP_VERSION
                        "Desktop Edition\n"
#else
                        "PDA-Edition\n"
                        "for: Zaurus 5500 / 7x0 / 8x0\n"
#endif

                        "(c) 2004 Ulf Schenk\n"
                        "(c) 2004 Lutz Rogowski\n"
                        "(c) 1997-2003, The KDE PIM Team\n"
  "Tobias Koenig Current maintainer\ntokoe@kde.org\n"
  "Don Sanders Original author\n"
  "Cornelius Schumacher Co-maintainer\nschumacher@kde.org\n"
  "Mike Pilone GUI and framework redesign\nmpilone@slac.com\n"
  "Greg Stern DCOP interface\n"
  "Mark Westcot Contact pinning\n"
  "Michel Boyer de la Giroday LDAP Lookup\n" "michel@klaralvdalens-datakonsult.se\n"
  "Steffen Hansen LDAP Lookup\nhansen@kde.org\n"
#ifdef _WIN32_
 "(c) 2004 Lutz Rogowski Import from OL\nrogowski@kde.org\n"
#endif
 );
}

void KABCore::setContactSelected( const QString &uid )
{
  KABC::Addressee addr = mAddressBook->findByUid( uid );
  if ( !mDetails->isHidden() )
    mDetails->setAddressee( addr );

  if ( !addr.isEmpty() ) {
    emit contactSelected( addr.formattedName() );
    KABC::Picture pic = addr.photo();
    if ( pic.isIntern() ) {
//US      emit contactSelected( pic.data() );
//US instead use:
      QPixmap px;
      if (pic.data().isNull() != true)
      {
        px.convertFromImage(pic.data());
      }

      emit contactSelected( px );
    }
  }


  mExtensionManager->setSelectionChanged();

  // update the actions
  bool selected = !uid.isEmpty();

  if ( mReadWrite ) {
    mActionCut->setEnabled( selected );
    mActionPaste->setEnabled( selected );
  }

  mActionCopy->setEnabled( selected );
  mActionDelete->setEnabled( selected );
  mActionEditAddressee->setEnabled( selected );
  mActionMail->setEnabled( selected );
  mActionMailVCard->setEnabled( selected );
  //if (mActionBeam)
  //mActionBeam->setEnabled( selected );

  if (mActionBeamVCard)
   mActionBeamVCard->setEnabled( selected );

  mActionWhoAmI->setEnabled( selected );
  mActionCategories->setEnabled( selected );
}

void KABCore::sendMail()
{
  sendMail( mViewManager->selectedEmails().join( ", " ) );
}

void KABCore::sendMail( const QString& emaillist )
{
  // the parameter has the form "name1 <abc@aol.com>,name2 <abc@aol.com>;... "
  if (emaillist.contains(",") > 0)
    ExternalAppHandler::instance()->mailToMultipleContacts( emaillist, QString::null );
  else
    ExternalAppHandler::instance()->mailToOneContact( emaillist );
}



void KABCore::mailVCard()
{
  QStringList uids = mViewManager->selectedUids();
  if ( !uids.isEmpty() )
    mailVCard( uids );
}

void KABCore::mailVCard( const QStringList& uids )
{
  QStringList urls;

//  QString tmpdir = locateLocal("tmp", KGlobal::getAppName());

  QString dirName = "/tmp/" + KApplication::randomString( 8 );



  QDir().mkdir( dirName, true );

  for( QStringList::ConstIterator it = uids.begin(); it != uids.end(); ++it ) {
    KABC::Addressee a = mAddressBook->findByUid( *it );

    if ( a.isEmpty() )
      continue;

    QString name = a.givenName() + "_" + a.familyName() + ".vcf";

    QString fileName = dirName + "/" + name;

    QFile outFile(fileName);

    if ( outFile.open(IO_WriteOnly) ) {  // file opened successfully
      KABC::VCardConverter converter;
      QString vcard;

      converter.addresseeToVCard( a, vcard );

      QTextStream t( &outFile );  // use a text stream
      t.setEncoding( QTextStream::UnicodeUTF8 );
      t << vcard;

      outFile.close();

      urls.append( fileName );
    }
  }

  bool result = ExternalAppHandler::instance()->mailToMultipleContacts( QString::null, urls.join(", ") );


/*US
  kapp->invokeMailer( QString::null, QString::null, QString::null,
                      QString::null,  // subject
                      QString::null,  // body
                      QString::null,
                      urls );  // attachments
*/

}

/**
  Beams the "WhoAmI contact.
*/
void KABCore::beamMySelf()
{
  KABC::Addressee a = KABC::StdAddressBook::self()->whoAmI();
  if (!a.isEmpty())
  {
    QStringList uids;
    uids << a.uid();

	beamVCard(uids);
  } else {
      KMessageBox::information( this, i18n( "Your personal contact is\nnot set! Please select it\nand set it with menu:\nSettings - Set Who Am I\n" ) );


  }
}

void KABCore::beamVCard()
{
  QStringList uids = mViewManager->selectedUids();
  if ( !uids.isEmpty() )
    beamVCard( uids );
}


void KABCore::beamVCard(const QStringList& uids)
{
/*US
	QString beamFilename;
	Opie::OPimContact c;
	if ( actionPersonal->isOn() ) {
		beamFilename = addressbookPersonalVCardName();
		if ( !QFile::exists( beamFilename ) )
			return; // can't beam a non-existent file
		Opie::OPimContactAccessBackend* vcard_backend = new Opie::OPimContactAccessBackend_VCard( QString::null,
											beamFilename );
		Opie::OPimContactAccess* access = new Opie::OPimContactAccess ( "addressbook", QString::null , vcard_backend, true );
		Opie::OPimContactAccess::List allList = access->allRecords();
		Opie::OPimContactAccess::List::Iterator it = allList.begin();  // Just take first
		c = *it;

		delete access;
	} else {
		unlink( beamfile ); // delete if exists
		mkdir("/tmp/obex/", 0755);
		c = m_abView -> currentEntry();
		Opie::OPimContactAccessBackend* vcard_backend = new Opie::OPimContactAccessBackend_VCard( QString::null,
											beamfile );
		Opie::OPimContactAccess* access = new Opie::OPimContactAccess ( "addressbook", QString::null , vcard_backend, true );
		access->add( c );
		access->save();
		delete access;

		beamFilename = beamfile;
	}

	owarn << "Beaming: " << beamFilename << oendl;
*/

#if 0
  QString tmpdir = locateLocal("tmp", KGlobal::getAppName());

  QString dirName = tmpdir + "/" + KApplication::randomString( 8 );

  QString name = "contact.vcf";

  QString fileName = dirName + "/" + name;
#endif
  // LR: we should use the /tmp dir, because: /tmp = RAM, (HOME)/kdepim = flash memory
  //
  QString fileName = "/tmp/kapibeamfile.vcf";


  //QDir().mkdir( dirName, true );

 
  KABC::VCardConverter converter;
  QString description;
  QString datastream;
  for( QStringList::ConstIterator it = uids.begin(); it != uids.end(); ++it ) {
      KABC::Addressee a = mAddressBook->findByUid( *it );

      if ( a.isEmpty() )
          continue;

      if (description.isEmpty())
          description = a.formattedName();

      QString vcard;
      converter.addresseeToVCard( a, vcard );
      int start = 0;
      int next;
      while ( (next = vcard.find("TYPE=", start) )>= 0 ) {
          int semi = vcard.find(";", next);
          int dopp = vcard.find(":", next);
          int sep;
          if ( semi < dopp && semi >= 0 )
              sep = semi ;
          else
              sep = dopp;
          datastream +=vcard.mid( start, next - start);
          datastream +=vcard.mid( next+5,sep -next -5 ).upper();
          start = sep;
      }
      datastream += vcard.mid( start,vcard.length() );
  }
#ifndef DESKTOP_VERSION
  QFile outFile(fileName);
  if ( outFile.open(IO_WriteOnly) ) { 
      datastream.replace ( QRegExp("VERSION:3.0") , "VERSION:2.1" );
      QTextStream t( &outFile );  // use a text stream
      t.setEncoding( QTextStream::UnicodeUTF8 );
      t <<datastream;
      outFile.close();
      Ir *ir = new Ir( this );
      connect( ir, SIGNAL( done(Ir*) ), this, SLOT( beamDone(Ir*) ) );
      ir->send( fileName, description, "text/x-vCard" );
  } else {
      qDebug("Error open temp  beam file ");
      return;
  }
#endif

}

void KABCore::beamDone( Ir *ir )
{
#ifndef DESKTOP_VERSION
  delete ir;
#endif
}


void KABCore::browse( const QString& url )
{
#ifndef KAB_EMBEDDED
  kapp->invokeBrowser( url );
#else //KAB_EMBEDDED
  qDebug("KABCore::browse must be fixed");
#endif //KAB_EMBEDDED
}

void KABCore::selectAllContacts()
{
  mViewManager->setSelected( QString::null, true );
}

void KABCore::deleteContacts()
{
  QStringList uidList = mViewManager->selectedUids();
  deleteContacts( uidList );
}

void KABCore::deleteContacts( const QStringList &uids )
{
  if ( uids.count() > 0 ) {
    PwDeleteCommand *command = new PwDeleteCommand( mAddressBook, uids );
    UndoStack::instance()->push( command );
    RedoStack::instance()->clear();

    // now if we deleted anything, refresh
    setContactSelected( QString::null );
    setModified( true );
  }
}

void KABCore::copyContacts()
{
  KABC::Addressee::List addrList = mViewManager->selectedAddressees();

  QString clipText = AddresseeUtil::addresseesToClipboard( addrList );

  kdDebug(5720) << "KABCore::copyContacts: " << clipText << endl;

  QClipboard *cb = QApplication::clipboard();
  cb->setText( clipText );
}

void KABCore::cutContacts()
{
  QStringList uidList = mViewManager->selectedUids();

//US  if ( uidList.size() > 0 ) {
  if ( uidList.count() > 0 ) {
    PwCutCommand *command = new PwCutCommand( mAddressBook, uidList );
    UndoStack::instance()->push( command );
    RedoStack::instance()->clear();

    setModified( true );
  }
}

void KABCore::pasteContacts()
{
  QClipboard *cb = QApplication::clipboard();

  KABC::Addressee::List list = AddresseeUtil::clipboardToAddressees( cb->text() );

  pasteContacts( list );
}

void KABCore::pasteContacts( KABC::Addressee::List &list  )
{
  KABC::Resource *resource = requestResource( this );
  KABC::Addressee::List::Iterator it;
  for ( it = list.begin(); it != list.end(); ++it )
    (*it).setResource( resource );

  PwPasteCommand *command = new PwPasteCommand( this, list );
  UndoStack::instance()->push( command );
  RedoStack::instance()->clear();

  setModified( true );
}

void KABCore::setWhoAmI()
{
  KABC::Addressee::List addrList = mViewManager->selectedAddressees();

  if ( addrList.count() > 1 ) {
    KMessageBox::sorry( this, i18n( "Please select only one contact." ) );
    return;
  }

  QString text( i18n( "<qt>Do you really want to use <b>%1</b> as your new personal contact?</qt>" ) );
  if ( KMessageBox::questionYesNo( this, text.arg( addrList[ 0 ].assembledName() ) ) == KMessageBox::Yes )
    static_cast<KABC::StdAddressBook*>( KABC::StdAddressBook::self() )->setWhoAmI( addrList[ 0 ] );
}

void KABCore::setCategories()
{
  KPIM::CategorySelectDialog dlg( KABPrefs::instance(), this, "", true );
  if ( !dlg.exec() )
    return;

  bool merge = false;
  QString msg = i18n( "Merge with existing categories?" );
  if ( KMessageBox::questionYesNo( this, msg ) == KMessageBox::Yes )
    merge = true;

  QStringList categories = dlg.selectedCategories();

  QStringList uids = mViewManager->selectedUids();
  QStringList::Iterator it;
  for ( it = uids.begin(); it != uids.end(); ++it ) {
    KABC::Addressee addr = mAddressBook->findByUid( *it );
    if ( !addr.isEmpty() ) {
      if ( !merge )
        addr.setCategories( categories );
      else {
        QStringList addrCategories = addr.categories();
        QStringList::Iterator catIt;
        for ( catIt = categories.begin(); catIt != categories.end(); ++catIt ) {
          if ( !addrCategories.contains( *catIt ) )
            addrCategories.append( *catIt );
        }
        addr.setCategories( addrCategories );
      }

      mAddressBook->insertAddressee( addr );
    }
  }

  if ( uids.count() > 0 )
    setModified( true );
}

void KABCore::setSearchFields( const KABC::Field::List &fields )
{
  mIncSearchWidget->setFields( fields );
}

void KABCore::incrementalSearch( const QString& text )
{
    mViewManager->doSearch( text, mIncSearchWidget->currentField() );
}

void KABCore::setModified()
{
  setModified( true );
}

void KABCore::setModifiedWOrefresh()
{
    // qDebug("KABCore::setModifiedWOrefresh() ");
    mModified = true;
    mActionSave->setEnabled( mModified );
#ifdef DESKTOP_VERSION
  mDetails->refreshView();
#endif

}
void KABCore::setModified( bool modified )
{
  mModified = modified;
  mActionSave->setEnabled( mModified );

  if ( modified )
    mJumpButtonBar->recreateButtons();

  mViewManager->refreshView();
  mDetails->refreshView();

}

bool KABCore::modified() const
{
  return mModified;
}

void KABCore::contactModified( const KABC::Addressee &addr )
{

  Command *command = 0;
  QString uid;

  // check if it exists already
  KABC::Addressee origAddr = mAddressBook->findByUid( addr.uid() );
  if ( origAddr.isEmpty() )
    command = new PwNewCommand( mAddressBook, addr );
  else {
    command = new PwEditCommand( mAddressBook, origAddr, addr );
    uid = addr.uid();
  }

  UndoStack::instance()->push( command );
  RedoStack::instance()->clear();

  setModified( true );
}

void KABCore::newContact()
{


  QPtrList<KABC::Resource> kabcResources = mAddressBook->resources();

  QPtrList<KRES::Resource> kresResources;
  QPtrListIterator<KABC::Resource> it( kabcResources );
  KABC::Resource *resource;
  while ( ( resource = it.current() ) != 0 ) {
    ++it;
    if ( !resource->readOnly() ) {
      KRES::Resource *res = static_cast<KRES::Resource*>( resource );
      if ( res )
        kresResources.append( res );
    }
  }

  KRES::Resource *res = KRES::SelectDialog::getResource( kresResources, this );
  resource = static_cast<KABC::Resource*>( res );

  if ( resource ) {
    KABC::Addressee addr;
    addr.setResource( resource );
    mEditorDialog->setAddressee( addr );
    KApplication::execDialog ( mEditorDialog );

  } else
    return;

  // mEditorDict.insert( dialog->addressee().uid(), dialog );


}

void KABCore::addEmail( QString aStr )
{
#ifndef KAB_EMBEDDED
  QString fullName, email;

  KABC::Addressee::parseEmailAddress( aStr, fullName, email );

  // Try to lookup the addressee matching the email address
  bool found = false;
  QStringList emailList;
  KABC::AddressBook::Iterator it;
  for ( it = mAddressBook->begin(); !found && (it != mAddressBook->end()); ++it ) {
    emailList = (*it).emails();
    if ( emailList.contains( email ) > 0 ) {
      found = true;
      (*it).setNameFromString( fullName );
      editContact( (*it).uid() );
    }
  }

  if ( !found ) {
    KABC::Addressee addr;
    addr.setNameFromString( fullName );
    addr.insertEmail( email, true );

    mAddressBook->insertAddressee( addr );
    mViewManager->refreshView( addr.uid() );
    editContact( addr.uid() );
  }
#else //KAB_EMBEDDED
  qDebug("KABCore::addEmail finsih method");
#endif //KAB_EMBEDDED
}

void KABCore::importVCard( const KURL &url, bool showPreview )
{
  mXXPortManager->importVCard( url, showPreview );
}
void KABCore::importFromOL()
{
#ifdef _WIN32_
  KAImportOLdialog* idgl = new KAImportOLdialog( i18n("Import Contacts from OL"),  mAddressBook, this );
  idgl->exec();
  KABC::Addressee::List list = idgl->getAddressList();
  if ( list.count() > 0 ) {
    KABC::Addressee::List listNew;
    KABC::Addressee::List listExisting;
    KABC::Addressee::List::Iterator it;
    KABC::AddressBook::Iterator iter;
    for ( it = list.begin(); it != list.end(); ++it ) {
      if ( mAddressBook->findByUid((*it).uid() ).isEmpty())
	listNew.append( (*it) );
      else
	listExisting.append( (*it) );
    }
    if ( listExisting.count()  > 0 )
      KMessageBox::information( this, i18n("%1 contacts not added to addressbook\nbecause they were already in the addressbook!").arg( listExisting.count() ));
    if ( listNew.count() > 0 ) {
      pasteWithNewUid = false;
      pasteContacts( listNew );
      pasteWithNewUid = true;
    }
  }
  delete idgl;
#endif
}

void KABCore::importVCard( const QString &vCard, bool showPreview )
{
  mXXPortManager->importVCard( vCard, showPreview );
}

//US added a second method without defaultparameter
void KABCore::editContact2() {
  editContact( QString::null );
}

void KABCore::editContact( const QString &uid )
{

  if ( mExtensionManager->isQuickEditVisible() )
    return;

  // First, locate the contact entry
  QString localUID = uid;
  if ( localUID.isNull() ) {
    QStringList uidList = mViewManager->selectedUids();
    if ( uidList.count() > 0 )
      localUID = *( uidList.at( 0 ) );
  }

  KABC::Addressee addr = mAddressBook->findByUid( localUID );
  if ( !addr.isEmpty() ) {
      mEditorDialog->setAddressee( addr );
      KApplication::execDialog ( mEditorDialog );
  }
}

/**
  Shows or edits the detail view for the given uid. If the uid is QString::null,
  the method will try to find a selected addressee in the view.
 */
void KABCore::executeContact( const QString &uid /*US = QString::null*/ )
{
  if ( mMultipleViewsAtOnce )
  {
    editContact( uid );
  }
  else
  {
    setDetailsVisible( true );
    mActionDetails->setChecked(true);
  }

}

void KABCore::save()
{
    if ( !mModified )
        return;
   QString text = i18n( "There was an error while attempting to save\n the "
    "address book. Please check that some \nother application is "
    "not using it. " );
   statusMessage(i18n("Saving addressbook ... "));
#ifndef KAB_EMBEDDED
  KABC::StdAddressBook *b = dynamic_cast<KABC::StdAddressBook*>( mAddressBook );
  if ( !b || !b->save() ) {
    KMessageBox::error( this, text, i18n( "Unable to Save" ) );
  }
#else //KAB_EMBEDDED
  KABC::StdAddressBook *b = (KABC::StdAddressBook*)( mAddressBook );
  if ( !b || !b->save() ) {
    QMessageBox::critical( this, i18n( "Unable to Save" ), text,  i18n("Ok"));
  }
#endif //KAB_EMBEDDED

  statusMessage(i18n("Addressbook saved!"));
  setModified( false );
}

void KABCore::statusMessage(QString mess , int time )
{
    //topLevelWidget()->setCaption( mess );
    // pending setting timer to revome message
}
void KABCore::undo()
{
  UndoStack::instance()->undo();

  // Refresh the view
  mViewManager->refreshView();
}

void KABCore::redo()
{
  RedoStack::instance()->redo();

  // Refresh the view
  mViewManager->refreshView();
}

void KABCore::setJumpButtonBarVisible( bool visible )
{
  if (mMultipleViewsAtOnce)
  {
    if ( visible )
      mJumpButtonBar->show();
    else
      mJumpButtonBar->hide();
  }
  else
  {
	// show the jumpbar only if "the details are hidden" == "viewmanager are shown"
    if (mViewManager->isVisible())
    {
      if ( visible )
        mJumpButtonBar->show();
      else
        mJumpButtonBar->hide();
	}
	else
	{
      mJumpButtonBar->hide();
	}
  }
}


void KABCore::setDetailsToState()
{
    setDetailsVisible(  mActionDetails->isChecked() );
}



void KABCore::setDetailsVisible( bool visible )
{
  if (visible && mDetails->isHidden())
  {
    KABC::Addressee::List addrList = mViewManager->selectedAddressees();
    if ( addrList.count() > 0 )
      mDetails->setAddressee( addrList[ 0 ] );
  }

  // mMultipleViewsAtOnce=false: mDetails is always visible. But we switch between
  // the listview and the detailview. We do that by changing the splitbar size.
  if (mMultipleViewsAtOnce)
  {
    if ( visible )
      mDetails->show();
    else
      mDetails->hide();
  }
  else
  {
    if ( visible ) {
      mViewManager->hide();
      mDetails->show();
    }
    else {
      mViewManager->show();
      mDetails->hide();
    }
    setJumpButtonBarVisible( !visible );
  }

}

void KABCore::extensionChanged( int id )
{
  //change the details view only for non desktop systems
#ifndef DESKTOP_VERSION

  if (id == 0)
  {
    //the user disabled the extension.

    if (mMultipleViewsAtOnce)
    { // enable detailsview again
      setDetailsVisible( true );
      mActionDetails->setChecked( true );
    }
    else
    {  //go back to the listview
      setDetailsVisible( false );
      mActionDetails->setChecked( false );
      mActionDetails->setEnabled(true);
	}

  }
  else
  {
    //the user enabled the extension.
    setDetailsVisible( false );
    mActionDetails->setChecked( false );

    if (!mMultipleViewsAtOnce)
    {
      mActionDetails->setEnabled(false);
	}

    mExtensionManager->setSelectionChanged();

  }

#endif// DESKTOP_VERSION

}


void KABCore::extensionModified( const KABC::Addressee::List &list )
{

  if ( list.count() != 0 ) {
    KABC::Addressee::List::ConstIterator it;
    for ( it = list.begin(); it != list.end(); ++it )
      mAddressBook->insertAddressee( *it );
    if ( list.count() > 1 )
        setModified();
    else
        setModifiedWOrefresh();
  }
  if ( list.count() == 0 )
    mViewManager->refreshView();
  else
     mViewManager->refreshView( list[ 0 ].uid() );



}

QString KABCore::getNameByPhone( const QString &phone )
{
#ifndef KAB_EMBEDDED
  QRegExp r( "[/*/-/ ]" );
  QString localPhone( phone );

  bool found = false;
  QString ownerName = "";
  KABC::AddressBook::Iterator iter;
  KABC::PhoneNumber::List::Iterator phoneIter;
  KABC::PhoneNumber::List phoneList;
  for ( iter = mAddressBook->begin(); !found && ( iter != mAddressBook->end() ); ++iter ) {
    phoneList = (*iter).phoneNumbers();
    for ( phoneIter = phoneList.begin(); !found && ( phoneIter != phoneList.end() );
          ++phoneIter) {
      // Get rid of separator chars so just the numbers are compared.
      if ( (*phoneIter).number().replace( r, "" ) == localPhone.replace( r, "" ) ) {
        ownerName = (*iter).formattedName();
        found = true;
      }
    }
  }

  return ownerName;
#else //KAB_EMBEDDED
  qDebug("KABCore::getNameByPhone finsih method");
  return "";
#endif //KAB_EMBEDDED

}

void KABCore::openConfigDialog()
{
	KABPrefs* kab_prefs = KABPrefs::instance();
	KPimGlobalPrefs* kpim_prefs = KPimGlobalPrefs::instance();

    KCMultiDialog*  ConfigureDialog = new KCMultiDialog( "PIM", this ,"kabconfigdialog", true );
    KCMKabConfig* kabcfg = new KCMKabConfig( kab_prefs, ConfigureDialog->getNewVBoxPage(i18n( "Addressbook")) , "KCMKabConfig" );
    ConfigureDialog->addModule(kabcfg );
    KCMKdePimConfig* kdelibcfg = new KCMKdePimConfig( kpim_prefs, ConfigureDialog->getNewVBoxPage(i18n( "Global")) , "KCMKdeLibConfig" );
    ConfigureDialog->addModule(kdelibcfg );



    connect( ConfigureDialog, SIGNAL( applyClicked() ),
             this, SLOT( configurationChanged() ) );
    connect( ConfigureDialog, SIGNAL( okClicked() ),
             this, SLOT( configurationChanged() ) );
    saveSettings();
#ifndef DESKTOP_VERSION
    ConfigureDialog->showMaximized();
#endif
    if ( ConfigureDialog->exec() )
        KMessageBox::information( this,  i18n("Some changes are only\neffective after a restart!\n") );
    delete ConfigureDialog;
}

void KABCore::openLDAPDialog()
{
#ifndef KAB_EMBEDDED
  if ( !mLdapSearchDialog ) {
    mLdapSearchDialog = new LDAPSearchDialog( mAddressBook, this );
    connect( mLdapSearchDialog, SIGNAL( addresseesAdded() ), mViewManager,
            SLOT( refreshView() ) );
    connect( mLdapSearchDialog, SIGNAL( addresseesAdded() ), this,
            SLOT( setModified() ) );
  } else
    mLdapSearchDialog->restoreSettings();

  if ( mLdapSearchDialog->isOK() )
    mLdapSearchDialog->exec();
#else //KAB_EMBEDDED
  qDebug("KABCore::openLDAPDialog() finsih method");
#endif //KAB_EMBEDDED
}

void KABCore::print()
{
#ifndef KAB_EMBEDDED
  KPrinter printer;
  if ( !printer.setup( this ) )
    return;

  KABPrinting::PrintingWizard wizard( &printer, mAddressBook,
                                      mViewManager->selectedUids(), this );

  wizard.exec();
#else //KAB_EMBEDDED
  qDebug("KABCore::print() finsih method");
#endif //KAB_EMBEDDED

}


void KABCore::addGUIClient( KXMLGUIClient *client )
{
  if ( mGUIClient )
    mGUIClient->insertChildClient( client );
  else
    KMessageBox::error( this, "no KXMLGUICLient");
}


void KABCore::configurationChanged()
{
  mExtensionManager->reconfigure();
}

void KABCore::addressBookChanged()
{
/*US
  QDictIterator<AddresseeEditorDialog> it( mEditorDict );
  while ( it.current() ) {
    if ( it.current()->dirty() ) {
      QString text = i18n( "Data has been changed externally. Unsaved "
                           "changes will be lost." );
      KMessageBox::information( this, text );
    }
    it.current()->setAddressee( mAddressBook->findByUid( it.currentKey() ) );
    ++it;
  }
*/
  if (mEditorDialog)
  {
    if (mEditorDialog->dirty())
    {
      QString text = i18n( "Data has been changed externally. Unsaved "
                           "changes will be lost." );
      KMessageBox::information( this, text );
    }
    QString currentuid = mEditorDialog->addressee().uid();
    mEditorDialog->setAddressee( mAddressBook->findByUid( currentuid ) );
  }
  mViewManager->refreshView();
//  mDetails->refreshView();


}

AddresseeEditorDialog *KABCore::createAddresseeEditorDialog( QWidget *parent,
                                                             const char *name )
{

    if ( mEditorDialog == 0 ) {
  mEditorDialog = new AddresseeEditorDialog( this, parent,
                                                 name ? name : "editorDialog" );


  connect( mEditorDialog, SIGNAL( contactModified( const KABC::Addressee& ) ),
           SLOT( contactModified( const KABC::Addressee& ) ) );
  //connect( mEditorDialog, SIGNAL( editorDestroyed( const QString& ) ),
  //       SLOT( slotEditorDestroyed( const QString& ) ) ;
    }

  return mEditorDialog;
}

void KABCore::slotEditorDestroyed( const QString &uid )
{
    //mEditorDict.remove( uid );
}

void KABCore::initGUI()
{
#ifndef KAB_EMBEDDED
  QHBoxLayout *topLayout = new QHBoxLayout( this );
  topLayout->setSpacing( KDialogBase::spacingHint() );

  mExtensionBarSplitter = new QSplitter( this );
  mExtensionBarSplitter->setOrientation( Qt::Vertical );

  mDetailsSplitter = new QSplitter( mExtensionBarSplitter );

  QVBox *viewSpace = new QVBox( mDetailsSplitter );
  mIncSearchWidget = new IncSearchWidget( viewSpace );
  connect( mIncSearchWidget, SIGNAL( doSearch( const QString& ) ),
           SLOT( incrementalSearch( const QString& ) ) );

  mViewManager = new ViewManager( this, viewSpace );
  viewSpace->setStretchFactor( mViewManager, 1 );

  mDetails = new ViewContainer( mDetailsSplitter );

  mJumpButtonBar = new JumpButtonBar( this, this );

  mExtensionManager = new ExtensionManager( this, mExtensionBarSplitter );

  topLayout->addWidget( mExtensionBarSplitter );
  topLayout->setStretchFactor( mExtensionBarSplitter, 100 );
  topLayout->addWidget( mJumpButtonBar );
  topLayout->setStretchFactor( mJumpButtonBar, 1 );

  mXXPortManager = new XXPortManager( this, this );

#else //KAB_EMBEDDED
  //US initialize viewMenu before settingup viewmanager.
  //  Viewmanager needs this menu to plugin submenues.
  viewMenu = new QPopupMenu( this );
  settingsMenu = new QPopupMenu( this );
  //filterMenu = new QPopupMenu( this );
  ImportMenu = new QPopupMenu( this );
  ExportMenu = new QPopupMenu( this );

  changeMenu= new QPopupMenu( this );

//US since we have no splitter for the embedded system, setup
// a layout with two frames. One left and one right.

  QBoxLayout *topLayout;

  // = new QHBoxLayout( this );
//  QBoxLayout *topLayout = (QBoxLayout*)layout();

//  QWidget *mainBox = new QWidget( this );
//  QBoxLayout * mainBoxLayout  = new QHBoxLayout(mainBox);

#ifdef DESKTOP_VERSION
  topLayout = new QHBoxLayout( this );


  mMiniSplitter = new KDGanttMinimizeSplitter( Qt::Horizontal, this);
  mMiniSplitter->setMinimizeDirection ( KDGanttMinimizeSplitter::Right );

  topLayout->addWidget(mMiniSplitter );

  mExtensionBarSplitter = new KDGanttMinimizeSplitter( Qt::Vertical,mMiniSplitter );
  mExtensionBarSplitter->setMinimizeDirection ( KDGanttMinimizeSplitter::Down );
  mViewManager = new ViewManager( this, mExtensionBarSplitter  );
  mDetails = new ViewContainer( mMiniSplitter );
  mExtensionManager = new ExtensionManager( this, mExtensionBarSplitter );
#else
  if ( QApplication::desktop()->width() > 480 ) {
      topLayout = new QHBoxLayout( this );
      mMiniSplitter = new KDGanttMinimizeSplitter( Qt::Horizontal, this);
      mMiniSplitter->setMinimizeDirection ( KDGanttMinimizeSplitter::Right );
  } else {

      topLayout = new QHBoxLayout( this );
      mMiniSplitter = new KDGanttMinimizeSplitter( Qt::Vertical, this);
      mMiniSplitter->setMinimizeDirection ( KDGanttMinimizeSplitter::Down );
  }

  topLayout->addWidget(mMiniSplitter );
  mViewManager = new ViewManager( this,  mMiniSplitter );
  mDetails = new ViewContainer( mMiniSplitter );


   mExtensionManager = new ExtensionManager( this, mMiniSplitter );
#endif
   //eh->hide();
  // topLayout->addWidget(mExtensionManager );


/*US
#ifndef KAB_NOSPLITTER
  QHBoxLayout *topLayout = new QHBoxLayout( this );
//US  topLayout->setSpacing( KDialogBase::spacingHint() );
  topLayout->setSpacing( 10 );

  mDetailsSplitter = new QSplitter( this );

  QVBox *viewSpace = new QVBox( mDetailsSplitter );

  mViewManager = new ViewManager( this, viewSpace );
  viewSpace->setStretchFactor( mViewManager, 1 );

  mDetails = new ViewContainer( mDetailsSplitter );

  topLayout->addWidget( mDetailsSplitter );
  topLayout->setStretchFactor( mDetailsSplitter, 100 );
#else //KAB_NOSPLITTER
  QHBoxLayout *topLayout = new QHBoxLayout( this );
//US  topLayout->setSpacing( KDialogBase::spacingHint() );
  topLayout->setSpacing( 10 );

//  mDetailsSplitter = new QSplitter( this );

  QVBox *viewSpace = new QVBox( this );

  mViewManager = new ViewManager( this, viewSpace );
  viewSpace->setStretchFactor( mViewManager, 1 );

  mDetails = new ViewContainer( this );

  topLayout->addWidget( viewSpace );
//  topLayout->setStretchFactor( mDetailsSplitter, 100 );
  topLayout->addWidget( mDetails );
#endif //KAB_NOSPLITTER
*/


#endif //KAB_EMBEDDED
  initActions();

#ifdef KAB_EMBEDDED
  addActionsManually();
  //US make sure the export and import menues are initialized before creating the xxPortManager.
  mXXPortManager = new XXPortManager( this, this );

  // LR mIncSearchWidget = new IncSearchWidget( mMainWindow->getIconToolBar() );
  //mMainWindow->toolBar()->insertWidget(-1, 4, mIncSearchWidget);
  // mActionQuit->plug ( mMainWindow->toolBar());
  //mIncSearchWidget = new IncSearchWidget( mMainWindow->toolBar() );
  //mMainWindow->toolBar()->insertWidget(-1, 0, mIncSearchWidget);
  // mIncSearchWidget->hide();
  connect( mIncSearchWidget, SIGNAL( doSearch( const QString& ) ),
           SLOT( incrementalSearch( const QString& ) ) );


  mJumpButtonBar = new JumpButtonBar( this, this );

  topLayout->addWidget( mJumpButtonBar );
//US  topLayout->setStretchFactor( mJumpButtonBar, 10 );

// mMainWindow->getIconToolBar()->raise();

#endif //KAB_EMBEDDED

}
void KABCore::initActions()
{
//US  qDebug("KABCore::initActions(): mIsPart %i", mIsPart);

#ifndef KAB_EMBEDDED
  connect( QApplication::clipboard(), SIGNAL( dataChanged() ),
           SLOT( clipboardDataChanged() ) );
#endif //KAB_EMBEDDED

  // file menu
  if ( mIsPart ) {
    mActionMail = new KAction( i18n( "&Mail" ), "mail_generic", 0, this,
                               SLOT( sendMail() ), actionCollection(),
                               "kaddressbook_mail" );
    mActionPrint = new KAction( i18n( "&Print" ), "fileprint", CTRL + Key_P, this,
                 SLOT( print() ), actionCollection(), "kaddressbook_print" );

  } else {
    mActionMail = KStdAction::mail( this, SLOT( sendMail() ), actionCollection() );
    mActionPrint = KStdAction::print( this, SLOT( print() ), actionCollection() );
  }


  mActionSave = new KAction( i18n( "&Save" ), "filesave", CTRL+Key_S, this,
                             SLOT( save() ), actionCollection(), "file_sync" );

  mActionNewContact = new KAction( i18n( "&New Contact..." ), "filenew", CTRL+Key_N, this,
                             SLOT( newContact() ), actionCollection(), "file_new_contact" );

  mActionMailVCard = new KAction(i18n("Mail &vCard..."), "mail_post_to", 0,
                                 this, SLOT( mailVCard() ),
                                 actionCollection(), "file_mail_vcard");

  mActionBeamVCard = 0;
 mActionBeam = 0;

#ifndef DESKTOP_VERSION
  if ( Ir::supported() ) {
    mActionBeamVCard = new KAction( i18n( "Beam selected v&Card(s)" ), "beam", 0, this,
                               SLOT( beamVCard() ), actionCollection(),
                               "kaddressbook_beam_vcard" );

    mActionBeam = new KAction( i18n( "&Beam personal vCard" ), "beam", 0, this,
                               SLOT( beamMySelf() ), actionCollection(),
                               "kaddressbook_beam_myself" );
  }
#endif

  mActionEditAddressee = new KAction( i18n( "&Edit Contact..." ), "edit", 0,
                                      this, SLOT( editContact2() ),
                                      actionCollection(), "file_properties" );

#ifdef KAB_EMBEDDED
  // mActionQuit = KStdAction::quit( mMainWindow, SLOT( exit() ), actionCollection() );
  mActionQuit = new KAction( i18n( "&Exit" ), "exit", 0,
                                      mMainWindow, SLOT( exit() ),
                                      actionCollection(), "quit" );
#endif //KAB_EMBEDDED

  // edit menu
  if ( mIsPart ) {
    mActionCopy = new KAction( i18n( "&Copy" ), "editcopy", CTRL + Key_C, this,
                               SLOT( copyContacts() ), actionCollection(),
                               "kaddressbook_copy" );
    mActionCut = new KAction( i18n( "Cu&t" ), "editcut", CTRL + Key_X, this,
                              SLOT( cutContacts() ), actionCollection(),
                              "kaddressbook_cut" );
    mActionPaste = new KAction( i18n( "&Paste" ), "editpaste", CTRL + Key_V, this,
                                SLOT( pasteContacts() ), actionCollection(),
                                "kaddressbook_paste" );
    mActionSelectAll = new KAction( i18n( "Select &All" ), CTRL + Key_A, this,
                  SLOT( selectAllContacts() ), actionCollection(),
                 "kaddressbook_select_all" );
    mActionUndo = new KAction( i18n( "&Undo" ), "undo", CTRL + Key_Z, this,
                               SLOT( undo() ), actionCollection(),
                               "kaddressbook_undo" );
    mActionRedo = new KAction( i18n( "Re&do" ), "redo", CTRL + SHIFT + Key_Z,
                               this, SLOT( redo() ), actionCollection(),
                               "kaddressbook_redo" );
  } else {
    mActionCopy = KStdAction::copy( this, SLOT( copyContacts() ), actionCollection() );
    mActionCut = KStdAction::cut( this, SLOT( cutContacts() ), actionCollection() );
    mActionPaste = KStdAction::paste( this, SLOT( pasteContacts() ), actionCollection() );
    mActionSelectAll = KStdAction::selectAll( this, SLOT( selectAllContacts() ), actionCollection() );
    mActionUndo = KStdAction::undo( this, SLOT( undo() ), actionCollection() );
    mActionRedo = KStdAction::redo( this, SLOT( redo() ), actionCollection() );
  }

  mActionDelete = new KAction( i18n( "&Delete Contact" ), "editdelete",
                               Key_Delete, this, SLOT( deleteContacts() ),
                               actionCollection(), "edit_delete" );

  mActionUndo->setEnabled( false );
  mActionRedo->setEnabled( false );

  // settings menu
#ifdef KAB_EMBEDDED
//US special menuentry to configure the addressbook resources. On KDE
// you do that through the control center !!!
  mActionConfigResources = new KAction( i18n( "Configure &Resources..." ), "configure_resources", 0, this,
               SLOT( configureResources() ), actionCollection(),
               "kaddressbook_configure_resources" );
#endif //KAB_EMBEDDED

  if ( mIsPart ) {
    mActionConfigKAddressbook = new KAction( i18n( "&Configure KAddressBook..." ), "configure", 0, this,
                 SLOT( openConfigDialog() ), actionCollection(),
                 "kaddressbook_configure" );

    mActionConfigShortcuts = new KAction( i18n( "Configure S&hortcuts..." ), "configure_shortcuts", 0,
                 this, SLOT( configureKeyBindings() ), actionCollection(),
                 "kaddressbook_configure_shortcuts" );
#ifdef KAB_EMBEDDED
  mActionConfigureToolbars = KStdAction::configureToolbars( this, SLOT( mMainWindow->configureToolbars() ), actionCollection() );
  mActionConfigureToolbars->setEnabled( false );
#endif //KAB_EMBEDDED

  } else {
    mActionConfigKAddressbook = KStdAction::preferences( this, SLOT( openConfigDialog() ), actionCollection() );

    mActionKeyBindings = KStdAction::keyBindings( this, SLOT( configureKeyBindings() ), actionCollection() );
  }

  mActionJumpBar = new KToggleAction( i18n( "Show Jump Bar" ), 0, 0,
                                      actionCollection(), "options_show_jump_bar" );
  connect( mActionJumpBar, SIGNAL( toggled( bool ) ), SLOT( setJumpButtonBarVisible( bool ) ) );

  mActionDetails = new KToggleAction( i18n( "Show Details" ), "listview", 0,
                                      actionCollection(), "options_show_details" );
  connect( mActionDetails, SIGNAL( toggled( bool ) ), SLOT( setDetailsVisible( bool ) ) );

  // misc
  // only enable LDAP lookup if we can handle the protocol
#ifndef KAB_EMBEDDED
  if ( KProtocolInfo::isKnownProtocol( KURL( "ldap://localhost" ) ) ) {
    new KAction( i18n( "&Lookup Addresses in Directory" ), "find", 0,
                 this, SLOT( openLDAPDialog() ), actionCollection(),
                 "ldap_lookup" );
  }
#else //KAB_EMBEDDED
  //qDebug("KABCore::initActions() LDAP has to be implemented");
#endif //KAB_EMBEDDED


  mActionWhoAmI = new KAction( i18n( "Set Who Am I" ), "personal", 0, this,
                               SLOT( setWhoAmI() ), actionCollection(),
                               "set_personal" );




  mActionCategories = new KAction( i18n( "Set Categories" ), 0, this,
                                   SLOT( setCategories() ), actionCollection(),
                                   "edit_set_categories" );

  mActionRemoveVoice = new KAction( i18n( "Remove \"voice\"..." ), 0, this,
                                    SLOT( removeVoice() ), actionCollection(),
                                   "remove_voice" );
  mActionImportOL = new KAction( i18n( "Import from OL..." ), 0, this,
                                    SLOT( importFromOL() ), actionCollection(),
                                   "import_OL" );
#ifdef KAB_EMBEDDED
 mActionLicence = new KAction( i18n( "Licence" ), 0,
                 this, SLOT( showLicence() ), actionCollection(),
                 "licence_about_data" );
 mActionFaq = new KAction( i18n( "Faq" ), 0,
                 this, SLOT( faq() ), actionCollection(),
                 "faq_about_data" );

  mActionAboutKAddressbook = new KAction( i18n( "&About KAddressBook" ), "kaddressbook2", 0,
                 this, SLOT( createAboutData() ), actionCollection(),
                 "kaddressbook_about_data" );
#endif //KAB_EMBEDDED

  clipboardDataChanged();
  connect( UndoStack::instance(), SIGNAL( changed() ), SLOT( updateActionMenu() ) );
  connect( RedoStack::instance(), SIGNAL( changed() ), SLOT( updateActionMenu() ) );
}

//US we need this function, to plug all actions into the correct menues.
// KDE uses a XML format to plug the actions, but we work her without this overhead.
void KABCore::addActionsManually()
{
//US  qDebug("KABCore::initActions(): mIsPart %i", mIsPart);

#ifdef KAB_EMBEDDED
  QPopupMenu *fileMenu = new QPopupMenu( this );
  QPopupMenu *editMenu = new QPopupMenu( this );
  QPopupMenu *helpMenu = new QPopupMenu( this );

  KToolBar* tb =  mMainWindow->toolBar();

#ifdef DESKTOP_VERSION
  QMenuBar* mb = mMainWindow->menuBar();

  //US setup menubar.
  //Disable the following block if you do not want to have a menubar.
  mb->insertItem( "&File", fileMenu );
  mb->insertItem( "&Edit", editMenu );
  mb->insertItem( "&View", viewMenu );
  mb->insertItem( "&Settings", settingsMenu );
  mb->insertItem( "&Change selected", changeMenu );
  mb->insertItem( "&Help", helpMenu );
  mIncSearchWidget = new IncSearchWidget( tb );
  // tb->insertWidget(-1, 0, mIncSearchWidget);

#else
  //US setup toolbar
  QPEMenuBar   *menuBarTB = new QPEMenuBar( tb );
  QPopupMenu *popupBarTB = new QPopupMenu( this );
  menuBarTB->insertItem( "ME", popupBarTB);
  tb->insertWidget(-1, 0, menuBarTB);
  mIncSearchWidget = new IncSearchWidget( tb );

  tb->enableMoving(false);
  popupBarTB->insertItem( "&File", fileMenu );
  popupBarTB->insertItem( "&Edit", editMenu );
  popupBarTB->insertItem( "&View", viewMenu );
  popupBarTB->insertItem( "&Settings", settingsMenu );
  mViewManager->getFilterAction()->plug ( popupBarTB);
  popupBarTB->insertItem( "&Change selected", changeMenu );
  popupBarTB->insertItem( "&Help", helpMenu );
  if (QApplication::desktop()->width() > 320 ) {
      // mViewManager->getFilterAction()->plug ( tb);
  }
#endif
  // mActionQuit->plug ( mMainWindow->toolBar());



  //US Now connect the actions with the menue entries.
  mActionPrint->plug( fileMenu );
  mActionMail->plug( fileMenu );
  fileMenu->insertSeparator();

  mActionNewContact->plug( fileMenu );
  mActionNewContact->plug( tb );

  mActionEditAddressee->plug( fileMenu );
  if ((KGlobal::getDesktopSize() > KGlobal::Small ) ||
      (!KABPrefs::instance()->mMultipleViewsAtOnce ))
    mActionEditAddressee->plug( tb );

  fileMenu->insertSeparator();
  mActionSave->plug( fileMenu );
  fileMenu->insertItem( "&Import", ImportMenu );
  fileMenu->insertItem( "&Emport", ExportMenu );
  fileMenu->insertSeparator();
  mActionMailVCard->plug( fileMenu );
#ifndef DESKTOP_VERSION
  if ( Ir::supported() ) mActionBeamVCard->plug( fileMenu );
  if ( Ir::supported() ) mActionBeam->plug(fileMenu );
#endif
  fileMenu->insertSeparator();
  mActionQuit->plug( fileMenu );
#ifdef _WIN32_
  mActionImportOL->plug( ImportMenu  );
#endif
  // edit menu
  mActionUndo->plug( editMenu );
  mActionRedo->plug( editMenu );
  editMenu->insertSeparator();
  mActionCut->plug( editMenu );
  mActionCopy->plug( editMenu );
  mActionPaste->plug( editMenu );
  mActionDelete->plug( editMenu );
  editMenu->insertSeparator();
  mActionSelectAll->plug( editMenu );

  mActionRemoveVoice->plug( changeMenu );
  // settings menu
//US special menuentry to configure the addressbook resources. On KDE
// you do that through the control center !!!
  mActionConfigResources->plug( settingsMenu );
  settingsMenu->insertSeparator();

  mActionConfigKAddressbook->plug( settingsMenu );

  if ( mIsPart ) {
    mActionConfigShortcuts->plug( settingsMenu );
    mActionConfigureToolbars->plug( settingsMenu );

  } else {
    mActionKeyBindings->plug( settingsMenu );
  }

  settingsMenu->insertSeparator();

  mActionJumpBar->plug( settingsMenu );
  mActionDetails->plug( settingsMenu );
  if (!KABPrefs::instance()->mMultipleViewsAtOnce || KGlobal::getDesktopSize() == KGlobal::Desktop  )
    mActionDetails->plug( tb );
  settingsMenu->insertSeparator();

  mActionWhoAmI->plug( settingsMenu );
  mActionCategories->plug( settingsMenu );

  mActionLicence->plug( helpMenu );
  mActionFaq->plug( helpMenu );
  mActionAboutKAddressbook->plug( helpMenu );

  if (KGlobal::getDesktopSize() > KGlobal::Small ) {

      mActionSave->plug( tb );
      mViewManager->getFilterAction()->plug ( tb);
      if (KGlobal::getDesktopSize() == KGlobal::Desktop ) {
	mActionUndo->plug( tb );
	mActionDelete->plug( tb );
	mActionRedo->plug( tb );
      }
  }
  //mActionQuit->plug ( tb );
  // tb->insertWidget(-1, 0, mIncSearchWidget, 6);

  //US link the searchwidget first to this.
  // The real linkage to the toolbar happens later.
//US  mIncSearchWidget->reparent(tb, 0, QPoint(50,0), TRUE);
//US  tb->insertItem( mIncSearchWidget );
/*US
  mIncSearchWidget = new IncSearchWidget( tb );
  connect( mIncSearchWidget, SIGNAL( doSearch( const QString& ) ),
           SLOT( incrementalSearch( const QString& ) ) );

  mJumpButtonBar = new JumpButtonBar( this, this );

//US topLayout->addWidget( mJumpButtonBar );
 this->layout()->add( mJumpButtonBar );
*/

#endif //KAB_EMBEDDED
}
void KABCore::showLicence()
{
    KApplication::showLicence();
}
void KABCore::removeVoice()
{
    if ( KMessageBox::questionYesNo( this, i18n("After importing, phone numbers\nmay have two or more types.\n(E.g. work+voice)\nThese numbers are shown as \"other\".\nClick Yes to remove the voice type\nfrom numbers with more than one type.\n\nRemove voice type?") ) == KMessageBox::No )
        return;
    KABC::Addressee::List list = mViewManager->selectedAddressees();
    KABC::Addressee::List::Iterator it;
    for ( it = list.begin(); it != list.end(); ++it ) {
        PhoneNumber::List phoneNumbers = (*it).phoneNumbers();
        PhoneNumber::List::Iterator phoneIt;
        bool found = false;
        for ( phoneIt = phoneNumbers.begin(); phoneIt != phoneNumbers.end(); ++phoneIt ) {
            if ( (*phoneIt).type() & PhoneNumber::Voice) { // voice found
                if ((*phoneIt).type() - PhoneNumber::Voice ) {
                    (*phoneIt).setType((*phoneIt).type() - PhoneNumber::Voice );
                    (*it).insertPhoneNumber( (*phoneIt) );
                    found = true;
                }
            }

        }
        if ( found )
            contactModified((*it) );
    }
}



void KABCore::clipboardDataChanged()
{

  if ( mReadWrite )
    mActionPaste->setEnabled( !QApplication::clipboard()->text().isEmpty() );

}

void KABCore::updateActionMenu()
{
  UndoStack *undo = UndoStack::instance();
  RedoStack *redo = RedoStack::instance();

  if ( undo->isEmpty() )
    mActionUndo->setText( i18n( "Undo" ) );
  else
    mActionUndo->setText( i18n( "Undo %1" ).arg( undo->top()->name() ) );

  mActionUndo->setEnabled( !undo->isEmpty() );

  if ( !redo->top() )
    mActionRedo->setText( i18n( "Redo" ) );
  else
    mActionRedo->setText( i18n( "Redo %1" ).arg( redo->top()->name() ) );

  mActionRedo->setEnabled( !redo->isEmpty() );
}

void KABCore::configureKeyBindings()
{
#ifndef KAB_EMBEDDED
  KKeyDialog::configure( actionCollection(), true );
#else //KAB_EMBEDDED
  qDebug("KABCore::configureKeyBindings() not implemented");
#endif //KAB_EMBEDDED
}

#ifdef KAB_EMBEDDED
void KABCore::configureResources()
{
  KRES::KCMKResources dlg( this, "" , 0 );

  if ( !dlg.exec() )
    return;
  KMessageBox::information( this,  i18n("Please restart to get the \nchanged resources (re)loaded!\n") );
}
#endif //KAB_EMBEDDED


/* this method will be called through the QCop interface from Ko/Pi to select addresses
 * for the attendees list of an event.
 */
void KABCore::requestForNameEmailUidList(const QString& sourceChannel, const QString& uid)
{
  QStringList nameList;
  QStringList emailList;
  QStringList uidList;

  KABC::Addressee::List list = KABC::AddresseeDialog::getAddressees(this);
  uint i=0;
  for (i=0; i < list.count(); i++)
  {
	nameList.append(list[i].realName());
	emailList.append(list[i].preferredEmail());
	uidList.append(list[i].uid());
  }

  bool res = ExternalAppHandler::instance()->returnNameEmailUidListFromKAPI(sourceChannel, uid, nameList, emailList, uidList);

}

/* this method will be called through the QCop interface from other apps to show details of a contact.
 */
void KABCore::requestForDetails(const QString& sourceChannel, const QString& sessionuid, const QString& name, const QString& email, const QString& uid)
{
    qDebug("KABCore::requestForDetails %s %s %s %s %s", sourceChannel.latin1(), sessionuid.latin1(), name.latin1(), email.latin1(), uid.latin1());

    QString foundUid = QString::null;
    if ( ! uid.isEmpty() ) {
        Addressee adrr = mAddressBook->findByUid( uid );
        if (  !adrr.isEmpty() ) {
            foundUid = uid;
        }
            if ( email == "sendbacklist" ) {
                //qDebug("ssssssssssssssssssssssend ");
                QStringList nameList;
                QStringList emailList;
                QStringList uidList;
                nameList.append(adrr.realName());
                emailList = adrr.emails();
                uidList.append( adrr.preferredEmail());
                bool res = ExternalAppHandler::instance()->returnNameEmailUidListFromKAPI("QPE/Application/ompi", uid, nameList, emailList, uidList);
                return;
            }
        
    }
  
  if ( email == "sendbacklist" ) 
      return;
  if (foundUid.isEmpty())
  {
    //find the uid of the person first
    Addressee::List namelist;
    Addressee::List emaillist;

    if (!name.isEmpty())
      namelist = mAddressBook->findByName( name );

    if (!email.isEmpty())
      emaillist = mAddressBook->findByEmail( email );
    qDebug("count %d %d ",  namelist.count(),emaillist.count() );
    //check if we have a match in Namelist and Emaillist
    if ((namelist.count() == 0) && (emaillist.count() > 0)) {
      foundUid = emaillist[0].uid();
    }
    else if ((namelist.count() > 0) && (emaillist.count() == 0))
      foundUid = namelist[0].uid();
    else
    {
      for (int i = 0; i < namelist.count(); i++)
      {
        for (int j = 0; j < emaillist.count(); j++)
        {
          if (namelist[i] == emaillist[j])
          {
            foundUid = namelist[i].uid();
          }
        }
      }
    }
  }
  else
  {
    foundUid = uid;
  }

  if (!foundUid.isEmpty())
  {
   
    // raise Ka/Pi if it is in the background
#ifndef DESKTOP_VERSION
#ifndef KORG_NODCOP
      //QCopEnvelope e("QPE/Application/kapi", "raise()");
#endif
#endif
    
    mMainWindow->showMaximized();
    mMainWindow-> raise();

    mViewManager->setSelected( "", false);
    mViewManager->refreshView( "" );
    mViewManager->setSelected( foundUid, true );
    mViewManager->refreshView( foundUid );

    if ( !mMultipleViewsAtOnce )
    {
      setDetailsVisible( true );
      mActionDetails->setChecked(true);
    }
  }
}


void  KABCore::faq()
{
  KApplication::showFile( "KA/Pi FAQ", "kdepim/kaddressbook/kapiFAQ.txt" );
}




#ifndef KAB_EMBEDDED
#include "kabcore.moc"
#endif //KAB_EMBEDDED
