/*
 *  linux/drivers/char/pwm520.c
 *
 *  Copyright (C) 2002 Lineo.
 *
 *  Original code write was authored by Craig Matsuura <cmatsuura@lineo.com>
 *  Parts of this code are from Sharp.
 *  This code falls under the license of the GPL.
 *
 *  This modules is for controlling the PWM audio and backlighting.
 *  Jumps for backlighting and audio must be set correctly on the LH79520
 *  Board for this module to work properly.  See Sharp LH79520 Documentation
 *  for jumper settings.
 */
//#define MODULES
#include <linux/config.h>
#include <linux/delay.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/uaccess.h>
#include <linux/module.h>
#include <limits.h>
#include <asm/arch/rcpc.h>
#include <asm/arch/iocon.h>
#include <asm/arch/hardware.h>

#include "pwm520.h"

pwmRegs_t   *pwmregs   = (pwmRegs_t *) IO_ADDRESS( PWM_PHYS);
ioconRegs_t *ioconregs = (ioconRegs_t *)IO_ADDRESS( IOCON_PHYS);
rcpcRegs_t  *rcpcregs  = (rcpcRegs_t *)IO_ADDRESS( RCPC_PHYS);

#define TYPE(dev) (MINOR(dev) >> 4) /* high nibble */
#define NUM(dev)  (MINOR(dev) & 0xf) /* low nibble */
#if OLDWAY
#define PCLK              51609600 /* ticks per second of AHB clock, 51MHz */
static int pwm_xtal_freq = XTAL_IN; // From hardware.h
static int pwm_clkin_freq = 0;      // PLL_CLOCK?  LH79520_CLKIN_FREQ;
#else
unsigned int hclkfreq_get( void);
#endif

static int pwm_major = 0;        // Major Number for Driver 0 indicates dynamic assignment
static int pwm_prescale = 1;

// Default Freq and Duty Cycles
static int pwm_audio_freq = 440;    // Freq of tone
static int pwm_audio_dcycle = 50;   // Duty Cycle for Audio
static int pwm_audio_duration = 100000;// Delay (udelay) for beep

static int pwm_backlight_dcycle = 100;// DC for Backlight
static int pwm_backlight_mode = 0;  // Normal Mode
static int pwm_backlight_freq = BACKLIGHT_INVERTER_PWM_FREQUENCY; // Freq of backlight

#ifdef MODULES
MODULE_PARM(pwm_major,"i");
MODULE_PARM(pwm_prescale,"i");
MODULE_PARM(pwm_xtal_freq,"i");
MODULE_PARM(pwm_clkin_freq,"i");
MODULE_PARM(pwm_audio_freq,"i");
MODULE_PARM(pwm_audio_dcycle,"i");
MODULE_PARM(pwm_audio_duration,"i");
MODULE_PARM(pwm_backlight_dcycle,"i");
MODULE_PARM(pwm_backlight_mode,"i");
MODULE_PARM(pwm_backlight_freq,"i");
#endif


//void showPWM0Registers();

#if OLDWAY
/**********************************************************************
*
* Function: rcpc_get_bus_clock - From Sharp Source
*
* Purpose:
*  return the frequency of the bus clock.
*
* Processing:
*  this function returns the frequency of the bus clock in Hz
*  based on the value of frequencies passed in and the value
*  of the RCPC control register CLK_SEL bit.
*
* Parameters:
*  xtalin: the frequency at the XTALIN pin; use 0 if there is no
*          crystal or external clock driving the pin.
*  clkin:  the frequency driving the CLKIN pin; use 0 if that pin
*          is not driven.
*
* Outputs: None
*
* Returns:
*  The bus clock frequency in Hz
*
* Notes:
*  The nominal crystal input frequency in 14745600 Hz.
*
**********************************************************************/
volatile unsigned long rcpc_get_bus_clock(unsigned long xtalin, unsigned long clkin)
{
   unsigned long timebase, divider;

   if ( (rcpcregs->control & RCPC_CTRL_CLKSEL_EXT)
                         == RCPC_CTRL_CLKSEL_EXT)
   {
      /* clock source is external clock */
      timebase = clkin;
   }
   else
   {
      /* clock source is from PLL output */
      timebase = xtalin * 21;
   }

   divider = rcpcregs->HCLKPrescale * 2;
   if (divider == 0)
      divider = 1;

   return timebase / divider;
}
#endif // OLDWAY

/**********************************************************************
*
* Function: pwm1_enable - Taken for Sharp Driver Example code
*
* Purpose:
*  Enable PWM output on the PWM1 channel of the LH79520
*
* Processing:
*   N/A
*
* Parameters: None
*
* Outputs: None
*
* Returns: Nothing
*
* Notes:
*
**********************************************************************/
inline volatile void pwm1_enable(void)
{
    pwmregs->pwm1.enable = PWM_EN_ENABLE;
}

/**********************************************************************
*
* Function: pwm1_disable - Taken for Sharp Driver Example code
*
* Purpose:
*  Disable PWM output on the PWM1 channel of the LH79520
*
* Processing:
*   N/A
*
* Parameters: None
*
* Outputs: None
*
* Returns: Nothing
*
* Notes:
*
**********************************************************************/
inline volatile void pwm1_disable(void)
{
    pwmregs->pwm1.enable = ~PWM_EN_ENABLE;
}

/**********************************************************************
*
* Function: pwm0_enable - Taken for Sharp Driver Example code
*
* Purpose:
*  Enable PWM output on the PWM0 channel of the LH79520
*
* Processing:
*   N/A
*
* Parameters: None
*
* Outputs: None
*
* Returns: Nothing
*
* Notes:
*
**********************************************************************/
inline volatile void pwm0_enable(void)
{
    pwmregs->pwm0.enable = PWM_EN_ENABLE;
}

/**********************************************************************
*
* Function: pwm0_disable - Taken for Sharp Driver Example code
*
* Purpose:
*  Disable PWM output on the PWM0 channel of the LH79520
*
* Processing:
*   N/A
*
* Parameters: None
*
* Outputs: None
*
* Returns: Nothing
*
* Notes:
*
**********************************************************************/
inline volatile void pwm0_disable(void)
{
    pwmregs->pwm0.enable = ~PWM_EN_ENABLE;
}

/**********************************************************************
*
* Function: pwm0_normal - Taken for Sharp Driver Example code146
*
* Purpose:
*  Restores the normal sense of the PWM output signal on the PWM0
*  channel of the LH79520
*
* Processing:
*   N/A
*
* Parameters: None
*
* Outputs: None
*
* Returns: Nothing
*
* Notes:
*
**********************************************************************/
inline volatile void pwm0_normal(void)
{
    pwmregs->pwm0.invert = ~PWM_INV_INVERT;
}


/**********************************************************************
*
* Function: pwm1_frequency - Taken for Sharp Driver Example code
*
* Purpose:
*  Sets the PWM1 output frequency to the value specified
*
* Processing:
*   Compute the prescale and period count to be programmed from the
*   PCLK frequency. If the frequency value is too high or too low to
*   be programmed within the permissible ranges of prescale and period,
*   signal an error.
*
* Parameters:
*   freq - Desired frequency, in Hz
*
* Outputs: None
*
* Returns:
*   -1 - if frequency specified is zero
*   -1 - if prescale is zero (an impossible condition)
*   -1 - if the frequency is out of bounds
*    0 - if the frequency is changed successfully
*
* Notes:
*   This function depends on the values of LH79520_XTAL_FREQ and
*   LH79520_CLKIN_FREQ to be set correctly in the LH79520_evb.h file.
*   If external clock is used, LH79520_CLKIN_FREQ should be set
*   to the external clock frequency. Otherwise, this function will
*   fail.
*
*
**********************************************************************/
long pwm1_frequency(unsigned long freq)
{
    unsigned long pclk, pwm1clk, prescale;
    unsigned long div;

    if (0 == freq) {
        return -1;
    }

#if OLDWAY
    pclk = rcpc_get_bus_clock(pwm_xtal_freq,pwm_clkin_freq);
#else
	pclk = hclkfreq_get();
#endif
    prescale = rcpcregs->PWM1Prescale;
    if (prescale > 0) {
        pwm1clk = pclk / (prescale * 2);
    } else {
        printk("<4>THIS IS BAD.  SHOULD NOT GET HERE...\n");
        //this should not happen
        return -1;
    }
    div = pwm1clk / freq;
    rcpcregs->control |= RCPC_CTRL_WRTLOCK_ENABLED;
    barrier();

    while (div > USHRT_MAX && prescale <= SHRT_MAX) {
        prescale += 1;
        rcpcregs->PWM1Prescale = prescale;
        pwm1clk = pclk / (prescale * 2);
        div = pwm1clk / freq;
    }
    rcpcregs->control &= ~RCPC_CTRL_WRTLOCK_ENABLED;
    pwmregs->pwm1.tc = div;

    return 0;
}


/**********************************************************************
*
* Function: pwm1_duty_cycle - Taken for Sharp Driver Example code
*
* Purpose:
*  Sets the PWM1 duty cycle to the value specified
*
* Processing:
*   Compute the duty cycle count to program into the dc register
*   from the percentage specified and the tc count. Accounts for
*   integer division truncation errors.
*
* Parameters:
*   dcpercent - Desired duty cycle as a percentage of the total period
*
* Outputs: None
*
* Returns:
*   the previous value of the duty cycle, as a percentage
*
* Notes:
*
**********************************************************************/
long pwm1_duty_cycle(short dcpercent)
{
    unsigned short period;
    unsigned long duty_cycle, prev_dc, prev_dc_percent;
    period = (unsigned short) pwmregs->pwm1.tc;
    prev_dc = pwmregs->pwm1.dc;
    if (period > 0) {
        prev_dc_percent = ((prev_dc * 100) + (period >> 1)) / period;
    } else {
        prev_dc_percent = 100;
    }
    duty_cycle = ((dcpercent * period) + 50) / 100;
    pwmregs->pwm1.dc = duty_cycle;
    return(long)prev_dc_percent;
}

/**********************************************************************
*
* Function: pwm0_frequency - Taken for Sharp Driver Example code
*
* Purpose:
*  Sets the PWM0 output frequency to the value specified
*
* Processing:
*   Compute the prescale and period count to be programmed from the
*   PCLK frequency. If the frequency value is too high or too low to
*   be programmed within the permissible ranges of prescale and period,
*   signal an error.
*
* Parameters:
*   freq - Desired frequency, in Hz
*
* Outputs: None
*
* Returns:
*   -1 - if frequency specified is zero
*   -1 - if prescale is zero (an impossible condition)
*   -1 - if the frequency is out of bounds
*    0 - if the frequency is changed successfully
*
* Notes:
*   This function depends on the values of LH79520_XTAL_FREQ and
*   LH79520_CLKIN_FREQ to be set correctly in the LH79520_evb.h file.
*   If external clock is used, LH79520_CLKIN_FREQ should be set
*   to the external clock frequency. Otherwise, this function will
*   fail.
*
*
**********************************************************************/
long pwm0_frequency(unsigned long freq)
{
    unsigned long pclk, pwm0clk, prescale;
    unsigned long div;

    if (0 == freq) {
        return -1;
    }

#if OLDWAY
    pclk = rcpc_get_bus_clock(pwm_xtal_freq,pwm_clkin_freq);
#else
	pclk = hclkfreq_get();
#endif
    prescale = rcpcregs->PWM0Prescale;
    if (prescale > 0) {
        pwm0clk = pclk / (prescale * 2);
    } else {
        printk("<4>THIS IS BAD.  SHOULD NOT GET HERE...\n");
        //this should not happen
        return -1;
    }
    div = pwm0clk / freq;
    rcpcregs->control |= RCPC_CTRL_WRTLOCK_ENABLED;
    barrier();

    while (div > USHRT_MAX && prescale <= SHRT_MAX) {
        prescale += 1;
        rcpcregs->PWM0Prescale = prescale;
        pwm0clk = pclk / (prescale * 2);
        div = pwm0clk / freq;
    }
    rcpcregs->control &= ~RCPC_CTRL_WRTLOCK_ENABLED;
    pwmregs->pwm0.tc = div;

    return 0;
}


/**********************************************************************
*
* Function: pwm0_duty_cycle - Taken for Sharp Driver Example code
*
* Purpose:
*  Sets the PWM0 duty cycle to the value specified
*
* Processing:
*   Compute the duty cycle count to program into the dc register
*   from the percentage specified and the tc count. Accounts for
*   integer division truncation errors.
*
* Parameters:
*   dcpercent - Desired duty cycle as a percentage of the total period
*
* Outputs: None
*
* Returns:
*   the previous value of the duty cycle, as a percentage
*
* Notes:
*
**********************************************************************/
long pwm0_duty_cycle(short dcpercent)
{
    unsigned short period;
    unsigned long duty_cycle, prev_dc, prev_dc_percent;
    period = (unsigned short) pwmregs->pwm0.tc;
    prev_dc = pwmregs->pwm0.dc;
    if (period > 0) {
        prev_dc_percent = ((prev_dc * 100) + (period >> 1)) / period;
    } else {
        prev_dc_percent = 100;
    }
    duty_cycle = ((dcpercent * period) + 50) / 100;
    pwmregs->pwm0.dc = duty_cycle;
    return(long)prev_dc_percent;
}

/**********************************************************************
*
* Function: backlight_decrease_brightness
*
* Purpose:
*  Decrease the LCD backlight brightness
*
* Processing:
*  Decrement static variable holding current brightness level and
*  set the PWM duty cycle.
*
* Parameters: None
*
* Outputs: None
*
* Returns: Nothing
*
* Notes:
*
**********************************************************************/
void backlight_increase_brightness(void)
{
    pwm_backlight_dcycle--;
    if (pwm_backlight_dcycle < 0)
    {
        pwm_backlight_dcycle = 0;
    }
    pwm0_duty_cycle(pwm_backlight_dcycle);
}

/**********************************************************************
*
* Function: backlight_increase_brightness
*
* Purpose:
*  Increase the LCD backlight brightness
*
* Processing:
*  Increment static variable holding current brightness level and
*  set the PWM duty cycle.
*
* Parameters: None
*
* Outputs: None
*
* Returns: Nothing
*
* Notes:
*
**********************************************************************/
void backlight_decrease_brightness(void)
{
    pwm_backlight_dcycle++;
    if (pwm_backlight_dcycle > 100)
    {
        pwm_backlight_dcycle = 100;
    }
    pwm0_duty_cycle(pwm_backlight_dcycle);
}

/**********************************************************************
*
* Function: backlight_set_brightness
*
* Purpose:
*  Set the LCD backlight brightness to the level specified
*
* Processing:
*  N/A
*
* Parameters:
*  bright - desired brightness level as a percentage of maximum
*           brightness
*
* Outputs: None
*
* Returns: Nothing
*
* Notes:
*
**********************************************************************/
void backlight_set_brightness(int bright)
{
    //bright = 0 means least brightness
    //bright = 100 means max brightness
    if (bright > 100)
    {
        bright = 100;
    }
    if (bright < 0)
    {
        bright = 0;
    }

    pwm_backlight_dcycle = 100 - bright;
    pwm0_duty_cycle(pwm_backlight_dcycle);
}


/****************************************************************************
 * Open Function - Open the device either the Backlight or the Audio
 * /dev/pwm0 c 254 0 is the Backlighting
 * /dev/pwm1 c 254 1 is the Audio
 * Keep in mind the 254 is only an example.  If you do not specify a
 * major code then a dynamic one will be assigned.  You will have to
 * look at /proc/devices to see the major code for pwm
 */
int pwm_open(struct inode * inode, struct file * filp)
{

    MOD_INC_USE_COUNT;

//    printk("<4>open pwm... dc=%d, bl_dc=%d\n",pwmregs->pwm0.dc,pwm_backlight_dcycle);
    switch (NUM(inode->i_rdev)) {
        case 0: // Backlight
            rcpcregs->control |= RCPC_CTRL_WRTLOCK_ENABLED;
            barrier();
            rcpcregs->periphClkCtrl &= ~RCPC_CLKCTRL_PWM0_DISABLE;
            rcpcregs->PWM0Prescale = ((~_BIT(15)) & pwm_prescale);
            rcpcregs->control &= ~RCPC_CTRL_WRTLOCK_ENABLED;
            ioconregs->MiscMux |= MISCMUX_PWM0;

            if (pwm_backlight_mode) {
                pwmregs->pwm0.sync = PWM_SYNC_SYNC;
                ioconregs->MiscMux |= MISCMUX_PWM0SYNC;
            }
            else
            {
                pwmregs->pwm0.sync = PWM_SYNC_NORMAL;
            }

            pwm0_frequency(pwm_backlight_freq);
            pwm0_duty_cycle(pwm_backlight_dcycle);
            pwm0_normal();
            pwm0_enable();
            break;
        case 1: // Audio
            rcpcregs->control |= RCPC_CTRL_WRTLOCK_ENABLED;
            barrier();
            rcpcregs->periphClkCtrl &= ~RCPC_CLKCTRL_PWM1_DISABLE;
            rcpcregs->PWM1Prescale = ((~_BIT(15)) & pwm_prescale);
            rcpcregs->control &= ~RCPC_CTRL_WRTLOCK_ENABLED;
            ioconregs->MiscMux |= MISCMUX_PWM1;
            break;
        default:
            printk("<4>Minor device unknown %d\n",NUM(inode->i_rdev));
            break;
    }

    return 0;
}

/****************************************************************************
 *
 */
int pwm_release(struct inode *inode, struct file * filp)
{
//    printk("<4>release pwm...\n");
    switch (NUM(inode->i_rdev)) {
        case 0: // Should be backlight of touchscreen
            //rcpcregs->control |= RCPC_CTRL_WRTLOCK_ENABLED;
            //barrier();
            //rcpcregs->periphClkCtrl |= RCPC_CLKCTRL_PWM0_DISABLE;
            //rcpcregs->control &= ~RCPC_CTRL_WRTLOCK_ENABLED;
            //ioconregs->MiscMux &= ~MISCMUX_PWM0;
            //pwmregs->pwm0.enable = ~PWM_EN_ENABLE;
            break;
        case 1: // Audio
            rcpcregs->control |= RCPC_CTRL_WRTLOCK_ENABLED;
            barrier();
            rcpcregs->periphClkCtrl |= RCPC_CLKCTRL_PWM1_DISABLE;
            rcpcregs->control &= ~RCPC_CTRL_WRTLOCK_ENABLED;
            ioconregs->MiscMux &= ~MISCMUX_PWM1;
            pwm1_disable();
            break;
        default:
            printk("<4>Minor device unknown %d\n",NUM(inode->i_rdev));
            break;
    }

    MOD_DEC_USE_COUNT;
    return 0;
}

/****************************************************************************
 *
 */
ssize_t pwm_read(struct file * file, char * buf,
                    size_t count, loff_t *ppos)
{
    return 0;
}

/****************************************************************************
 *
 */
ssize_t pwm_write(struct file * file, const char * buf,
                     size_t count, loff_t *ppos)
{
    return 0;
}

/****************************************************************************
 *
 */
int pwm_ioctl(struct inode *inode, struct file * file,
                 unsigned int cmd, unsigned long arg)
{
    //int err = 0, size = _IOC_SIZE(cmd);

    if (_IOC_TYPE(cmd) != PWM520_IOC_MAGIC) {
        return -EINVAL;
    }
    if (_IOC_NR(cmd) > PWM520_IOC_MAXNR) {
        return -EINVAL;
    }

    /*
        Should check for direction bit's see page 101 in "Linux Device Drivers Book"
        size and err used here.
    */

    switch (NUM(inode->i_rdev)) {
        case 0: // Should be backlight of touchscreen
            switch (cmd) {
                case PWM520_IOCRESET:
//                    printk("Reset pwm0\n");
                    pwm_backlight_dcycle = 100;// DC for Backlight
                    pwm_backlight_mode = 0;  // Normal Mode
                    pwm_backlight_freq = BACKLIGHT_INVERTER_PWM_FREQUENCY; // Freq of backlight
                    break;
                case PWM520_IOCSTOPPWM0: // Stop PWM0
                    rcpcregs->control |= RCPC_CTRL_WRTLOCK_ENABLED;
                    barrier();
                    rcpcregs->periphClkCtrl |= RCPC_CLKCTRL_PWM0_DISABLE;
                    rcpcregs->control &= ~RCPC_CTRL_WRTLOCK_ENABLED;
                    ioconregs->MiscMux &= ~MISCMUX_PWM0;
                    pwmregs->pwm0.enable = ~PWM_EN_ENABLE;
                    break;
                case PWM520_IOCINCREASEBL:
//                    printk("Brighter\n");
                    backlight_increase_brightness();
                    break;

                case PWM520_IOCDECREASEBL:
//                    printk("Lighter\n");
                    backlight_decrease_brightness();
                    break;

                case PWM520_IOCSETBL:
//                    printk("Set to %ld\n",arg);
                    backlight_set_brightness(arg);
                    break;
            }
            break;

        case 1: // Audio
            switch (cmd) {
                case PWM520_IOCBEEP:
                    pwm1_frequency(pwm_audio_freq);
                    pwm1_duty_cycle(pwm_audio_dcycle);
                    pwm1_enable();
                    udelay(pwm_audio_duration);
                    pwm1_disable();
                    break;

                case PWM520_IOCSTARTSND:
                    pwm1_frequency(pwm_audio_freq);
                    pwm1_duty_cycle(pwm_audio_dcycle);
                    pwm1_enable();
                    break;

                case PWM520_IOCSTOPSND:
                    pwm1_disable();
                    break;

                case PWM520_IOCSETFREQ: // Set Frequency
                    pwm_audio_freq = arg;
                    break;

                case PWM520_IOCSETDCYCLE: // Set Duty Cycle
                    pwm_audio_dcycle = arg;
                    break;

                case PWM520_IOCGETFREQ: // Get Frequency
                    __put_user(pwm_audio_freq, (int *) arg);
                    break;

                case PWM520_IOCGETDCYCLE: // Get Duty Cycle
                    __put_user(pwm_audio_dcycle, (int *) arg);
                    break;
            }
            break;
    }

    return 0;
}


/****************************************************************************
 *
 */
struct file_operations  pwm_fops = {
    owner:          THIS_MODULE,
    llseek:     no_llseek,
    read:       pwm_read,
    write:      pwm_write,
    ioctl:      pwm_ioctl,
    open:       pwm_open,
    release:    pwm_release,
};

#if 0
void test_pwm1()
{
    printk("TEST CASE PLEASE REMOVE ONCE DONE\n");

    rcpcregs->control |= RCPC_CTRL_WRTLOCK_ENABLED;
    barrier();
    rcpcregs->periphClkCtrl &= ~RCPC_CLKCTRL_PWM1_DISABLE;
    rcpcregs->PWM1Prescale = ((~_BIT(15)) & pwm_prescale);
    rcpcregs->control &= ~RCPC_CTRL_WRTLOCK_ENABLED;
    ioconregs->MiscMux |= MISCMUX_PWM1;

    pwm1_frequency(pwm_audio_freq);
    pwm1_duty_cycle(pwm_audio_dcycle);
    pwm1_enable();
}

void showPWM1Registers()
{
    printk("PWM1 Registers\n");
    printk("=============\n\n");
    printk("pwm1.tc     = %x\n",pwmregs->pwm1.tc);
    printk("pwm1.dc     = %x\n",pwmregs->pwm1.dc);
    printk("pwm1.enable = %x\n",pwmregs->pwm1.enable);
    printk("pwm1.invert = %x\n",pwmregs->pwm1.invert);
    printk("pwm1.sync   = %x\n",pwmregs->pwm1.sync);
}

void showPWM0Registers()
{
    printk("PWM0 Registers\n");
    printk("=============\n\n");
    printk("pwm0.tc     = %d\n",pwmregs->pwm0.tc);
    printk("pwm0.dc     = %d\n",pwmregs->pwm0.dc);
    printk("pwm0.enable = 0x%x\n",pwmregs->pwm0.enable);
    printk("pwm0.invert = 0x%x\n",pwmregs->pwm0.invert);
    printk("pwm0.sync   = 0x%x\n",pwmregs->pwm0.sync);
}

#endif

/****************************************************************************
 *
 */
static int __init pwm_init_module(void)
{
    int result;

    //printk("<1>Start PWM Module\n");
    printk("<1>Sharp LH79520 PWM Driver Copyright 2002 Lineo\n");

    result = register_chrdev(pwm_major,"pwm",&pwm_fops);
    if (result < 0) {
        printk("<4>pwm: can't get major number %d\n",pwm_major);
        return result;
    }
    if (pwm_major == 0) {
        pwm_major = result; /* Dynamic Allocation of major number */
        printk("<4>PWM Dynamic Major Number %d\n",pwm_major);
    }

    return 0;
}

/****************************************************************************
 *
 */
static void __exit pwm_cleanup_module(void)
{
    int result;

    printk("<1>End PWM Module...\n");
    result = unregister_chrdev(pwm_major,"pwm");
}

module_init(pwm_init_module);
module_exit(pwm_cleanup_module);
