/*
 *  linux/include/asm-arm/arch-lh79520/time.h
 *
 *  Copyright (C) 2002 Lineo, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  Changelog:
 *   07-Jan-2002 Duck	Created
 */

#ifndef __ASM_ARCH_TIME_H
#define __ASM_ARCH_TIME_H

#include <asm/arch/hardware.h>
#include <asm/arch/rcpc.h>

/*
 * Dual Timer Module Register Structure
 * The LH79520 has two of these.
 */ 
typedef struct {
	unsigned int	Timer1Load;
	unsigned int	Timer1Value;
	unsigned int	Timer1Control;
	unsigned int	Timer1Clear;
	unsigned int	Timer1Test;
	unsigned int	reservedtmr1[3];
	unsigned int	Timer2Load;
	unsigned int	Timer2Value;
	unsigned int	Timer2Control;
	unsigned int	Timer2Clear;
	unsigned int	Timer2Test;
	unsigned int	reservedtmr2[3];
} timerRegs_t;


/*
 * Timer Control Register Bit Field constants
 * All other bits in the Timer Control Register must be written as
 * zero
 */ 
#define TMRCTRL_ENABLE		_SBF(7,1)
#define TMRCTRL_DISABLE		_SBF(7,0)
#define TMRCTRL_MODE_PERIODIC	_SBF(6,1)
#define TMRCTRL_MODE_FREERUN	_SBF(6,0)
#define TMRCTRL_CASCADE_ENABLE	_SBF(4,1)
#define TMRCTRL_CASCADE_DISABLE	_SBF(4,0)
#define TMRCTRL_PRESCALE1	_SBF(2,0)
#define TMRCTRL_PRESCALE16	_SBF(2,1)
#define TMRCTRL_PRESCALE256	_SBF(2,2)

/*
 * what to load the timer with
 * it's 14.745600 MHz * 21 (PLL multiplier) / 6 (PCLK prescalar) / 16 (TIMER_PRESCALE) / HZ
 * this gives us timerLoad=32256  for Hz=100
 */
#define TIMER_PRESCALE		16

static void
timer_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	timerRegs_t	*mod1Timer = (timerRegs_t *)IO_ADDRESS( TIMER0_PHYS);

	mod1Timer->Timer1Clear = 1;			/* clear interrupt */

	do_timer(regs);
}


static inline void
setup_timer(void)
{
	rcpcRegs_t	*rcpc = (rcpcRegs_t *)IO_ADDRESS( RCPC_PHYS);
	timerRegs_t	*mod1Timer = (timerRegs_t *)IO_ADDRESS( TIMER0_PHYS),   /* first timer module */
	    		*mod2Timer = (timerRegs_t *)IO_ADDRESS( TIMER1_PHYS);   /* second timer module */
	u32		timerLoad;

	timerLoad = hclkfreq_get() / TIMER_PRESCALE / HZ;
	printk( "setup_timer(): timerLoad=%d\n", timerLoad);

	/* stop all timers */
	mod1Timer->Timer1Control = 0;
	mod1Timer->Timer2Control = 0;
	mod2Timer->Timer1Control = 0;
	mod2Timer->Timer2Control = 0;

	/* enable clock to first timer */
	rcpc->control       |= RCPC_CTRL_WRTLOCK_ENABLED;		/* unlock RCPC registers */
	barrier();

	rcpc->periphClkCtrl &= ~RCPC_CLKCTRL_CT0_DISABLE;
	rcpc->control       &= ~RCPC_CTRL_WRTLOCK_ENABLED;		/* lock RCPC registers   */
	
	/* setup the FRC in the first timer in the first module.  */
	mod1Timer->Timer1Load    = timerLoad;

	mod1Timer->Timer1Control = TMRCTRL_ENABLE |
				   TMRCTRL_MODE_PERIODIC |
				   TMRCTRL_PRESCALE16;

	timer_irq.handler = timer_interrupt;
	timer_irq.flags = SA_INTERRUPT;
	setup_arm_irq( IRQ_TIMER0, &timer_irq);
}

#endif
