<?php
/*
 * Copyright (c) 2013, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
 */

namespace MySQLFabric;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Sharding\ShardManager;
use Doctrine\DBAL\Sharding\ShardingException;

/**
 * Sharding Manager gives access to APIs to implementing sharding on top of
 * Doctrine\DBAL\Connection instances.
 *
 * For simplicity and developer ease-of-use (and understanding) the sharding
 * API only covers single shard queries, no fan-out support. It is primarily
 * suited for multi-tenant applications.
 *
 * The assumption about sharding here
 * is that a distribution value can be found that gives access to all the
 * necessary data for all use-cases. Switching between shards should be done with
 * caution, especially if lazy loading is implemented. Any query is always
 * executed against the last shard that was selected. If a query is created for
 * a shard Y but then a shard X is selected when its actually executed you
 * will hit the wrong shard.
 *
 * @author Johannes Schlüter <johannes.schlueter@oracle.com>
 */
class MySQLFabricShardManager implements ShardManager
{
    /**
     * @var Connection
     */
    private $conn;
    
    /**
     * @var string
     */
    private $table;
    
    /**
     * Current sharding distribution value
     * @var string
     */
    private $distributionValue;
    
    public function __construct(Connection $conn)
    {
        $this->conn = $conn;
        $params = $conn->getParams();

        if ( ! isset($params['sharding']['shardTable'])) {
            throw MySQLFabricException::missingShardTable();
        }
        $this->table = $params['sharding']['shardTable'];

        $fabricHosts = mysqlnd_ms_dump_fabric_hosts($conn->getWrappedConnection());
        if ( ! $this->fabricHosts || ! count($this->fabricHosts) ) {
            throw MySQLFabricException::notFabric();
        }
    }

    /**
     * Selects global database with global data.
     *
     * This is the default database that is connected when no shard is
     * selected.
     *
     * @return void
     */
    function selectGlobal()
    {
        $this->distributionValue = null;
        mysqlnd_ms_fabric_select_global($this->conn->getWrappedConnection(), $this->table);
    }

    /**
     * Selects the shard against which the queries after this statement will be issued.
     *
     * @param string $distributionValue
     *
     * @return void
     *
     * @throws \Doctrine\DBAL\Sharding\MSQLFabric\MySQLFabricException If no value is passed as shard identifier.
     */
    function selectShard($distributionValue)
    {
        if ( ! $distributionValue) {
            throw ShardingException::noShardDistributionValue();
        }

        mysqlnd_ms_fabric_select_shard($this->conn->getWrappedConnection(), $this->table, $distributionValue);
        $this->distributionValue = $distributionValue;
    }

    /**
     * {@inheritDoc}
     */
    function getCurrentDistributionValue()
    {
        return $this->distributionValue;
    }

    /**
     * {@inheritDoc}
     */
    function getShards()
    {
        throw ShardingException::notImplemented();
    }

    /**
     * {@inheritDoc}
     */
    function queryAll($sql, array $params, array $types)
    {
        throw ShardingException::notImplemented();
    }
}

